//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.


// Cuts a 2D curves into continuous segments. 
// x: array of x coordinates (typically longitude)
// y: array of y coordinates (typically latitude)
// xmin/xmax : target interval for x. 
// NB: x is supposed defined modulo xmod = xmax-xmin. 
// Algorithm: 
// x is first translated modulo xmod into [xmin,xmax]
// if |x(k+1) - x(k)| > xmod, intermediate points are added. 
// output: lxy = list([x;y]) : continuous segments 
// NB: no segment in lxy is empty

function [lxy] = CL__plot_contSegsList(x, y, xmin, xmax)

  if (xmax <= xmin)
     CL__error('xmax should be strictly greater than xmin');
  end

  if size(x,1) > 1 | size(y,1) > 1
     CL__error('wrong dimension for x or y: should be 1xN');
  end

  if size(x,2) <> size(y,2) 
     CL__error('wrong dimension for x or y: should have the same length');
  end

  A = (xmax-xmin)/2; 
  x = CL_rMod(x, xmin, xmax); 

  delta_x = x(2:$) - x(1:$-1);  // OK if size of x is less than 2 (-> [])  
  I = find(abs(delta_x) > A); // discontinuities

  lxy = list(); 
  n = length(I); 
 
  // case : no data or no discontinuity
  if (n == 0)
    if (length(x) > 0) 
       lxy($+1) = [x; y]; 
    end
    return; // ***  return ! ***
  end
 
  // next point after discontinuity -> close to previous point
  // NOTE : X1 <> X(I) (in principle) 
  X1 = CL_rMod(x(I+1), x(I)-A, x(I)+A); 
 
  // 1er point intermdiaire
  XA = xmin + (2*A) * round((x(I)-xmin)/(2*A));  // xmin or xmax 
  YA = y(I) + ((y(I+1)-y(I))./(X1-x(I))) .* (XA-x(I)); 

  // 2eme point intermdiaire
  XB = xmin + xmax - XA; 
  YB = YA; 

  lxy($+1) = [[x(1:I(1)), XA(1)]; [y(1:I(1)), YA(1)]]; 
 
  for (i=1:n-1)
    lxy($+1) = [ [ XB(i), x(I(i)+1:I(i+1)), XA(i+1) ]; 
                 [ YB(i), y(I(i)+1:I(i+1)), YA(i+1) ] ]; 
  end

  lxy($+1) = [ [XB(n), x(I(n)+1:$) ]; [ YB(n), y(I(n)+1:$) ] ]; 
   
endfunction
