//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  SALOME VTKViewer : build VTK viewer into Salome desktop
//  File   : VVTK_InteractorStyle.cxx
//  Author : Christophe ATTANASIO
//  Module : SALOME
//  $Header$
//
#include "VVTK_InteractorStyle.h"
#include "VISU_GaussPtsSettings.h"
#include "SVTK_Selector.h"
#include "VISU_Event.h" 

#include <vtkObjectFactory.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkCallbackCommand.h>
#include <vtkCommand.h>
#include <vtkRenderer.h>
#include <vtkCamera.h>


//----------------------------------------------------------------------------
vtkStandardNewMacro(VVTK_InteractorStyle);
//----------------------------------------------------------------------------

VVTK_InteractorStyle
::VVTK_InteractorStyle():
  myIsMidButtonDown( false ),
  myIsLeftButtonDown( false ),
  mySMDecreaseMagnificationBtn(10),
  mySMIncreaseMagnificationBtn(11)
{
}

//----------------------------------------------------------------------------
VVTK_InteractorStyle
::~VVTK_InteractorStyle() 
{
}

//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::OnLeftButtonDown(int ctrl, int shift, 
		   int x, int y) 
{
  myIsLeftButtonDown = true;

  if (this->HasObserver(vtkCommand::LeftButtonPressEvent)) {
    this->InvokeEvent(vtkCommand::LeftButtonPressEvent,NULL);
    return;
  }
  this->FindPokedRenderer(x, y);
  if (this->CurrentRenderer == NULL) {
    return;
  }
  myShiftState = shift;
  // finishing current viewer operation
  if (State != VTK_INTERACTOR_STYLE_CAMERA_NONE) {
    onFinishOperation();
    startOperation(VTK_INTERACTOR_STYLE_CAMERA_NONE);
  }
  myOtherPoint = myPoint = QPoint(x, y);
  if (ForcedState != VTK_INTERACTOR_STYLE_CAMERA_NONE) {
    startOperation(ForcedState);
  } 
  else {
    if (!(ctrl||shift)){
      if (myIsMidButtonDown){
	startOperation(VTK_INTERACTOR_STYLE_CAMERA_ZOOM);
      }
      else{
	startOperation(VTK_INTERACTOR_STYLE_CAMERA_ROTATE);
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::OnMiddleButtonDown(int ctrl,
		     int shift, 
		     int x, int y) 
{
  myIsMidButtonDown = true;

  if (this->HasObserver(vtkCommand::MiddleButtonPressEvent))  {
    this->InvokeEvent(vtkCommand::MiddleButtonPressEvent,NULL);
    return;
  }
  this->FindPokedRenderer(x, y);
  if (this->CurrentRenderer == NULL)    {
    return;
  }
  myShiftState = shift;
  // finishing current viewer operation
  if (State != VTK_INTERACTOR_STYLE_CAMERA_NONE) {
    onFinishOperation();
    startOperation(VTK_INTERACTOR_STYLE_CAMERA_NONE);
  }
  myOtherPoint = myPoint = QPoint(x, y);
  if (ForcedState != VTK_INTERACTOR_STYLE_CAMERA_NONE) {
    startOperation(ForcedState);
  }
  else {
    if (!(ctrl||shift)){
      if ( myIsLeftButtonDown ){
	startOperation(VTK_INTERACTOR_STYLE_CAMERA_ZOOM);
      }
      else{
	startOperation(VTK_INTERACTOR_STYLE_CAMERA_PAN);
      }
    }
  }
}

//----------------------------------------------------------------------------
void 
VVTK_InteractorStyle
::OnLeftButtonUp(int ctrl, int shift, int x, int y)
{
  myIsLeftButtonDown = false;
  SVTK_InteractorStyle::OnLeftButtonUp( ctrl, shift, x, y );

  if ( myIsMidButtonDown )
    OnMiddleButtonDown( ctrl, shift, x, y );
}

//----------------------------------------------------------------------------
void  
VVTK_InteractorStyle
::OnMiddleButtonUp(int ctrl, int shift, int x, int y)
{
  myIsMidButtonDown = false;
  SVTK_InteractorStyle::OnMiddleButtonUp( ctrl, shift, x, y );

  if ( myIsLeftButtonDown )
    OnLeftButtonDown( ctrl, shift, x, y );
}
//----------------------------------------------------------------------------
void  VVTK_InteractorStyle::onFinishOperation()
{
  Superclass::onFinishOperation();
}


//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::SetInteractor( vtkRenderWindowInteractor* theInteractor )
{
  // register EventCallbackCommand as observer of standard events (keypress, mousemove, etc)
  Superclass::SetInteractor( theInteractor );
 
  if(theInteractor){ 
    theInteractor->AddObserver( VISU::SetSMDecreaseMagnificationEvent, EventCallbackCommand, Priority );
    theInteractor->AddObserver( VISU::SetSMIncreaseMagnificationEvent, EventCallbackCommand, Priority );
  }
}


//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::ProcessEvents( vtkObject* object,
		 unsigned long event,
		 void* clientData, 
		 void* callData )
{
  if ( clientData ) {
    vtkObject* anObject = reinterpret_cast<vtkObject*>( clientData );
    VVTK_InteractorStyle* self = dynamic_cast<VVTK_InteractorStyle*>( anObject );
    if( self ){
      switch ( event ) {
      case VISU::SetSMDecreaseMagnificationEvent:
	self->mySMDecreaseMagnificationBtn = *((int*)callData);
	return;
      case VISU::SetSMIncreaseMagnificationEvent:
	self->mySMIncreaseMagnificationBtn = *((int*)callData);
	return;
      }
    }
  }

  Superclass::ProcessEvents( object, event, clientData, callData );
}


//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::onSpaceMouseButton( int button )
{
  if( mySMDecreaseMagnificationBtn == button )    
    DecreaseGaussPointMagnification();
  if( mySMIncreaseMagnificationBtn == button )    
    IncreaseGaussPointMagnification();

  Superclass::onSpaceMouseButton( button );
}

//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::DecreaseGaussPointMagnification()
{
  Interactor->InvokeEvent(VISU::SetSMDecreaseMagnificationEvent,NULL);
}

//----------------------------------------------------------------------------
void
VVTK_InteractorStyle
::IncreaseGaussPointMagnification()
{
  Interactor->InvokeEvent(VISU::SetSMIncreaseMagnificationEvent,NULL);
}
//============================================================================
//
//  Controllers
//
vtkStandardNewMacro(VVTK_ControllerIncrement);
//----------------------------------------------------------------------------
VVTK_ControllerIncrement::VVTK_ControllerIncrement()
{
  myIncrement=10;
}
//----------------------------------------------------------------------------
VVTK_ControllerIncrement::~VVTK_ControllerIncrement()
{
}
//----------------------------------------------------------------------------
int VVTK_ControllerIncrement::Increase()
{
  myIncrement*=2;
  return myIncrement;
}
//----------------------------------------------------------------------------
int VVTK_ControllerIncrement::Decrease()
{
  myIncrement/=2;
  if (!myIncrement){
    myIncrement=1;
  }
  return myIncrement;
}
//
vtkStandardNewMacro(VVTK_ControllerOnKeyDown);
//----------------------------------------------------------------------------
VVTK_ControllerOnKeyDown::VVTK_ControllerOnKeyDown()
{
}
//----------------------------------------------------------------------------
VVTK_ControllerOnKeyDown::~VVTK_ControllerOnKeyDown()
{
}
//----------------------------------------------------------------------------
bool VVTK_ControllerOnKeyDown::OnKeyDown(vtkInteractorStyle* theIS)
{
  SVTK_InteractorStyle *pIS=dynamic_cast<SVTK_InteractorStyle *>(theIS);
  if (pIS){
    if(pIS->GetSelector()->SelectionMode()==GaussPointSelection){
      char key = pIS->GetInteractor()->GetKeyCode();
      //
      if (key == 'S') {
	pIS->ActionPicking();
	return false;
      }
    }
  }
  return Superclass::OnKeyDown(theIS);
}
