//  Copyright (C) 2005-2008  OPEN CASCADE
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
///////////////////////////////////////////////////////////
// File    : SIERPINSKY_Gen_i.cxx
// Author  : Vadim SANDLER (OCN)
// Created : 13/07/05
///////////////////////////////////////////////////////////
//
#include "SIERPINSKY_Gen_i.hxx"
#include <MED_Factory.hxx>
#include <gd.h>

using namespace MED;

/*!
 * Engine factory
 */
extern "C"
{
  PortableServer::ObjectId * SIERPINSKYEngine_factory( CORBA::ORB_ptr            orb,
						       PortableServer::POA_ptr   poa, 
						       PortableServer::ObjectId* contId,
						       const char*               instanceName,
 						       const char*               interfaceName )
{
  SIERPINSKY_Gen_i* anEngine = new SIERPINSKY_Gen_i( orb, poa, contId, instanceName, interfaceName );
  return anEngine->getId() ;
}
}

/*!
 * Default constructor
 */
SIERPINSKY_Gen_i::SIERPINSKY_Gen_i()
{
}

/*!
 * Constructor
 */
SIERPINSKY_Gen_i::SIERPINSKY_Gen_i( CORBA::ORB_ptr            orb,
				    PortableServer::POA_ptr   poa,
				    PortableServer::ObjectId* contId, 
				    const char*               instanceName, 
				    const char*               interfaceName ) 
: Engines_Component_i( orb, poa, contId, instanceName, interfaceName ) 
{
  // activate servant
  _thisObj = this;
  _id = poa->activate_object(_thisObj);
  // set default values
  Reset();
}

/*!
 * Destructor
 */
SIERPINSKY_Gen_i::~SIERPINSKY_Gen_i()
{
  myPoints.clear();
}
  
/*!
 * Initializes engine with three reference points
 */
void SIERPINSKY_Gen_i::Init( CORBA::Double theX1, CORBA::Double theY1, 
			     CORBA::Double theX2, CORBA::Double theY2, 
			     CORBA::Double theX3, CORBA::Double theY3 )
{
  myRefPoints[0] = MyPoint( theX1, theY1 );
  myRefPoints[1] = MyPoint( theX2, theY2 );
  myRefPoints[2] = MyPoint( theX3, theY3 );
  myPoints.clear();
}
  
/*!
 * Initializes engine with three default reference points: (0.5, 1), (0, 0), (1, 0)
 */
void SIERPINSKY_Gen_i::Reset()
{
  myRefPoints[0] = MyPoint( 0.5, 1.0 );
  myRefPoints[1] = MyPoint( 0.0, 0.0 );
  myRefPoints[2] = MyPoint( 1.0, 0.0 );
  myPoints.clear();
}
  
/*!
 * Generates next iteration point
 */
void SIERPINSKY_Gen_i::NextPoint( CORBA::Double  theX,     CORBA::Double  theY, 
				  CORBA::Long    theIter, 
				  CORBA::Double& theNextX, CORBA::Double& theNextY )
{
  double x = theIter < 1 || theIter > 3 ? theX : ( theX + myRefPoints[ theIter-1 ].myX ) / 2;
  double y = theIter < 1 || theIter > 3 ? theY : ( theY + myRefPoints[ theIter-1 ].myY ) / 2;
  myPoints.push_back( MyPoint( x, y ) );
  theNextX = x;
  theNextY = y;
}
  
/*!
 * Exports data to the JPEG image
 */
CORBA::Boolean SIERPINSKY_Gen_i::ExportToJPEG( const char* theFileName, CORBA::Long theSize )
{
  if ( theSize <= 0 ) return false;

  // open file
  FILE* fileDescriptor = fopen( theFileName, "wb" );
  if ( !fileDescriptor ) return false;

  // create an image
  gdImagePtr image = gdImageCreate( theSize, theSize );
  int white = gdImageColorAllocate( image, 255, 255, 255 );
  int black = gdImageColorAllocate( image,   0,   0,   0 );

  gdImageRectangle( image, 0, 0, theSize-1, theSize-1, white );

  // draw points
  std::list<MyPoint>::const_iterator iter;
  for ( iter = myPoints.begin(); iter != myPoints.end(); ++iter ) {
    gdImageSetPixel( image, (int)( (*iter).myX * theSize ), theSize - (int)( (*iter).myY * theSize ), black );
  }

  // export jpeg image
  gdImageJpeg( image, fileDescriptor, 95 );
  fclose( fileDescriptor );
  gdImageDestroy( image );
  
  return true;
}
  
/*!
 * Exports data to the MED file
 */
CORBA::Boolean SIERPINSKY_Gen_i::ExportToMED( const char* theFileName, CORBA::Double theSize )
{
  TErr anError;
  
  // if file already exists - remove it (MED cannot overwrite files correctly)
  FILE* aFile = fopen( theFileName, "rb" );
  if ( aFile ) {
    fclose( aFile );
    if ( remove( theFileName ) ) return false; // can't remove file
  }
  
  // create MED 2.2 file
  PWrapper aMed = CrWrapper( theFileName, MED::eV2_2 );

  // create 2D mesh
  PMeshInfo aMesh = aMed->CrMeshInfo( 2, "Sierpinsky" );
  aMed->SetMeshInfo( aMesh, &anError );
  if ( anError < 0 ) return false;

  // create nodes
  TFloatVector nodes;
  TIntVector   connect;
  std::list<MyPoint>::const_iterator iter;
  int ind = 1;
  for ( iter = myPoints.begin(); iter != myPoints.end(); ++iter ) {
    nodes.push_back( (*iter).myX * theSize );
    nodes.push_back( (*iter).myY * theSize );
    connect.push_back( ind++ );
  }
  PNodeInfo aNodes = aMed->CrNodeInfo( aMesh, nodes, MED::eFULL_INTERLACE, MED::eCART, TStringVector(2), TStringVector(2), TIntVector( myPoints.size() ), TIntVector() );
  aMed->SetNodeInfo( aNodes, &anError );
  if ( anError < 0 ) return false;

  PCellInfo aCells = aMed->CrCellInfo( aMesh, MED::eMAILLE, MED::ePOINT1, connect, eNOD, TIntVector( myPoints.size() ), TIntVector( myPoints.size() ) );
  aMed->SetCellInfo( aCells, &anError );
  if ( anError < 0 ) return false;

  return true;
}
