/*
 * cmsfs-fuse - CMS EDF filesystem support for Linux
 * Allocation map functions.
 *
 * Copyright IBM Corp. 2010
 * Author(s): Jan Glauber <jang@linux.vnet.ibm.com>
 */

#define _GNU_SOURCE
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <stdint.h>
#include "zt_common.h"
#include "helper.h"
#include "edf.h"
#include "cmsfs-fuse.h"

/*
 * Get block number from address.
 */
static int amap_blocknumber(off_t addr)
{
	return addr / BYTES_PER_BLOCK;
}

/*
 * Get the block number for a specific level.
 */
static int amap_blocknumber_level(int level, off_t addr)
{
	int entry = amap_blocknumber(addr);

	while (level-- > 1)
		entry /= PTRS_PER_BLOCK;
	return entry;
}

/*
 * Return address of to the allocation map for a block number > 0.
 */
static off_t get_amap_addr(int level, off_t addr, off_t ptr)
{
	int block = amap_blocknumber_level(level, addr);

	if (cmsfs.amap_levels == 0)
		return cmsfs.amap;

	if (level--) {
		ptr = get_fixed_pointer(ptr + block * PTR_SIZE);
		if (!ptr)
			DIE("amap invalid ptr at addr: %lx\n",
				ptr + block * PTR_SIZE);
		return get_amap_addr(level, addr, ptr);
	}
	return ptr;
}

/*
 * Mark disk address as allocated in alloc map. Unaligned addr is tolerated.
 */
static void amap_block_set(off_t addr)
{
	off_t amap = get_amap_addr(cmsfs.amap_levels, addr, cmsfs.amap);
	int rc, block = amap_blocknumber(addr);
	unsigned int byte, bit;
	u8 entry;

	if (block > 0)
		addr -= block * BYTES_PER_BLOCK;

	addr >>= BITS_PER_DATA_BLOCK;
	byte = addr / 8;
	bit = addr % 8;

	rc = _read(&entry, sizeof(entry), amap + byte);
	BUG(rc < 0);

	/* already used */
	BUG(entry & (1 << (7 - bit)));

	entry |= (1 << (7 - bit));
	rc = _write(&entry, sizeof(entry), amap + byte);
	BUG(rc < 0);
}

/*
 * Mark disk address as free in alloc map. Unaligned addr is tolerated.
 */
static void amap_block_clear(off_t addr)
{
	off_t amap = get_amap_addr(cmsfs.amap_levels, addr, cmsfs.amap);
	int rc, block = amap_blocknumber(addr);
	unsigned int byte, bit;
	u8 entry;

	if (block > 0)
		addr -= block * BYTES_PER_BLOCK;

	addr >>= BITS_PER_DATA_BLOCK;
	byte = addr / 8;
	bit = addr % 8;

	rc = _read(&entry, sizeof(entry), amap + byte);
	BUG(rc < 0);

	/* already cleared */
	BUG(!(entry & (1 << (7 - bit))));

	entry &= ~(1 << (7 - bit));
	rc = _write(&entry, sizeof(entry), amap + byte);
	BUG(rc < 0);
}

/*
 * Return the first free bit in one byte.
 */
static int find_first_empty_bit(u8 entry)
{
	u8 i;

	for (i = 0; i < 8; i++)
		if (!(entry & 1 << (7 - i)))
			return i;
	/* unreachable */
	return -1;
}

/*
 * Look for the first unallocated block and return addr of allocated block.
 */
static off_t __get_free_block(int level, off_t amap, int block_nr)
{
	off_t ptr, addr = amap;
	unsigned int bit;
	int left, rc, i;
	u8 entry;

	if (level > 0) {
		level--;
		left = PTRS_PER_BLOCK;
		while (left--) {
			ptr = get_fixed_pointer(addr);
			if (!ptr)
				return 0;
			ptr = __get_free_block(level, ptr, block_nr);
			if (ptr)
				return ptr;
			addr += PTR_SIZE;
			block_nr++;
		}
		return 0;
	}

	for (i = 0; i < cmsfs.blksize; i++) {
		rc = _read(&entry, sizeof(entry), amap + i);
		BUG(rc < 0);

		if (entry != 0xff) {
			/* get first empty bit and add to addr */
			bit = find_first_empty_bit(entry);

			/* bit -> addr */
			addr = (i * cmsfs.blksize * 8 + cmsfs.blksize * bit)
				+ (block_nr * BYTES_PER_BLOCK);
			amap_block_set(addr);
			return addr;
		}
	}
	return 0;
}

/*
 * Allocate a free block and increment label block counter.
 */
off_t get_free_block(void)
{
	off_t addr;

	if (cmsfs.used_blocks + cmsfs.reserved_blocks >= cmsfs.total_blocks)
		return -ENOSPC;
	addr = __get_free_block(cmsfs.amap_levels, cmsfs.amap, 0);
	BUG(!addr);

	cmsfs.used_blocks++;
	return addr;
}

/*
 * Allocate a zero-filled block and increment label block counter.
 */
off_t get_zero_block(void)
{
	off_t addr = get_free_block();
	int rc;

	if (addr < 0)
		return -ENOSPC;

	rc = _zero(addr, cmsfs.blksize);
	if (rc < 0)
		return rc;
	return addr;
}

/*
 * Free a block and decrement label block counter.
 */
void free_block(off_t addr)
{
	if (addr) {
		amap_block_clear(addr);
		cmsfs.used_blocks--;
	}
}
