# Code for the Gnome route editor
#
# Copyright (C) 1996-2004 Chris Lawrence
# This file may be freely distributed under the terms of the RoutePlanner
# license.  A copy should appear as 'LICENSE' in the archive that this
# file was included in.
#
# $Id: gredit.py,v 1.23 2004/06/20 09:45:09 lordsutch Exp $

import pygtk
pygtk.require('2.0')

import rpdbase
import gtk
import gtk.gdk
import gnome
import gnome.ui

APP = 'RouteEdit'
VERSION = rpdbase.VERID
COPYRIGHT = "Copyright (C) 1996-2004 Chris Lawrence"
gnome.init(APP, VERSION)

import gtk.glade
import gobject
import gconf

import sys, os, glob, commands, time
import rpcountry
import rpcity
import rproute
from rpcitylist import CityList
from rpprogress import ProgressWin

from rpunits import *

GUIFILE = '/usr/share/routeplanner/routeedit.glade2'

# i18n
DIR = 'i18n'

import locale, gettext
locale.setlocale (locale.LC_ALL, '')
gettext.bindtextdomain (APP, DIR)
gettext.textdomain (APP)
gettext.install (APP, DIR, unicode=1)

# Whether or not we use the new file selector
newfilesel = 'FileChooserDialog' in dir(gtk)

def hide_window(widget, event=None):
    widget.hide()
    return True

def clip(minval, x, maxval):
    'Clips the value of x to minval <= x <= maxval'
    return min(max(minval, x), maxval)

# Convenience functions for old GNOME dialogs
def WarningDialog(parent, message_format, *args):
    if args:
        message_format = message_format % args
    dialog = gtk.MessageDialog(parent, 0, gtk.MESSAGE_WARNING,
                               gtk.BUTTONS_CLOSE, message_format)
    return dialog

def ErrorDialog(parent, message_format, *args):
    if args:
        message_format = message_format % args
    dialog = gtk.MessageDialog(parent, 0, gtk.MESSAGE_ERROR,
                               gtk.BUTTONS_CLOSE, message_format)
    return dialog

def QuestionDialog(parent, message_format, *args):
    if args:
        message_format = message_format % args
    dialog = gtk.MessageDialog(parent, 0, gtk.MESSAGE_QUESTION,
                               gtk.BUTTONS_YES_NO, message_format)
    return dialog

class Application:
    def __init__(self, filename=''):
        if os.path.exists('routeedit.glade2'):
            self.wtree = gtk.glade.XML('routeedit.glade2')
        else:
            self.wtree = gtk.glade.XML(GUIFILE)
        
        self.appwin = self.wtree.get_widget('app1')
        dict = {}
        for key in dir(self.__class__):
            dict[key] = getattr(self, key)

        self.wtree.signal_autoconnect(dict)
        self.db = rpdbase.RPDatabase()

        self.citylist = self.wtree.get_widget('citylist')
        self.roadlist = self.wtree.get_widget('roadlist')

        # Wire up the city list
        self.citystore = gtk.ListStore(gobject.TYPE_PYOBJECT,
                                       gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        col = gtk.TreeViewColumn('City', renderer, text=1)
        self.citylist.append_column(col)
        self.citylist.set_reorderable(False)
        self.citylist.set_model(self.citystore)
        self.citysel = self.citylist.get_selection()
        self.citysel.set_mode(gtk.SELECTION_SINGLE)
        self.citysel.connect("changed", self.on_citylist_select_row)

        # Wire up the road list
        self.roadstore = gtk.ListStore(gobject.TYPE_PYOBJECT,
                                       gobject.TYPE_PYOBJECT,
                                       gobject.TYPE_STRING,
                                       gobject.TYPE_STRING,
                                       gobject.TYPE_STRING,
                                       gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        col = gtk.TreeViewColumn('Destination', renderer, text=2)
        self.roadlist.append_column(col)
        col = gtk.TreeViewColumn('Route', renderer, text=3)
        self.roadlist.append_column(col)
        col = gtk.TreeViewColumn('Distance', renderer, text=4)
        self.roadlist.append_column(col)
        col = gtk.TreeViewColumn('Type', renderer, text=5)
        self.roadlist.append_column(col)
        self.roadlist.set_reorderable(False)
        self.roadlist.set_model(self.roadstore)
        self.roadsel = self.roadlist.get_selection()
        self.roadsel.set_mode(gtk.SELECTION_SINGLE)
        self.roadsel.connect("changed", self.on_roadlist_select_row)

        self.dbname = ''
        self.modified = False
        self.new_database()
        if filename:
            self.open_database(filename)

        menu = gtk.Menu()
        menu2 = gtk.Menu()
        menu.show()
        menu2.show()
        for type in rpdbase.classify:
            item = gtk.MenuItem(type)
            item.show()
            menu.append(item)
            item2 = gtk.MenuItem(type)
            item2.show()
            menu2.append(item2)

        selector = self.wtree.get_widget('routesel')
        selector.set_menu(menu)
        selector = self.wtree.get_widget('routesel2')
        selector.set_menu(menu2)

        menu = gtk.Menu()
        menu.show()

        countries = rpcountry.countries.items()
        countries.sort(lambda x, y: cmp(x[1], y[1]))
        self.countrylookup = countries

        countries = [x[1] for x in self.countrylookup]

        combo = self.wtree.get_widget('countrynamecombo')
        combo.set_popdown_strings(countries)
        combo.set_value_in_list(True, False)

        # Make sure dialogs are just hidden, not destroyed...
        for dialog in ('routeprop', 'cityprop', 'dbaseproperties',
                       'choosecity', 'checkdialog'):
            d = self.wtree.get_widget(dialog)
            d.connect('close', hide_window)
            d.connect('delete_event', hide_window)

        self.citywin = None
        self.findtext = ''
        self.appwin.connect("destroy", self.quit) 
        self.appwin.show()

    def set_titlebar(self):
        modw = ''
        if self.modified: modw = '* '

        if self.dbname:
            self.appwin.set_title('RouteEdit: '+modw+self.dbname)
        else:
            self.appwin.set_title('RouteEdit: '+modw+'(untitled)')

    def citybuttons(self, state=True):
        self.wtree.get_widget('removecity').set_sensitive(state)
        self.wtree.get_widget('cityprops').set_sensitive(state)
        self.wtree.get_widget('addroute').set_sensitive(state)

    def routebuttons(self, state=True):
        widgets = ['extendroute', 'removeroute', 'jump', 'routeproperties',
                   'splitroute']
        for widget in widgets:
            self.wtree.get_widget(widget).set_sensitive(state)

    def new_database(self):
        del self.db
        self.db = rpdbase.RPDatabase()
        self.citylist.get_model().clear()
        self.roadlist.get_model().clear()
        self.modified = False
        self.dbname = ''
        self.update_propswin()
        self.citybuttons(False)
        self.routebuttons(False)
        self.wtree.get_widget('routeframe').set_label("Routes")
        self.set_titlebar()

    def open_database(self, filename):
        progwin = ProgressWin(os.path.basename(filename)+'...', self.appwin)
        progwin.show()
        try:
            self.db = rpdbase.RPDatabase(filename, quiet=1,
                                         progressbar=progwin)
        except (rpdbase.invalidformat, IOError), x:
            progwin.destroy()
            error = ErrorDialog(self.appwin, 'Unable to open %s:\n\n%s',
                                filename, str(x))
            error.run()
            return

        progwin.destroy()
        #self.make_widgets_sensitive()
        self.roadstore.clear()
        self.citystore.clear()

        for city in self.db.cities:
            self.citystore.append( [city, unicode(city)] )
            
        self.citybuttons(False)
        self.wtree.get_widget('routeframe').set_label("Routes")
        self.routebuttons(False)
        self.modified = False
        self.dbname = filename
        self.update_propswin()
        self.set_titlebar()

    def save_database(self, filename):
        progwin = ProgressWin(os.path.basename(filename)+'...', self.appwin)
        progwin.show()
        try:
            self.db.Save(filename, progressbar=progwin)
        except (rpdbase.invalidformat, IOError, OSError), x:
            progwin.destroy()
            error = GnomeErrorDialog('Unable to save '+filename+':\n\n'+
                                     str(x), self.appwin)
            error.run()
            error.destroy()
            return

        #self.make_widgets_sensitive()
        progwin.destroy()
        self.modified = False
        self.dbname = filename
        self.set_titlebar()

    def quit(self, *args):
        if self.modified:
            x = QuestionDialog(self.appwin, 'Database modified.  Do you '
                               'really want to quit?')
            response = x.run()
            x.destroy()
            if response != gtk.RESPONSE_YES:
                return True
        
        self.really_quit()

    def really_quit(self, clicked=False):
        if not clicked:
            self.appwin.hide()
            raise SystemExit
        else:
            self.appwin.show()

    def on_new_file1_activate(self, *args):
        if self.modified:
            x = QuestionDialog(self.appwin, 'Database modified.  Do you '
                               'really want to start a new one?')
            response = x.run()
            x.destroy()
            if response != gtk.RESPONSE_YES:
                return

        self.new_database()

    def on_open1_activate(self, *args):
        if self.modified:
            x = QuestionDialog(self.appwin, 'Database modified.  Do you '
                               'really want to open another?')
            response = x.run()
            x.destroy()
            if response != gtk.RESPONSE_YES:
                return

        if newfilesel:
            fs = gtk.FileChooserDialog('Select database', self.appwin,
                                       gtk.FILE_CHOOSER_ACTION_OPEN,
                                       (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                        gtk.STOCK_OPEN, gtk.RESPONSE_ACCEPT))
            fs.set_filename(self.dbname)
            #fs.set_current_folder('/usr/share/routeplanner')
            response = fs.run()
            fs.hide()
            if response == gtk.RESPONSE_ACCEPT:
                self.open_database(fs.get_filename())
            fs.destroy()
        else:
            self.fw = gtk.FileSelection('Select database')
            self.fw.set_transient_for(self.appwin)
            self.fw.selection_entry.connect("activate",
                                             self.on_open_ok_clicked)
            self.fw.ok_button.connect("pressed", self.on_open_ok_clicked)
            self.fw.cancel_button.connect("pressed", self.fw.destroy)
            self.fw.set_modal(True)
            self.fw.hide_on_delete()
            self.fw.set_filename(self.dbname)
            self.fw.show()

    def update_propswin(self):
        self.wtree.get_widget('author').set_text(self.db.author)
        self.wtree.get_widget('authoremail').set_text(self.db.author_email)
        self.wtree.get_widget('comment').set_text(self.db.comment)
        xmap = { 'metric': UNITS_METRIC, 'units_us': UNITS_US,
                 'imperial': UNITS_IMPERIAL }
        for widget, units in xmap.items():
            self.wtree.get_widget(widget).set_active(False)
            if self.db.units == units:
                self.wtree.get_widget(widget).set_active(True)

    def on_propswin_changed(self, *args):
        changed = False
        
        if self.db.author != self.wtree.get_widget('author').get_text():
            changed = True
        elif self.db.author_email != \
             self.wtree.get_widget('authoremail').get_text():
            changed = True
        elif self.db.comment != self.wtree.get_widget('comment').get_text():
            changed = True
        else:
            xmap = { 'metric': UNITS_METRIC, 'units_us': UNITS_US,
                     'imperial': UNITS_IMPERIAL }
            for widget, units in xmap.items():
                if self.wtree.get_widget(widget).get_active():
                    if self.db.units != units:
                        changed = True

        self.wtree.get_widget('dbaseproperties').set_response_sensitive(
            gtk.RESPONSE_APPLY, changed)

    def on_dbaseproperties_response(self, propswin, response, *args):
        if response in (gtk.RESPONSE_APPLY, gtk.RESPONSE_OK):
            self.db.author = self.wtree.get_widget('author').get_text()
            self.db.author_email = self.wtree.get_widget('authoremail').get_text()
            self.db.comment = self.wtree.get_widget('comment').get_text()
            xmap = { 'metric': UNITS_METRIC, 'units_us': UNITS_US,
                     'imperial': UNITS_IMPERIAL }
            for widget, units in xmap.iteritems():
                if self.wtree.get_widget(widget).get_active():
                    self.db.units = units

            self.wtree.get_widget('dbaseproperties').set_response_sensitive(
                gtk.RESPONSE_APPLY, False)

            self.modified = True
            self.set_titlebar()

        if response != gtk.RESPONSE_APPLY:
            propswin.hide()

    def on_open_ok_clicked(self, *args):
        self.fw.hide()
        self.open_database(self.fw.get_filename())
        self.fw.destroy()

    def on_save1_activate(self, *args):
        if not self.dbname:
            self.on_save_as1_activate()
        else:
            self.save_database(self.dbname)

    def on_save_as1_activate(self, *args):
        if newfilesel:
            fs = gtk.FileChooserDialog('Save database as...', self.appwin,
                                       gtk.FILE_CHOOSER_ACTION_SAVE,
                                       (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                        gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))
            fs.set_filename(self.dbname)
            response = fs.run()
            fs.hide()
            if response == gtk.RESPONSE_ACCEPT:
                filename = fs.get_filename()
                self.save_database(filename)
            fs.destroy()
        else:
            self.fw = gtk.FileSelection('Save database as...')
            self.fw.set_transient_for(self.appwin)
            self.fw.selection_entry.connect("activate",
                                             self.on_save_ok_clicked)
            self.fw.ok_button.connect("pressed", self.on_save_ok_clicked)
            self.fw.cancel_button.connect("pressed", lambda x: self.fw.destroy())
            self.fw.set_modal(True)
            self.fw.hide_on_delete()
            self.fw.set_filename(self.dbname)
            self.fw.show()
        
    def on_save_ok_clicked(self, *args):
        self.fw.hide()
        self.save_database(self.fw.get_filename())
        self.fw.destroy()

    def on_about1_activate(self, *args):
        about = gnome.ui.About(
            APP, VERSION, COPYRIGHT,
            _("Highway database editor for RoutePlanner."),
            ['Chris Lawrence <lawrencc@debian.org>'],
            [], '')
        #gtk.gdk.pixbuf_new_from_file(LOGO)
        about.show()

    def on_properties1_activate(self, *args):
        props = self.wtree.get_widget('dbaseproperties')
        props.hide_on_delete()
        props.show()

    def on_check_database_activate(self, *args):
        bits = {}
        
        for city in self.db.cities:
            routes = city.roads
            bits.setdefault(len(routes), []).append(unicode(city))

        output = []
        for i in range(0, 3):
            word = 'routes'
            if i == 1: word = 'route'
            
            output += [(u"Cities with %d %s:\n  " % (i, word))+
                       (u"\n  ".join(bits.get(i, [])) or
                        u"None")]
            
        text = u'\n\n'.join(output)
        buff = gtk.TextBuffer()
        self.wtree.get_widget('checkview').set_buffer(buff)
        buff.set_text(text)
        dialog = self.wtree.get_widget('checkdialog')
        dialog.hide_on_delete()
        dialog.connect('response', lambda w, *args: w.hide())
        dialog.show()

    def on_citylist_select_row(self, sel, *args):
        if not sel.get_selected()[1]:
            self.citybuttons(False)
            self.routebuttons(False)
            self.roadstore.clear()
            return

        roadmod, row = self.roadsel.get_selected()
        if row:
            thisroute = roadmod.get_value(row, 0)
        else:
            thisroute = None

        self.roadstore.clear()
        citymod, seliter = self.citysel.get_selected()
        city = citymod.get_value(seliter, 0)
        cityname = unicode(city)
        self.wtree.get_widget('routeframe').set_label(u'Routes from '+cityname)
        selected = False
        for (i, route) in enumerate(city.roads):
            picked = route[2]
            self.roadstore.append( [route, picked, unicode(picked),
                                    route[0].name, unicode(route[0].distance),
                                    rpdbase.declassify[route[0].speedcode]] )
            if route[0] is thisroute:
                self.roadsel.select_path(i)
                selected = True

        self.citybuttons(True)
        self.routebuttons(selected)

    def on_roadlist_select_row(self, *args):
        self.routebuttons(True)

    def on_find1_activate(self, *args):
        findstr = self.wtree.get_widget('citytofind')
        findwin = self.wtree.get_widget('findwin')
        text = findstr.get_text()
        findwin.set_response_sensitive(gtk.RESPONSE_ACCEPT, bool(text))
        findwin.show()
        #findstr.grab_focus()

    def on_citytofind_changed(self, *args):
        text = self.wtree.get_widget('citytofind').get_text()
        self.wtree.get_widget('findwin').set_response_sensitive(
            gtk.RESPONSE_ACCEPT, bool(text))

    def on_findwin_response(self, findwin, response, *args):
        if response == gtk.RESPONSE_ACCEPT:
            text = self.wtree.get_widget('citytofind').get_text()
            self.findtext = text
            self.citysel.unselect_all()
            self.on_find_again1_activate()
        findwin.hide()

    def on_find_again1_activate(self, *args):
        self.findtext = self.findtext.lower()
        model, it = self.citysel.get_selected()
        if not it:
            it = self.citystore.get_iter_first()
            if not it:
                dialog = WarningDialog(self.appwin, 'Nothing to find.')
                dialog.run()
                dialog.hide()
                return
            word = ''
        else:
            it = self.citystore.iter_next(it)
            word = 'more '

        while it:
            cityname = self.citystore.get_value(it, 1)
            if self.findtext in cityname.lower():
                self.citysel.select_iter(it)
                pos = self.citystore.get_path(it)
                self.citylist.scroll_to_cell(pos, None, False, 0.5, 0)
                return
            it = self.citystore.iter_next(it)
        
        dialog = WarningDialog(self.appwin, 'No %scities matching %s.',
                               word, self.findtext)
        dialog.run()
        dialog.destroy()

    def on_citylist_button_press_event(self, button, event, *args):
        if event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1 \
           and self.citysel.get_selected()[1]:
            self.on_cityprops_clicked()

    def on_countrybutton_clicked(self, *args):
        self.wtree.get_widget('countrysel').show()

    def on_cityprop_response(self, citywin, result, *args):
        if result != gtk.RESPONSE_OK:
            citywin.hide()
            self.editing = None
            return

        if self.editing is not None:
            city = self.editing
        else:
            city = rpcity.City()

        cname, sname = (self.wtree.get_widget('cityname').get_text(),
                        self.wtree.get_widget('statename').get_text())
        sname = rpdbase.expand_state(sname)
        country = self.wtree.get_widget('countryname').get_text()
        code = rpcountry.code_for[country]

        cities = self.db.CitiesMatching(cname+', '+sname)
        cities = filter(lambda x, city=city: x is not city, cities)
	for c in cities:
            if c.city == cname and c.state == sname and \
               c.country == code:
                citywin.hide()
                error = ErrorDialog(citywin,
                                    u'There is already a city named %s.', c)
                error.run()
                error.destroy()
                return

        city.update_info(cname, sname, code,
                         self.wtree.get_widget('longitude').get_value(),
                         self.wtree.get_widget('latitude').get_value(),
                         self.wtree.get_widget('interchange').get_active())

        self.modified = True
        self.set_titlebar()
        if self.editing is not None:
            self.db.cities.remove(self.editing)

        # Insert city in list
        thiscity = city
        self.db.cities.append(city)
        self.db.Sort()

        self.citystore.clear()
        for (i, city) in enumerate(self.db.cities):
            self.citystore.append( [city, unicode(city)] )
            if city is thiscity:
                self.citysel.select_path(i)
                self.citylist.scroll_to_cell((i,), None, False, 0.5, 0)
        citywin.hide()

    def on_addcity_clicked(self, *args):
        citywin = self.wtree.get_widget('cityprop')
        self.wtree.get_widget('cityname').set_text('')
        #self.wtree.get_widget('statename').set_text('')
        #self.wtree.get_widget('countryname').set_text('United States')
        self.wtree.get_widget('longitude').set_value(0.0)
        self.wtree.get_widget('latitude').set_value(0.0)
        self.wtree.get_widget('interchange').set_active(False)
        self.wtree.get_widget('cityname').grab_focus()
        self.editing = None
        citywin.show()

    def on_statename_activate(self, *args):
        sname = self.wtree.get_widget('statename').get_text()
        if sname and len(sname) == 2:
            sname = rpdbase.expand_state(sname)

        if sname:
            self.wtree.get_widget('statename').set_text(sname)
            cname = rpdbase.autodetect_country(sname)
            if cname:
                cname = rpcountry.expand_country(cname)
                self.wtree.get_widget('countryname').set_text(cname)

    def on_cityprops_clicked(self, *args):
        citymod, selit = self.citysel.get_selected()
        if not selit:
            return

        city = self.editing = citymod.get_value(selit, 0)

        citywin = self.wtree.get_widget('cityprop')
        self.wtree.get_widget('cityname').set_text(city.city)
        self.wtree.get_widget('statename').set_text(city.state)
        cname = rpcountry.expand_country(city.country)
        self.wtree.get_widget('countryname').set_text(cname)
        self.wtree.get_widget('longitude').set_value(city.longitude)
        self.wtree.get_widget('latitude').set_value(city.latitude)
        self.wtree.get_widget('interchange').set_active(city.intersection)
        self.wtree.get_widget('cityname').grab_focus()
        citywin.show()

    def on_removecity_clicked(self, *args):
        model, it = self.citysel.get_selected()
        if not it:
            return

        dialog = QuestionDialog(self.appwin, 'Remove this city and all '
                                'connected routes?')
        response = dialog.run()
        dialog.destroy()
        if response == gtk.RESPONSE_YES:
            thiscity = model.get_value(it, 0)
            item = model.get_path(it)

            roads = [x[0] for x in thiscity.roads]
            for road in roads:
                for city in road.city:
                    city.roads = [route for route in city.roads
                                  if route[0] is not road]
                try:
                    self.db.routes.remove(road)
                except ValueError:
                    pass

            del roads
            self.db.cities.remove(thiscity)
            self.db.Rehash()

            self.citystore.clear()
            for city in self.db.cities:
                self.citystore.append( [city, unicode(city)] )

            self.citysel.select_path(item)
            self.citylist.scroll_to_cell(item, None, False, 0.5, 0)

            self.modified = True
            self.set_titlebar()

    def on_roadlist_button_press_event(self, button, event, *args):
        if event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1 \
           and self.roadsel.get_selected()[1]:
            self.on_routeproperties_clicked()

    def on_jump_clicked(self, *args):
        model, it = self.roadsel.get_selected()
        if not it: return

        route = self.roadstore.get_value(it, 0)
        othercity = self.roadstore.get_value(it, 1)
        for (i, city) in enumerate(self.citystore):
            if city[0] is othercity:
                self.citysel.select_path(i)
                self.citylist.scroll_to_cell(i, None, False, 0.5, 0)
                while gtk.events_pending():
                    gtk.main_iteration(False)
                for (j, road) in enumerate(self.roadstore):
                    if road[0][0] is route[0]:
                        self.roadsel.select_path(j)
                        self.roadlist.scroll_to_cell(j, None, False, 0.5, 0)
                        break
                return

    def on_removeroute_clicked(self, *args):
        model, it = self.roadsel.get_selected()
        if not it: return

        road = model.get_value(it, 0)
        for city in road[0].city:
            city.roads = [route for route in city.roads
                          if route[0] is not road[0]]
            
        self.db.routes.remove(road[0])
        model.remove(it)

        self.modified = True
        self.set_titlebar()

    def show_via_entries(self, show=True):
        WIDGETS = ['vialabel', 'city3', 'choosecity3', 'exitlabel3',
                   'exit3', 'rightbox']
        if show:
            [self.wtree.get_widget(w).show() for w in WIDGETS]
        else:
            [self.wtree.get_widget(w).hide() for w in WIDGETS]
        self.update_box_labels()

    def update_box_labels(self):
        fullbox = self.wtree.get_widget('rightbox').flags() & gtk.VISIBLE
        if fullbox:
            via = self.wtree.get_widget('city3').get_text()
        else:
            via = self.wtree.get_widget('city2').get_text()
        
        self.wtree.get_widget('leftbox').set_label(
            'From '+self.wtree.get_widget('city1').get_text()+' to '+via)
        if fullbox:
            self.wtree.get_widget('rightbox').set_label(
                'From '+via+' to '+self.wtree.get_widget('city2').get_text())
        #self.wtree.get_widget('leftbox').draw_default()
        #self.wtree.get_widget('rightbox').draw_default()

    def on_addroute_clicked(self, *args):
        mod, it = self.citysel.get_selected()
        if not it:
            return
        
        thiscity = mod.get_value(it, 0)
        
        self.wtree.get_widget('city1').set_text(unicode(thiscity))
        for widget in ['routename', 'city2', 'exit1', 'exit2']:
            self.wtree.get_widget(widget).set_text('')
        for widget in ['toll', 'scenic', 'ferry']:
            self.wtree.get_widget(widget).set_active(False)
        if self.db.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(True)
        else:
            self.wtree.get_widget('miles').set_active(True)
        self.wtree.get_widget('distance').set_value(0)
        self.wtree.get_widget('speedtime').set_value(0)
        self.editing = None
        #selector = self.wtree.get_widget('routesel')
        #selector.set_history(0)
        self.show_via_entries(False)
        self.update_box_labels()
        self.wtree.get_widget('routeok').set_sensitive(False)
        self.wtree.get_widget('routeprop').show()
        self.wtree.get_widget('routename').grab_focus()
        self.mode = 'add'
        
    def on_routeproperties_clicked(self, *args):
        mod, it = self.citysel.get_selected()
        if not it:
            return
        
        thiscity = mod.get_value(it, 0)
        mod, it = self.roadsel.get_selected()
        if not it:
            return

        road = mod.get_value(it, 0)
        rinfo, thatcity = road[1], road[2]
        thisexit, thatexit = road[0].exits[rinfo], road[0].exits[1-rinfo]
        self.editing = road

        self.wtree.get_widget('routename').set_text(road[0].name)
        self.wtree.get_widget('city1').set_text(unicode(thiscity))
        self.wtree.get_widget('city2').set_text(unicode(thatcity))
        self.wtree.get_widget('exit1').set_text(thisexit)
        self.wtree.get_widget('exit2').set_text(thatexit)

        for widget in ['toll', 'scenic', 'ferry']:
            self.wtree.get_widget(widget).set_active(
                road[0].flagmap.get(widget, False))

        if road[0].distance.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(True)
        else:
            self.wtree.get_widget('miles').set_active(True)
        
        self.wtree.get_widget('distance').set_value(road[0].distance.value)

        pos = rpdbase.classify.index(rpdbase.declassify[road[0].speedcode])

        selector = self.wtree.get_widget('routesel')
        selector.set_history(pos)
        
        self.wtree.get_widget('speedtime').set_value(road[0].speed)
        self.show_via_entries(False)
        self.wtree.get_widget('routeok').set_sensitive(True)
        self.wtree.get_widget('routename').grab_focus()
        self.wtree.get_widget('routeprop').show()
        self.mode = 'edit'

    def on_routeprop_response(self, routewin, response, *args):
        routewin.hide()
        if response != gtk.RESPONSE_OK:
            self.editing = None
            return

        breakmode = self.mode in ('break', 'extend')

        name = self.wtree.get_widget('routename').get_text()
        city1 = self.wtree.get_widget('city1').get_text().decode('UTF-8')
        city2 = self.wtree.get_widget('city2').get_text().decode('UTF-8')
        city3 = self.wtree.get_widget('city3').get_text().decode('UTF-8')
        exit1 = self.wtree.get_widget('exit1').get_text()
        exit2 = self.wtree.get_widget('exit2').get_text()
        exit3 = self.wtree.get_widget('exit3').get_text()

        if (not name or not city1 or not city2 or (breakmode and not city3)):
            routewin.show()
            win = self.wtree.get_widget('routeprop')
            d = ErrorDialog(win, 'You must include the route name and cities.')
            d.run()
            d.destroy()
            return

        if city1 == city2 or (breakmode and city2 == city3) or \
           (breakmode and city1 == city3):
            routewin.show()
            win = self.wtree.get_widget('routeprop')
            d = ErrorDialog(win, 'All cities selected must be different.')
            d.run()
            d.destroy()
            return

        if self.wtree.get_widget('km').get_active():
            units = UNITS_METRIC
        else:
            units = UNITS_US

        if self.wtree.get_widget('km2').get_active():
            units2 = UNITS_METRIC
        else:
            units2 = UNITS_US

        distance = Distance(self.wtree.get_widget('distance').get_value_as_int(), units)
        distance2 = Distance(self.wtree.get_widget('distance2').get_value_as_int(), units2)
        if not distance or (self.mode in ('break', 'extend') and
                            not distance2):
            routewin.show()
            win = self.wtree.get_widget('routeprop')
            d = ErrorDialog(win, 'Distances must be greater than zero.')
            d.run()
            d.destroy()
            return

        # If we're editing an existing route...
        if self.editing is not None:
            road = self.editing[0]
            for city in road.city:
                city.roads = [route for route in city.roads
                              if route[0] is not road]
            try:
                self.db.routes.remove(road)
            except:
                pass

        flagmap = {}
        for widget in ['toll', 'scenic', 'ferry']:
            flagmap[widget] = self.wtree.get_widget(widget).get_active()

        speed = self.wtree.get_widget('speedtime').get_value_as_int()
        code = self.wtree.get_widget('routesel').get_history()
        speedcode = rpdbase.classifications[rpdbase.classify[code]]

        if self.mode in ('add', 'edit'):
            route = rproute.Route([ city1, city2, str(distance), speed,
                                    speedcode, exit1, exit2, name, flagmap ],
                                  self.db.cityhash)
            self.db.routes.append(route)
        else:
            route = rproute.Route([ city1, city3, str(distance), speed,
                                    speedcode, exit1, exit3, name, flagmap ],
                                  self.db.cityhash)
            self.db.routes.append(route)

            flagmap = {}
            for widget in ['toll', 'scenic', 'ferry']:
                flagmap[widget] = self.wtree.get_widget(widget+'2').\
                                  get_active()

            speed = self.wtree.get_widget('speedtime2').get_value_as_int()
            code = self.wtree.get_widget('routesel2').get_history()
            speedcode = rpdbase.classifications[rpdbase.classify[code]]

            route2 = rproute.Route([ city2, city3, str(distance2), speed,
                                     speedcode, exit2, exit3, name, flagmap ],
                                   self.db.cityhash)
            self.db.routes.append(route2)
            if self.mode == 'extend':
                # Highlight the new segment
                route, route2 = route2, route
            
        thisroute = route
        self.modified = True
        self.set_titlebar()

        mod, it = self.citysel.get_selected()
        city = mod.get_value(it, 0)

        self.roadstore.clear()
        selected = False
        for (i, route) in enumerate(city.roads):
            picked = route[2]
            self.roadstore.append( [route, picked, unicode(picked),
                                    route[0].name, unicode(route[0].distance),
                                    rpdbase.declassify[route[0].speedcode]] )
            if route[0] is thisroute:
                self.roadsel.select_path(i)
                selected = True

        self.citybuttons(True)
        self.routebuttons(selected)

    def on_extendroute_clicked(self, *args):
        mod, it = self.citysel.get_selected()
        if not it:
            return
        
        thiscity = mod.get_value(it, 0)
        mod, it = self.roadsel.get_selected()
        if not it:
            return

        road = mod.get_value(it, 0)
        rinfo, thatcity = road[1], road[2]
        thisexit, thatexit = road[0].exits[rinfo], road[0].exits[1-rinfo]
        self.editing = road
        
        self.show_via_entries(True)
        self.wtree.get_widget('routename').set_text(road[0].name)
        self.wtree.get_widget('city3').set_text(unicode(thiscity))
        self.wtree.get_widget('city2').set_text('')
        self.wtree.get_widget('city1').set_text(unicode(thatcity))
        
        self.wtree.get_widget('exit3').set_text(thisexit)
        self.wtree.get_widget('exit2').set_text('')
        self.wtree.get_widget('exit1').set_text(thatexit)

        for widget in ['toll', 'scenic', 'ferry']:
            status = road[0].flagmap.get(widget, False)
            self.wtree.get_widget(widget).set_active(status)
            self.wtree.get_widget(widget+'2').set_active(status)

        if road[0].distance.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(True)
            self.wtree.get_widget('km2').set_active(True)
        else:
            self.wtree.get_widget('miles').set_active(True)
            self.wtree.get_widget('miles2').set_active(True)
        
        self.wtree.get_widget('distance').set_value(float(road[0].distance))
        self.wtree.get_widget('distance2').set_value(0)

        pos = rpdbase.classify.index(rpdbase.declassify[road[0].speedcode])

        selector = self.wtree.get_widget('routesel')
        selector.set_history(pos)
        selector = self.wtree.get_widget('routesel2')
        selector.set_history(pos)
        
        self.wtree.get_widget('speedtime').set_value(road[0].speed)
        self.wtree.get_widget('speedtime2').set_value(road[0].speed)
        #self.wtree.get_widget('lock').hide()
        self.wtree.get_widget('lock').set_sensitive(False)
        self.wtree.get_widget('lock').set_active(False)
        self.wtree.get_widget('routeok').set_sensitive(False)
        self.update_box_labels()
        self.wtree.get_widget('routename').grab_focus()
        self.wtree.get_widget('routeprop').show()
        self.mode = 'extend'

    def on_splitroute_clicked(self, *args):
        mod, it = self.citysel.get_selected()
        if not it:
            return
        
        thiscity = mod.get_value(it, 0)
        mod, it = self.roadsel.get_selected()
        if not it:
            return

        road = mod.get_value(it, 0)
        rinfo, thatcity = road[1], road[2]
        thisexit, thatexit = road[0].exits[rinfo], road[0].exits[1-rinfo]
        self.editing = road
        
        self.show_via_entries(True)
        self.wtree.get_widget('routename').set_text(road[0].name)
        self.wtree.get_widget('city1').set_text(unicode(thiscity))
        self.wtree.get_widget('city2').set_text(unicode(thatcity))
        self.wtree.get_widget('city3').set_text('')
        
        self.wtree.get_widget('exit1').set_text(thisexit)
        self.wtree.get_widget('exit2').set_text(thatexit)
        self.wtree.get_widget('exit3').set_text('')

        for widget in ['toll', 'scenic', 'ferry']:
            status = road[0].flagmap.get(widget, False)
            self.wtree.get_widget(widget).set_active(status)
            self.wtree.get_widget(widget+'2').set_active(status)

        if road[0].distance.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(True)
            self.wtree.get_widget('km2').set_active(True)
        else:
            self.wtree.get_widget('miles').set_active(True)
            self.wtree.get_widget('miles2').set_active(True)

        self.totaldist = road[0].distance
        self.wtree.get_widget('distance').set_value(float(self.totaldist))
        self.wtree.get_widget('distance2').set_value(0)

        pos = rpdbase.classify.index(rpdbase.declassify[road[0].speedcode])

        selector = self.wtree.get_widget('routesel')
        selector.set_history(pos)
        selector = self.wtree.get_widget('routesel2')
        selector.set_history(pos)
        
        self.wtree.get_widget('speedtime').set_value(road[0].speed)
        self.wtree.get_widget('speedtime2').set_value(road[0].speed)
        self.wtree.get_widget('lock').set_sensitive(True)
        self.wtree.get_widget('lock').set_active(True)
        #self.wtree.get_widget('lock').show()
        self.wtree.get_widget('routeok').set_sensitive(False)
        self.update_box_labels()
        self.wtree.get_widget('routename').grab_focus()
        self.wtree.get_widget('routeprop').show()
        self.mode = 'break'

    def on_miles_toggled(self, *args):
        if not self.wtree.get_widget('routeprop').flags() & gtk.VISIBLE:
            return
        
        from_u, to_u = UNITS_US, UNITS_METRIC
        if self.wtree.get_widget('miles').get_active():
            from_u, to_u = UNITS_METRIC, UNITS_US

        # When locked, change the locked units measure at the same time
        if not (self.wtree.get_widget('miles2').flags() & gtk.SENSITIVE):
            if to_u == UNITS_US:
                self.wtree.get_widget('miles2').set_active(True)
            else:
                self.wtree.get_widget('km2').set_active(True)
        
        dwidget = self.wtree.get_widget('distance')
        distance = Distance(dwidget.get_value(), from_u)
        dwidget.set_value(float(distance.AsUnit(to_u)))

        # Don't convert time
        code = self.wtree.get_widget('routesel').get_history()
        speedcode = rpdbase.classifications[rpdbase.classify[code]]
        if speedcode == 1: return

        dwidget = self.wtree.get_widget('speedtime')
        distance = Distance(dwidget.get_value(), from_u)
        dwidget.set_value(float(distance.AsUnit(to_u)))

    def on_choosecity1_clicked(self, *args):
        if not self.citywin: self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.set_city_widget,
                             self.wtree.get_widget('city1'))

    def on_choosecity2_clicked(self, *args):
        if not self.citywin: self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.set_city_widget,
                             self.wtree.get_widget('city2'))

    def on_choosecity3_clicked(self, *args):
        if not self.citywin: self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.set_city_widget,
                             self.wtree.get_widget('city3'))

    def set_city_widget(self, cityid, citynum, widget):
        widget.set_text(unicode(cityid))
        self.update_box_labels()
        self.update_sensitivity()

    def update_sensitivity(self, *args):
        city1 = self.wtree.get_widget('city1').get_text()
        city2 = self.wtree.get_widget('city2').get_text()
        city3 = self.wtree.get_widget('city3').get_text()
        name = self.wtree.get_widget('routename').get_text()
        val = False
        if city1 and city2 and name and \
           (self.mode not in ('break', 'extend') or city3):
            val = True
        self.wtree.get_widget('routeok').set_sensitive(val)

    def on_lock_toggled(self, *args):
        locked = self.wtree.get_widget('lock').get_active()
        self.wtree.get_widget('miles2').set_sensitive(not locked)
        self.wtree.get_widget('km2').set_sensitive(not locked)
        if locked:
            self.wtree.get_widget('miles2').set_active(
                self.wtree.get_widget('miles').get_active())
            self.wtree.get_widget('km2').set_active(
                self.wtree.get_widget('km').get_active())

    def on_distance_changed(self, *args):
        if not self.wtree.get_widget('rightbox').flags() & gtk.VISIBLE:
            return
        if not self.wtree.get_widget('lock').get_active():
            return

        units = UNITS_US
        if self.wtree.get_widget('km').get_active():
            units = UNITS_METRIC
        
        widget = args[0]
        widget.update()
        total = float(self.totaldist.AsUnit(units))
        if widget == self.wtree.get_widget('distance'):
            dist1 = widget.get_value()
            dist2 = total - dist1
        else:
            dist2 = widget.get_value()
            dist1 = total - dist2

        dist1 = clip(0, dist1, total)
        dist2 = clip(0, dist2, total)

        self.wtree.get_widget('distance').set_value(dist1)
        self.wtree.get_widget('distance2').set_value(dist2)

def main(filename=''):
    app = Application(filename)
    gtk.main()
