#ifndef _RHEOLEF_SPACE_H
#define _RHEOLEF_SPACE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/geo.h"
#include "rheolef/geo_domain.h"
#include "rheolef/element.h"

namespace rheolef {

// =====================================================================
// space_act = a domain + an act (block, unblock)
// =====================================================================

template <class M>
class space_act {
public:

// typedefs:

    enum act_type {
        block = 0,
        unblock = 1
    };

// allocators:

    space_act() : _dom(), _act() {}
    space_act(const domain_indirect_basic<M>& dom, act_type act)
	: _dom(dom), _act(act) {}
    space_act(const space_act<M>& sp)
	: _dom(sp._dom), _act(sp._act) {}

// accessors:

    const domain_indirect_basic<M>& get_domain_indirect() const { return _dom; }
    act_type                    get_act()         const { return _act; }
    
// data:
protected:
    domain_indirect_basic<M> _dom;
    act_type                 _act;
};
// =====================================================================
// space_constitution = a table of acts
// =====================================================================

template <class T, class M>
class space_constitution_rep : public std::vector<space_act<M> > {
public:
    typedef std::vector<space_act<M> >     base;
    typedef typename  base::size_type      size_type;
    typedef typename  base::const_iterator const_iterator;

// allocators:

    space_constitution_rep()
      : base(), _omega(), _element() {}
    space_constitution_rep(const geo_basic<T,M>& omega, std::string approx)
      : base(), _omega(compact(omega)), _element(approx) {}
    space_constitution_rep (const space_constitution_rep<T,M>& scr)
      : base(scr), _omega(scr._omega), _element(scr._element)
    {
	error_macro ("physical copy of space_constitution_rep: size="<< base::size());
    }

// accessors:

    const geo_basic<T,M>& get_geo() const { return _omega; }
    const element<T>& get_element() const { return _element; }

    size_type      size()  const { return base::size(); }
    const_iterator begin() const { return base::begin(); }
    const_iterator end()   const { return base::end(); }

// modifiers

    void set_geo (const geo_basic<T,M>& omega)  { _omega = omega; }
    void set_element (const element<T>& elt)    { _element = elt; }

    void block  (const domain_indirect_basic<M>& dom) {
	base::push_back (space_act<M> (dom, space_act<M>::block));
    }
    void unblock(const domain_indirect_basic<M>& dom) {
	base::push_back (space_act<M> (dom, space_act<M>::unblock));
    }

// comparator:

    bool operator== (const space_constitution_rep<T,M>& V2) const { return _omega == V2._omega && _element == V2._element; }

// data:
protected:
    geo_basic<T,M>                   _omega;
    element<T>                       _element;
};
template <class T, class M>
class space_constitution : public smart_pointer<space_constitution_rep<T,M> > {
public:

    typedef space_constitution_rep<T,M>   rep;
    typedef smart_pointer<rep>            base;
    typedef typename  rep::size_type      size_type;
    typedef typename  rep::const_iterator const_iterator;

// allocators:

    space_constitution()
     : smart_pointer<rep>(new_macro(rep)) {}

    space_constitution(const geo_basic<T,M>& omega, std::string approx)
     : smart_pointer<rep>(new_macro(rep (omega,approx))) {}

// accessors:

    const geo_basic<T,M>& get_geo() const { return base::data().get_geo(); }
    const element<T>& get_element() const { return base::data().get_element(); }

    size_type      size()  const { return base::data().size(); }
    const_iterator begin() const { return base::data().begin(); }
    const_iterator end()   const { return base::data().end(); }

// comparator:

    bool operator== (const space_constitution<T,M>& V2) const { return base::data().operator==(V2.data()); }

// modifiers

    void set_geo (const geo_basic<T,M>& omega) { base::data().set_geo (omega); }
    void set_element (const element<T>& elt)   { base::data().set_element (elt); }

    void block  (const domain_indirect_basic<M>& dom) {
	base::data().block(dom);
    }
    void unblock(const domain_indirect_basic<M>& dom) { base::data().unblock(dom); }
};
// =====================================================================
// a dof = a degree-of-freedom 
//       = space_pair
//       = pair (bool is_blocked ; size_t iub)
// =====================================================================
// TODO: compact the bool as an extra bit in size_type ?
struct space_pair_type {
    typedef array<size_t>::size_type size_type;
    space_pair_type () : _blk(false), _iub (std::numeric_limits<size_type>::max()) {}
    space_pair_type (bool blk, size_type iub) : _blk(blk), _iub(iub) {}
    bool is_blocked() const { return _blk; }
    size_type iub()   const { return _iub; }
    void set_iub (size_type iub) { _iub = iub; }
    void set_blocked (bool blk)  { _blk = blk; }
    friend std::ostream&  operator<< (std::ostream& os, const space_pair_type& x) {
	return os << "{" << x.is_blocked() << "," << x.iub() << "}"; }
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) { ar & _blk; ar & _iub; }
protected:
    bool      _blk;
    size_type _iub;
};

} // namespace rheolef

#ifdef _RHEOLEF_HAVE_MPI
// =====================================================================
// Some serializable types, like geo_element, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::space_pair_type> : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

namespace rheolef {

// forward declaration:
template <class T, class M> class field_basic;

// =====================================================================
// 1) representation: space_base_rep and space_rep
// =====================================================================
template <class T, class M>
class space_base_rep {
public:

// typedefs:

    typedef typename space_pair_type::size_type size_type;

// allocators:

    space_base_rep ();
    space_base_rep (const geo_basic<T,M>& omega, std::string approx);

// accessors:

    const distributor&  ownership() const { return _idof2blk_iub.ownership(); }
    size_type                size() const { return ownership().size(); }
    size_type            dis_size() const { return ownership().dis_size(); }
    const communicator&      comm() const { return ownership().comm(); }

    const geo_basic<T,M>& get_geo() const { return _constit.get_geo(); }
    const element<T>& get_element() const { return _constit.get_element(); }

    std::string get_stamp() const; /// e.g. "P1(square)", for field_expr<Expr> checks

    void block  (const domain_indirect_basic<M>& dom) { no_freeze_guard(); _constit.block  (dom); }
    void unblock(const domain_indirect_basic<M>& dom) {no_freeze_guard(); _constit.unblock(dom);}

    bool                 is_blocked (size_type     idof) const { freeze_guard(); return _idof2blk_iub [idof].is_blocked(); }
    size_type                   iub (size_type     idof) const { freeze_guard(); return _idof2blk_iub [idof].iub(); }

    const distributor& iu_ownership() const { freeze_guard(); return _iu_ownership; }
    const distributor& ib_ownership() const { freeze_guard(); return _ib_ownership; }

    void get_dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const;

    template <class Function>
    T momentum (Function f, size_type idof) const;

    array<size_type, M> build_indirect_array (
    	const domain_indirect_basic<M>& indirect) const;

// comparator:

    bool operator== (const space_base_rep<T,M>& V2) const { 
	return _constit.operator==(V2._constit); } // TODO: compare also blocked/unknown sizes:

    friend bool are_compatible (const space_base_rep<T,M>& V1, const space_base_rep<T,M>& V2) {
    	return V1._constit.operator==(V2._constit); }

protected:
    template <class T1, class M1> friend class field_basic;
// internal:
    void freeze_guard() const {
    	if (_is_freezed) return;
    	_is_freezed = true;
    	freeze_body();
    }
    void no_freeze_guard() const {
    	check_macro (!_is_freezed, "freezed space cannot accept new (un)blocked domains");
    }
    void base_freeze_body() const;
    virtual void freeze_body() const { return base_freeze_body(); }
// data:
    space_constitution<T,M>          _constit;
    mutable bool                     _is_freezed;
    mutable array<space_pair_type,M> _idof2blk_iub;  // pair (is_blocked ; iu_or_ib); use ownership
    mutable distributor              _iu_ownership;  // unknown values distribution
    mutable distributor              _ib_ownership;  // blocked values distribution
};
// ---------------------------------------------------------------------
// momentum/interpolation implementation
// ---------------------------------------------------------------------
template <class T, class M>
template <class Function>
/* TODO: typename function_traits<Function>::result_type */
inline
T
space_base_rep<T,M>::momentum (
    Function  f,
    size_type idof) const
{
#ifdef TO_CLEAN
    // P1 Lagrange only ; TODO: extend
    const basic_point<T>& x = get_geo().vertex (idof);
    return f(x);
#endif // TO_CLEAN
    basic_point<T> x = get_element().x_dof (idof, get_geo());
    return f(x);
}
// ---------------------------------------------------------------------
template <class T, class M> class space_rep {};

template <class T>
class space_rep<T,sequential> : public space_base_rep<T,sequential> {
public:

// typedefs:

    typedef space_base_rep<T,sequential> base;
    typedef typename base::size_type size_type;

// allocators:

    space_rep (const geo_basic<T,sequential>& omega = (geo_basic<T,sequential>()), std::string approx = "");

// compatibility with the distributed interface:

    bool             dis_is_blocked (size_type dis_idof) const { return is_blocked(dis_idof); }
    size_type               dis_iub (size_type dis_idof) const { return iub(dis_idof); }

    const distributor& ios_ownership() const { return base::ownership(); }
    size_type     idof2ios_dis_idof (size_type     idof) const { return idof; }
    size_type     ios_idof2dis_idof (size_type ios_idof) const { return ios_idof; }
};
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class space_rep<T,distributed> : public space_base_rep<T,distributed> {
public:

// typedefs:

    typedef space_base_rep<T,distributed> base;
    typedef typename base::size_type size_type;

// allocators:

    space_rep (const geo_basic<T,distributed>& omega = (geo_basic<T,distributed>()), std::string approx = "");

// accessors:

    const communicator&      comm() const { return base::comm(); }

    bool             dis_is_blocked (size_type dis_idof) const;
    size_type               dis_iub (size_type dis_idof) const;

    const distributor& ios_ownership() const { return _ios_idof2dis_idof.ownership(); }
    size_type     idof2ios_dis_idof (size_type     idof) const { base::freeze_guard(); return _idof2ios_dis_idof [idof]; }
    size_type     ios_idof2dis_idof (size_type ios_idof) const { base::freeze_guard(); return _ios_idof2dis_idof [ios_idof]; }
#ifdef TO_CLEAN
    void get_ios_dis_idof (const geo_element& K, std::vector<size_type>& ios_dis_idof) const;
#endif // TO_CLEAN

protected:
    template <class T1, class M1> friend class field_basic;
    typedef std::map <size_type, size_type, std::less<size_type>,
            heap_allocator<std::pair<size_type,size_type> > >           map_type;
    typedef std::map <size_type, space_pair_type, std::less<size_type>,
            heap_allocator<std::pair<size_type,space_pair_type> > >     map_pair_type;
// internal procedures:
    void freeze_body() const;
    void append_external_dof (const geo_basic<T,distributed>& dom, std::set<size_type>& ext_dof_set) const;
// data:
    array<size_type>               _idof2ios_dis_idof;   // permut to/from ios dof numbering (before geo part), for i/o
    array<size_type>               _ios_idof2dis_idof;
// mutable data, affected by freeze_*()const:
    mutable map_pair_type          _ext_dis_idof2blk_dis_iub; // same for partition bdry dofs
};
#endif // _RHEOLEF_HAVE_MPI
// ====================================================================
// 2) wrapper class: seq & mpi specializations
// ====================================================================
/*Class:space
NAME: @code{space} -- piecewise polynomial finite element space
@clindex space
DESCRIPTION:
@noindent
The @code{space} class contains some numbering 
for unknowns and blocked degrees of freedoms
related to a given mesh and polynomial approximation.
SEE ALSO: "geo"(3)
AUTHORS: Pierre.Saramito@imag.fr
DATE:   14 december 2010
End:
*/
/// @brief the finite element space
template <class T, class M = rheo_default_memory_model>
class space_basic {
public:
};
typedef space_basic<Float> space;
// ---------------------------------------------------------------------
//<verbatim:
template <class T>
class space_basic<T,sequential> : public smart_pointer<space_rep<T,sequential> > {
public:

// typedefs:

    typedef space_rep<T,sequential> rep;
    typedef smart_pointer<rep>      base;
    typedef typename rep::size_type size_type;

// allocators:

    space_basic (const geo_basic<T,sequential>& omega = (geo_basic<T,sequential>()), std::string approx = "");

// accessors:

    void block  (std::string dom_name);
    void unblock(std::string dom_name);
    void block  (const domain_indirect_basic<sequential>& dom);
    void unblock(const domain_indirect_basic<sequential>& dom);

    const distributor&  ownership() const;
    const communicator& comm() const;
    size_type           size() const;
    size_type           dis_size() const;

    const geo_basic<T,sequential>& get_geo() const;
    const element<T>&   get_element() const;
    std::string get_stamp() const;

    void get_dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const;

    const distributor& iu_ownership() const;
    const distributor& ib_ownership() const;

    bool      is_blocked (size_type     idof) const;
    size_type        iub (size_type     idof) const;
    bool  dis_is_blocked (size_type dis_idof) const;
    size_type    dis_iub (size_type dis_idof) const;

    const distributor& ios_ownership() const;
    size_type idof2ios_dis_idof (size_type idof) const;
    size_type ios_idof2dis_idof (size_type ios_idof) const;

    template <class Function>
    T momentum (Function f, size_type idof) const;

    array<size_type, sequential>
    build_indirect_array (const geo_basic<T,sequential>& dom) const;

// comparator:

    bool operator== (const space_basic<T,sequential>& V2) const { return base::data().operator==(V2.data()); }
    bool operator!= (const space_basic<T,sequential>& V2) const { return ! operator== (V2); }
    friend bool are_compatible (const space_basic<T,sequential>& V1, const space_basic<T,sequential>& V2) {
	return are_compatible (V1.data(), V2.data()); }
};
//>verbatim:
template<class T>
inline
space_basic<T,sequential>::space_basic (const geo_basic<T,sequential>& omega, std::string approx)
 : base (new_macro(rep(omega, approx)))
{
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::ownership() const
{
    return base::data().ownership();
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::ios_ownership() const
{
    return base::data().ios_ownership();
}
template<class T>
inline
const communicator&
space_basic<T,sequential>::comm() const
{
    return base::data().comm();
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::size() const
{
    return base::data().size();
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::dis_size() const
{
    return base::data().dis_size();
}
template<class T>
inline
const geo_basic<T,sequential>&
space_basic<T,sequential>::get_geo() const
{
    return base::data().get_geo();
}
template<class T>
inline
const element<T>&
space_basic<T,sequential>::get_element() const
{
    return base::data().get_element();
}
template<class T>
inline
std::string
space_basic<T,sequential>::get_stamp() const
{
    return base::data().get_stamp();
}
template<class T>
inline
void
space_basic<T,sequential>::get_dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const
{
    return base::data().get_dis_idof (K, dis_idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::idof2ios_dis_idof (size_type idof) const
{
    return base::data().idof2ios_dis_idof (idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::ios_idof2dis_idof (size_type ios_idof) const
{
    return base::data().ios_idof2dis_idof (ios_idof);
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::iu_ownership() const
{
    return base::data().iu_ownership();
}
template<class T>
inline
const distributor&
space_basic<T,sequential>::ib_ownership() const
{
    return base::data().ib_ownership();
}
template<class T>
inline
bool
space_basic<T,sequential>::is_blocked (size_type idof) const
{
    return base::data().is_blocked (idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::iub (size_type idof) const
{
    return base::data().iub (idof);
}
template<class T>
inline
bool
space_basic<T,sequential>::dis_is_blocked (size_type dis_idof) const
{
    return base::data().dis_is_blocked (dis_idof);
}
template<class T>
inline
typename space_basic<T,sequential>::size_type
space_basic<T,sequential>::dis_iub (size_type dis_idof) const
{
    return base::data().dis_iub (dis_idof);
}
template<class T>
inline
void
space_basic<T,sequential>::block (std::string dom_name)
{
    return base::data().block (get_geo().get_domain_indirect(dom_name));
}
template<class T>
inline
void
space_basic<T,sequential>::unblock (std::string dom_name)
{
    return base::data().unblock (get_geo().get_domain_indirect(dom_name));
}
template<class T>
inline
void
space_basic<T,sequential>::block (const domain_indirect_basic<sequential>& dom)
{
    return base::data().block (dom);
}
template<class T>
inline
void
space_basic<T,sequential>::unblock (const domain_indirect_basic<sequential>& dom)
{
    return base::data().unblock (dom);
}
template<class T>
template <class Function>
inline
T
space_basic<T,sequential>::momentum (Function f, size_type idof) const
{
    return base::data().momentum (f, idof);
}
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
//<verbatim:
template <class T>
class space_basic<T,distributed> : public smart_pointer<space_rep<T,distributed> > {
public:

// typedefs:

    typedef space_rep<T,distributed> rep;
    typedef smart_pointer<rep>       base;
    typedef typename rep::size_type  size_type;

// allocators:

    space_basic (const geo_basic<T,distributed>& omega = (geo_basic<T,distributed>()), std::string approx = "");

// accessors:

    void block  (std::string dom_name);
    void unblock(std::string dom_name);
    void block  (const domain_indirect_basic<distributed>& dom);
    void unblock(const domain_indirect_basic<distributed>& dom);

    const distributor&  ownership() const;
    const communicator& comm() const;
    size_type           size() const;
    size_type           dis_size() const;

    const geo_basic<T,distributed>& get_geo() const;
    const element<T>&   get_element() const;
    std::string get_stamp() const;

    void get_dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const;

    const distributor& iu_ownership() const;
    const distributor& ib_ownership() const;

    bool      is_blocked (size_type     idof) const;
    size_type        iub (size_type     idof) const;

    bool  dis_is_blocked (size_type dis_idof) const;
    size_type    dis_iub (size_type dis_idof) const;

    const distributor& ios_ownership() const;
    size_type idof2ios_dis_idof (size_type idof) const;
    size_type ios_idof2dis_idof (size_type ios_idof) const;

    template <class Function>
    T momentum (Function f, size_type idof) const;

    array<size_type, distributed>
    build_indirect_array (const geo_basic<T,distributed>& dom) const;

// comparator:

    bool operator== (const space_basic<T,distributed>& V2) const { return base::data().operator==(V2.data()); }
    bool operator!= (const space_basic<T,distributed>& V2) const { return ! operator== (V2); }
    friend bool are_compatible (const space_basic<T,distributed>& V1, const space_basic<T,distributed>& V2) {
	return are_compatible (V1.data(), V2.data()); }
};
//>verbatim:

template<class T>
inline
space_basic<T,distributed>::space_basic (const geo_basic<T,distributed>& omega, std::string approx)
 : base (new_macro(rep(omega, approx)))
{
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::ownership() const
{
    return base::data().ownership();
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::ios_ownership() const
{
    return base::data().ios_ownership();
}
template<class T>
inline
const communicator&
space_basic<T,distributed>::comm() const
{
    return base::data().comm();
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::size() const
{
    return base::data().size();
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::dis_size() const
{
    return base::data().dis_size();
}
template<class T>
inline
const geo_basic<T,distributed>&
space_basic<T,distributed>::get_geo() const
{
    return base::data().get_geo();
}
template<class T>
inline
const element<T>&
space_basic<T,distributed>::get_element() const
{
    return base::data().get_element();
}
template<class T>
inline
std::string
space_basic<T,distributed>::get_stamp() const
{
    return base::data().get_stamp();
}
template<class T>
inline
void
space_basic<T,distributed>::get_dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const
{
    return base::data().get_dis_idof (K, dis_idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::idof2ios_dis_idof (size_type idof) const
{
    return base::data().idof2ios_dis_idof (idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::ios_idof2dis_idof (size_type ios_idof) const
{
    return base::data().ios_idof2dis_idof (ios_idof);
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::iu_ownership() const
{
    return base::data().iu_ownership();
}
template<class T>
inline
const distributor&
space_basic<T,distributed>::ib_ownership() const
{
    return base::data().ib_ownership();
}
template<class T>
inline
bool
space_basic<T,distributed>::is_blocked (size_type idof) const
{
    return base::data().is_blocked (idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::iub (size_type idof) const
{
    return base::data().iub (idof);
}
template<class T>
inline
bool
space_basic<T,distributed>::dis_is_blocked (size_type dis_idof) const
{
    return base::data().dis_is_blocked (dis_idof);
}
template<class T>
inline
typename space_basic<T,distributed>::size_type
space_basic<T,distributed>::dis_iub (size_type dis_idof) const
{
    return base::data().dis_iub (dis_idof);
}
template<class T>
inline
void
space_basic<T,distributed>::block (std::string dom_name)
{
    return base::data().block (get_geo().get_domain_indirect(dom_name));
}
template<class T>
inline
void
space_basic<T,distributed>::unblock (std::string dom_name)
{
    return base::data().unblock (get_geo().get_domain_indirect(dom_name));
}
template<class T>
inline
void
space_basic<T,distributed>::block (const domain_indirect_basic<distributed>& dom)
{
    base::data().block (dom);
}
template<class T>
inline
void
space_basic<T,distributed>::unblock (const domain_indirect_basic<distributed>& dom)
{
    base::data().unblock (dom);
}
template<class T>
template <class Function>
inline
T
space_basic<T,distributed>::momentum (Function f, size_type idof) const
{
    return base::data().momentum (f, idof);
}
#endif // _RHEOLEF_HAVE_MPI

// only valid when M=sequential or M=distributed => use a macro
#define _RHEOLEF_space_build_indirect_array(M)						\
template<class T>									\
inline											\
array<typename space_basic<T,M>::size_type, M>						\
space_basic<T,M>::build_indirect_array (const geo_basic<T,M>& dom) const		\
{											\
    if (dom.data().variant() == geo_abstract_base_rep<T>::geo_domain_indirect) {	\
      const geo_domain_indirect_rep<T,M>& dom_rep					\
       = dynamic_cast<const geo_domain_indirect_rep<T,M>&>(dom.data());			\
      return base::data().build_indirect_array (dom_rep.get_indirect());		\
    }											\
    error_macro ("build_indirect_array: unsupported non domain_indirect geo variant");	\
    return array<size_type, M>(); /* not  reached */					\
}
_RHEOLEF_space_build_indirect_array(sequential)

#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_space_build_indirect_array(distributed)
#endif // _RHEOLEF_HAVE_MPI

#undef _RHEOLEF_space_build_indirect_array

} // namespace rheolef
#endif // _RHEOLEF_SPACE_H
