# ifndef _RHEOLEF_FIELD_H
# define _RHEOLEF_FIELD_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/skit.h"
#include "rheolef/space.h"
#include "rheolef/scalar_traits.h"
#include "rheolef/misc_algo.h" // copy_n

namespace rheolef {

// forward declaration:
template <class Expr> struct field_expr;
template <class T, class M> class field_indirect;
template <class T, class M> class field_indirect_const;

//<field:  
/*Class:field
NAME:  @code{field} - piecewise polynomial finite element field
DESCRIPTION:       
  @noindent
  Store degrees of freedom associated to a mesh and
  a piecewise polynomial approximation, with respect
  to the numbering defined by the underlying @ref{space class}.

  @noindent
  This class contains two vectors, namely unknown and blocked
  degrees of freedoms, and the associated finite element space.
  Blocked and unknown degrees of freedom can be set by using
  domain name indexation:
  @example
        geo omega_h ("circle");
        space Vh (omega_h, "P1");
        Vh.block ("boundary");
        field uh (Vh);
        uh ["boundary"] = 0;
  @end example
FEATURES:
   Linear algebra, such as uh+vh, uh-vh, lambda*uh + mu*vh, ...are supported.
   The Euclidian dot product writes simply dot(uh,vh).
   The application of a bilinear form (@pxref{form class})
   writes m(uh,vh) and is equivalent to dot(m*uh,vh). 

   Non-linear operations, such as sqrt(uh) applies to the 
   set of degrees-of-freedom: sqrt(uh) returns the Lagrange
   interpolant of the square root of uh into the same space as uh.
   All standard unary and binary math functions abs, cos, sin... are available
   on fields. Also sqr(uh), the square of a field, and min(uh,vh), mx(uh,vh)
   are provided. Binary functions can be used also with a scalar, as in
   @example
      field wh = max (abs(uh), 0);
      field zh = pow (abs(uh), 1./3);
   @end example
   For applying a user-provided function to a field, use:
@example
   field vh = compose(f, uh);
   field wh = compose(f, uh, vh);
@end example
   The composition supports also general unary and binary class-functions.

TODO:
  Interpolation of a function @code{u} in a field @code{uh} with respect to 
  the interpolation writes:
  @example
        Float u (const point&);
        uh = interpolate (Vh, u);
  @end example

EXAMPLE:
  Here is a complete example using the field class:
  @example
        Float u (const point& x) @{ return x[0]*x[1]; @}
        int main() @{
          geo omega_h ("square");
          space Vh (omega_h, "P1");
          field uh (Vh);
          uh = interpolate (Vh, u);
          cout << plotmtv << u;
        @}
  @end example
TODO:
   Vector-valued and tensor-valued field support is yet partial only.
   This feature will be more documented in the future.

IMPLEMENTATION NOTE:
  The field expression use the expression template technics in
  order to avoid temporaries when evaluating complex expressions.
AUTHOR: Pierre.Saramito@imag.fr
DATE:   23 february 2011
End:
*/
//<field:
template <class T, class M = rheo_default_memory_model>
class field_basic : public std::unary_function<basic_point<typename scalar_traits<T>::type>,T> {
public :
// typedefs:

    typedef typename std::size_t            size_type;
    typedef T                               value_type;
    typedef typename scalar_traits<T>::type float_type;
    typedef geo_basic  <float_type,M>       geo_type;
    typedef space_basic<float_type,M>       space_type;
    class iterator;
    class const_iterator;

// allocator/deallocator:
  
    field_basic();

    explicit field_basic (
	const space_type& V, 
	const T& init_value = std::numeric_limits<T>::max());
   
    void resize (
	const space_type& V, 
	const T& init_value = std::numeric_limits<T>::max());

    field_basic                   (const field_indirect<T,M>&);
    field_basic<T, M>& operator=  (const field_indirect<T,M>&);
    field_basic                   (const field_indirect_const<T,M>&);
    field_basic<T, M>& operator=  (const field_indirect_const<T,M>&);
    template <class Expr> field_basic                   (const field_expr<Expr>&);
    template <class Expr> field_basic<T, M>& operator=  (const field_expr<Expr>&);
    field_basic<T, M>& operator=  (const T&);

// accessors:

    const space_type& get_space() const { return _V; }
    const distributor& ownership() const { return get_space().ownership(); }
    const communicator& comm() const { return ownership().comm(); }
    size_type dis_size() const { return ownership().dis_size(); }
    size_type size() const { return ownership().size(); }
    std::string get_stamp() const { return _V.get_stamp(); }

          T& dof (size_type idof);
    const T& dof (size_type idof) const;

    field_indirect<T,M>       operator[] (const geo_basic<T,M>& dom);
    field_indirect<T,M>       operator[] (std::string dom_name);
    field_indirect_const<T,M> operator[] (const geo_basic<T,M>& dom) const;
    field_indirect_const<T,M> operator[] (std::string dom_name) const;

    iterator begin();
    iterator end();
    const_iterator begin() const;
    const_iterator end() const;

// input/output:

    idiststream& get (idiststream& ips);
    odiststream& put (odiststream& ops) const;

// data:
protected:
    space_type  _V;
public:
    vec<T,M>    u;
    vec<T,M>    b;
};
template <class T, class M>
idiststream& operator >> (odiststream& ips, field_basic<T,M>& u);

template <class T, class M>
odiststream& operator << (odiststream& ops, const field_basic<T,M>& u);

typedef field_basic<Float> field;
//>field:

// =================================================================================
// field::iterator
// =================================================================================
template <class T, class M>
class field_basic<T,M>::iterator {
protected:
  typedef typename vec<T,M>::iterator                       data_t; // random acess
  typedef typename array<space_pair_type,M>::const_iterator iter_t; // forward access
public:

// typedefs:

  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef T&                                reference;
  typedef T*                                pointer;
  typedef std::ptrdiff_t                    difference_type;

// allocators:

  iterator (iter_t blk_iub_iter, data_t u, data_t b)
    : _blk_iub_iter(blk_iub_iter), _u(u), _b(b)
    {}

// accessors & modifiers:

  reference operator* () const { 
      size_type iub = (*_blk_iub_iter).iub();
      size_type blk = (*_blk_iub_iter).is_blocked();
      if (!blk) return _u[iub]; else return _b[iub];
  }
  iterator& operator++ () { _blk_iub_iter++; return *this; }
  iterator  operator++ (int) { iterator tmp = *this; operator++(); return *this; }
  iterator& operator+= (difference_type n) { _blk_iub_iter += n; return *this; }
  iterator  operator+  (difference_type n) const { iterator tmp = *this; return tmp += n; }
  reference operator[] (size_type n) const { return *(*this + n); }

// comparators:

  bool operator== (const iterator& j) const { return _blk_iub_iter == j._blk_iub_iter; }
  bool operator!= (const iterator& j) const { return ! operator== (j); }
protected:
  template <class T1, class M1> friend class field_basic<T1,M1>::const_iterator;
// data:
  iter_t _blk_iub_iter;
  data_t _u;
  data_t _b;
};
template <class T, class M>
inline
typename field_basic<T,M>::iterator
field_basic<T,M>::begin ()
{
    return iterator (_V.data()._idof2blk_iub.begin(), u.begin(), b.begin());
}
template <class T, class M>
inline
typename field_basic<T,M>::iterator
field_basic<T,M>::end ()
{
    return iterator (_V.data()._idof2blk_iub.end(), u.begin(), b.begin());
}
// =================================================================================
// field::const_iterator
// =================================================================================
template <class T, class M>
class field_basic<T,M>::const_iterator {
protected:
  typedef typename vec<T,M>::const_iterator                 data_t;
  typedef typename array<space_pair_type,M>::const_iterator iter_t;
public:

// typedefs:

  typedef std::forward_iterator_tag         iterator_category;
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef const T&                          reference;
  typedef const T*                          pointer;
  typedef std::ptrdiff_t                    difference_type;

// allocators:

  const_iterator (iter_t blk_iub_iter, data_t u, data_t b)
    : _blk_iub_iter(blk_iub_iter), _u(u), _b(b)
    {}
  const_iterator (iterator i)
    : _blk_iub_iter(i._blk_iub_iter), _u(i._u), _b(i._b)
    {}

// accessors & modifiers:

  reference operator* () const { 
      size_type iub = (*_blk_iub_iter).iub();
      size_type blk = (*_blk_iub_iter).is_blocked();
      if (!blk) return _u[iub]; else return _b[iub];
  }
  const_iterator& operator++ () { _blk_iub_iter++; return *this; }
  const_iterator  operator++ (int) { const_iterator tmp = *this; operator++(); return *this; }
  const_iterator& operator+= (difference_type n) { _blk_iub_iter += n; return *this; }
  const_iterator  operator+  (difference_type n) const { const_iterator tmp = *this; return tmp += n; }
  reference operator[] (size_type n) const { return *(*this + n); }

// comparators:

  bool operator== (const const_iterator& j) const { return _blk_iub_iter == j._blk_iub_iter; }
  bool operator!= (const const_iterator& j) const { return ! operator== (j); }
protected:
// data:
  iter_t _blk_iub_iter;
  data_t _u;
  data_t _b;
};
template <class T, class M>
inline
typename field_basic<T,M>::const_iterator
field_basic<T,M>::begin () const
{
    return const_iterator (_V.data()._idof2blk_iub.begin(), u.begin(), b.begin());
}
template <class T, class M>
inline
typename field_basic<T,M>::const_iterator
field_basic<T,M>::end () const
{
    return const_iterator (_V.data()._idof2blk_iub.end(), u.begin(), b.begin());
}
// =================================================================================
// field_indirect 
// as for:
//   uh["left"] = value
//   uh["left"] = gh
// =================================================================================
template <class T, class M = rheo_default_memory_model>
class field_indirect {
public:
//protected:

// typedefs

  typedef typename field_basic<T,M>::size_type size_type;
  typedef T                                    value_type;
  class iterator;
  class const_iterator;

// allocators:

  field_indirect (field_basic<T,M>& uh, const geo_basic<T,M>& dom)
    : _indirect(uh.get_space().build_indirect_array(dom)), _val(uh.begin()),
      _V(uh.get_space()), _dom(dom)
    {
    }
// accessors & modifiers:
public:

  field_indirect<T,M>& operator=  (const T& alpha);
  field_indirect<T,M>& operator=  (const field_basic<T,M>& gh);

//protected:
  const distributor& ownership() const { return _indirect.ownership(); }
  const communicator& comm() const { return ownership().comm(); }
  size_type     size() const { return ownership().size(); }
  size_type dis_size() const { return ownership().dis_size(); }
  std::string get_stamp() const;
  space_basic<T,M> get_space() const {
        space_basic<T,M> Wh (_dom, _V.get_element().name());
	warning_macro ("field_indirect::get_space: build a new space " << Wh.get_stamp());
        return Wh;
  }
        T& dof (size_type idof)       { return _val [_indirect [idof]]; }
  const T& dof (size_type idof) const { return _val [_indirect [idof]]; }

  iterator begin();
  iterator end();
  const_iterator begin() const;
  const_iterator end() const;
protected:
  friend class field_basic<T,M>;
  friend class field_indirect_const<T,M>;
  array<size_type,M>                    _indirect; // temporary
  typename field_basic<T,M>::iterator   _val;      // iterator = reference on an external variable uh
						   // => cstor is protected : prevent copy outside the field class
  space_basic<T,M>                      _V;
  geo_basic<T,M>                        _dom;
};
template<class T, class M>
inline
field_indirect<T,M>&
field_indirect<T,M>::operator=  (const T& alpha)
{
  std::fill (begin(), end(), alpha);
  return *this;
}
template<class T, class M>
inline
field_indirect<T,M>&
field_indirect<T,M>::operator=  (const field_basic<T,M>& expr)
{
  check_macro (get_stamp() == expr.get_stamp(), "field[domain] = field : incompatible spaces "
		<< get_stamp() << " and " << expr.size());
  algo::copy_n (expr.begin(), expr.size(), begin());
  return *this;
}
template<class T, class M>
inline
std::string
field_indirect<T,M>::get_stamp() const
{
  // e.g. "P1(square.left)", an unique signature for field_expr<Expr> size-like checks
  return _V.get_element().name() + "(" + _V.get_geo().name() + "." + _dom.name() + ")";
}
template<class T, class M>
inline
field_indirect<T,M>
field_basic<T,M>::operator[] (const geo_basic<T,M>& dom)
{
  return field_indirect<T,M> (*this, dom);
}
template<class T, class M>
inline
field_indirect<T,M>
field_basic<T,M>::operator[] (std::string dom_name)
{
  return operator[] (get_space().get_geo().operator[] (dom_name));
}
template<class T, class M>
inline
field_basic<T,M>&
field_basic<T,M>::operator=  (const field_indirect<T,M>& expr)
{
  if (get_stamp() == "") {
    resize (expr.get_space());
  } else {
    check_macro (get_stamp() == expr.get_stamp(), "incompatible spaces "
	<< get_stamp() << " and " << expr.get_stamp()
	<< " in field = field[domain]");
  }
  algo::copy_n (expr.begin(), expr.size(), begin());
  return *this;
}
template<class T, class M>
inline
field_basic<T,M>::field_basic (const field_indirect<T,M>& expr)
 : _V(),
   u(),
   b()
{
    operator= (expr);
}
// =================================================================================
// field_indirect::iterator
// =================================================================================
template <class T, class M>
class field_indirect<T,M>::iterator {
public:
// typedef:
  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef T&                                reference;
  typedef T*                                pointer;
  typedef std::ptrdiff_t                    difference_type;
// allocator:
  iterator (typename array<size_type,M>::const_iterator idof_iter, typename field_basic<T,M>::iterator val)
    : _idof_iter(idof_iter), _val(val) {}
 
// accessors & modifiers:
        T& operator* ()       { return _val [*_idof_iter]; }
  const T& operator* () const { return _val [*_idof_iter]; }
  iterator& operator++()      { ++_idof_iter; return *this; }

// comparators:

  bool operator== (const iterator& j) const { return _idof_iter == j._idof_iter; }
  bool operator!= (const iterator& j) const { return ! operator== (j); }
protected:
// data:
  typename array<size_type,M>::const_iterator _idof_iter;
  typename field_basic<T,M>::iterator         _val;
};
template<class T, class M>
inline
typename field_indirect<T,M>::iterator
field_indirect<T,M>::begin()
{
  return iterator (_indirect.begin(), _val);
}
template<class T, class M>
inline
typename field_indirect<T,M>::iterator
field_indirect<T,M>::end()
{
  return iterator (_indirect.end(), _val);
}
// =================================================================================
// field_indirect::const_iterator
// =================================================================================
template <class T, class M>
class field_indirect<T,M>::const_iterator {
public:
// typedef:
  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef const T&                          reference;
  typedef const T*                          pointer;
  typedef std::ptrdiff_t                    difference_type;
// allocator:
  const_iterator (typename array<size_type,M>::const_iterator idof_iter, typename field_basic<T,M>::const_iterator val)
    : _idof_iter(idof_iter), _val(val) {}
 
// accessors & modifiers:
  const T& operator* () const   { return _val [*_idof_iter]; }
  const_iterator& operator++()  { ++_idof_iter; return *this; }

// comparators:

  bool operator== (const const_iterator& j) const { return _idof_iter == j._idof_iter; }
  bool operator!= (const const_iterator& j) const { return ! operator== (j); }
protected:
// data:
  typename array<size_type,M>::const_iterator _idof_iter;
  typename field_basic<T,M>::const_iterator   _val;
};
template<class T, class M>
inline
typename field_indirect<T,M>::const_iterator
field_indirect<T,M>::begin() const
{
  return const_iterator (_indirect.begin(), _val);
}
template<class T, class M>
inline
typename field_indirect<T,M>::const_iterator
field_indirect<T,M>::end() const
{
  return const_iterator (_indirect.end(), _val);
}
// =================================================================================
// field_indirect_const 
// as for:
//   gh = uh["left"]
// =================================================================================
template <class T, class M = rheo_default_memory_model>
class field_indirect_const {
public:
//protected:

// typedefs:

  typedef typename field_basic<T,M>::size_type size_type;
  typedef T                                    value_type;
  class const_iterator;

// allocators:

  field_indirect_const (const field_basic<T,M>& uh, const geo_basic<T,M>& dom)
    : _indirect(uh.get_space().build_indirect_array(dom)), _val(uh.begin()),
      _V(uh.get_space()), _dom(dom)
    {
    }

  field_indirect_const (field_indirect<T,M> gh)
    : _indirect(gh._indirect), _val(gh._val),
      _V(gh._V), _dom(gh._dom)
    {}

// accessors:
//protected:

  const distributor& ownership() const { return _indirect.ownership(); }
  const communicator& comm() const { return ownership().comm(); }
  size_type     size() const { return ownership().size(); }
  size_type dis_size() const { return ownership().dis_size(); }
  std::string get_stamp() const;
  space_basic<T,M> get_space() const {
        space_basic<T,M> Wh (_dom, _V.get_element().name());
	warning_macro ("field_indirect_const::get_space: build a new space " << Wh.get_stamp());
        return Wh;
  }

  const T& dof(size_type idof) const { return _val [_indirect [idof]]; }

  const_iterator begin() const;
  const_iterator end() const;
  friend class field_basic<T,M>;
protected:
  array<size_type,M>                          _indirect; // temporary
  typename field_basic<T,M>::const_iterator   _val;      // iterator = reference on an external variable uh
				   		         // thus cstor is protected to prevent copy outside the field class
  space_basic<T,M>                            _V;
  geo_basic<T,M>                              _dom;
};
template<class T, class M>
inline
std::string
field_indirect_const<T,M>::get_stamp() const
{
  // e.g. "P1(square.left)", an unique signature for field_expr<Expr> size-like checks
  return _V.get_element().name() + "(" + _V.get_geo().name() + "." + _dom.name() + ")";
}
template<class T, class M>
inline
field_indirect_const<T,M>
field_basic<T,M>::operator[] (const geo_basic<T,M>& dom) const
{
  return field_indirect_const<T,M> (*this, dom);
}
template<class T, class M>
inline
field_indirect_const<T,M>
field_basic<T,M>::operator[] (std::string dom_name) const
{
  return operator[] (get_space().get_geo().operator[] (dom_name));
}
template<class T, class M>
inline
field_basic<T,M>&
field_basic<T,M>::operator=  (const field_indirect_const<T,M>& expr)
{
  if (get_stamp() == "") {
    resize (expr.get_space());
  } else {
    check_macro (get_stamp() == expr.get_stamp(), "incompatible spaces "
	<< get_stamp() << " and " << expr.get_stamp()
	<< " in field = field[domain]");
  }
  algo::copy_n (expr.begin(), expr.size(), begin());
  return *this;
}
template<class T, class M>
inline
field_basic<T,M>::field_basic (const field_indirect_const<T,M>& expr)
 : _V(),
   u(),
   b()
{
    operator= (expr);
}
// =================================================================================
// field_indirect_const::const_iterator
// =================================================================================
template <class T, class M>
class field_indirect_const<T,M>::const_iterator {
public:
// typedefs:

  typedef std::forward_iterator_tag         iterator_category; // TODO: not fully random yet
  typedef typename vec<T,M>::size_type      size_type;
  typedef T                                 value_type;
  typedef const T&                          reference;
  typedef const T*                          pointer;
  typedef std::ptrdiff_t                    difference_type;

// allocator:
  const_iterator (typename array<size_type,M>::const_iterator idof_iter, typename field_basic<T,M>::const_iterator val)
    : _idof_iter(idof_iter), _val(val) {}
 
// accessors & modifiers:
  const T& operator* () const { return _val [*_idof_iter]; }
  const_iterator& operator++()      { ++_idof_iter; return *this; }

// comparators:

  bool operator== (const const_iterator& j) const { return _idof_iter == j._idof_iter; }
  bool operator!= (const const_iterator& j) const { return ! operator== (j); }
protected:
// data:
  typename array<size_type,M>::const_iterator _idof_iter;
  typename field_basic<T,M>::const_iterator   _val;
};
template<class T, class M>
inline
typename field_indirect_const<T,M>::const_iterator
field_indirect_const<T,M>::begin() const
{
  return const_iterator (_indirect.begin(), _val);
}
template<class T, class M>
inline
typename field_indirect_const<T,M>::const_iterator
field_indirect_const<T,M>::end() const
{
  return const_iterator (_indirect.end(), _val);
}
// =================================================================================
// field: inlined
// =================================================================================
template <class T, class M>
inline
field_basic<T,M>::field_basic ()
 : _V(),
   u(),
   b()
{
}
template<class T, class M>
inline
field_basic<T,M>&
field_basic<T,M>::operator=  (const T& value)
{
    check_macro (get_stamp() != "", "field=constant : uninitialized field in affectation");
    std::fill (begin(), end(), value);
    return *this;
}
template <class T, class M>
inline
T& 
field_basic<T,M>::dof (size_type idof)
{
    if (! _V.is_blocked(idof)) {
      return u [_V.iub(idof)]; 
    } else {
      return b [_V.iub(idof)];
    }
}
template <class T, class M>
inline
const T&
field_basic<T,M>::dof (size_type idof) const
{
    if (! _V.is_blocked(idof)) {
      return u [_V.iub(idof)];
    } else {
      return b [_V.iub(idof)];
    }
}
template <class T, class M>
inline
idiststream& operator >> (idiststream& ips, field_basic<T,M>& uh)
{
    return uh.get (ips);
}
template <class T, class M>
inline
odiststream& operator << (odiststream& ops, const field_basic<T,M>& uh)
{
    return uh.put (ops);
}
template <class T, class M>
inline
odiststream& operator << (odiststream& ops, const field_indirect<T,M>& uh)
{
    field_basic<T,M> tmp = uh;
    return tmp.put (ops);
}
template <class T, class M>
inline
odiststream& operator << (odiststream& ops, const field_indirect_const<T,M>& uh)
{
    field_basic<T,M> tmp = uh;
    return tmp.put (ops);
}

}// namespace rheolef
# endif /* _RHEOLEF_FIELD_H */
