#ifndef _RHEOLEF_ELEMENT_H
#define _RHEOLEF_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/compiler.h"
#include "rheolef/basis.h"
#include "rheolef/basis_on_lattice.h"
#include "rheolef/numbering.h"
#include "rheolef/geo.h"

namespace rheolef {

/*Class:element
NAME:  @code{element} - the finite element method
@cindex  finite element method
@clindex element
@clindex space
DESCRIPTION:
  Defines the finite element method in use: it is a combination of three
  choosen ingredients: a geometrical element, a polynomial @code{basis}
  (@pxref{basis internal})
  and a set of momentums.
  The finite element method is defined on various @code{reference_element}s 
  (@pxref{reference_element internal})
  and then
  transformed on any concrete @code{geo_element} of a @code{geo} by using
  the Piola transformation
  (@pxref{geo_element internal}, @ref{geo class}).
  The @code{element} class defines also how the basis and momentums are also transformed
  by the Piola transformation.
TO DO:
  The @code{element} class is a virtual class, basis of a hierarchy
  of classes such as @code{lagrange}, @code{hermitte}, etc.
  It is used by the @code{space} class for assembling @code{field}s and @code{form}s
  (@pxref{space class}, @ref{field class}, @ref{form class}).
DATE:   20 dec 2010
End:
*/
template<class T>
class element_rep {
public:
    typedef typename basis::size_type size_type;

    element_rep (std::string name = "", std::string numb = "");
    virtual ~element_rep () {}

    std::string name() const { return _name; }
    const basis& get_basis() const { return _b; }
    const numbering& get_numbering() const { return _numb; }
    const basis&  get_p1_transformation  () const { return _tr_p1; }  
    template <class M>
    basic_point<T> x_dof (size_type idof, const geo_basic<T,M>& omega) const;
    template <class M>
    basic_point<T> x_dof (const geo_element& K, size_type loc_idof, const geo_basic<T,M>& omega) const;

// data:
protected:
    std::string          _name;
    basis                _b; 		// TODO: basis<T> instead of basis
    numbering            _numb;
    basis                _tr_p1;	// TODO: or Pk piola transformation
    basis_on_nodal_basis _tr_p1_value;
// internal:
public:
    static element_rep<T>* make_ptr (std::string name);
};
//<verbatim:
template<class T>
class element : public smart_pointer<element_rep<T> > {
public:
// typedefs:
    typedef element_rep<T>	    rep;
    typedef smart_pointer<rep>	    base;
    typedef typename rep::size_type size_type;

// allocators:

    element (std::string name = "");

// accessors:

    std::string name() const;
    const basis& get_basis() const;
    const numbering& get_numbering() const;
    const basis&  get_p1_transformation() const;

    template <class M>
    basic_point<T> x_dof (size_type idof, const geo_basic<T,M>& omega) const;
    template <class M>
    basic_point<T> x_dof (const geo_element& K, size_type loc_idof, const geo_basic<T,M>& omega) const;

// comparator:

    bool operator== (const element<T>& elt) const { return name() == elt.name(); }
};
//>verbatim:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template<class T>
inline
element<T>::element (std::string name)
 : base (rep::make_ptr(name))
{
}
template<class T>
inline
std::string
element<T>::name() const
{
    if (base::pointer() == 0) { return ""; }
    return base::data().name();
}
template<class T>
inline
const basis& 
element<T>::get_basis() const
{
    return base::data().get_basis();
}
template<class T>
inline
const numbering& 
element<T>::get_numbering() const
{
    return base::data().get_numbering();
}
template<class T>
inline
const basis& 
element<T>::get_p1_transformation() const
{
    return base::data().get_p1_transformation();
}
template<class T>
template<class M>
inline
basic_point<T>
element<T>::x_dof (size_type idof, const geo_basic<T,M>& omega) const
{
    return base::data().x_dof (idof, omega);
}
template<class T>
template<class M>
inline
basic_point<T>
element<T>::x_dof (const geo_element& K, size_type loc_idof, const geo_basic<T,M>& omega) const
{
    return base::data().x_dof (K, loc_idof, omega);
}
// -----------------------------------------------------------
// the lagrange element
// -----------------------------------------------------------
template<class T>
class lagrange : public element_rep<T> {
public:
    lagrange (std::string name, std::string numb = "")
     : element_rep<T>(name,numb)
    {
    }
    ~lagrange()
    {
    }
};
// -----------------------------------------------------------
// the P1 Lagrange momentum
// -----------------------------------------------------------
#ifdef TO_CLEAN
template <class T, class M, class Function>
/* typename Function::result_type */
T
momentum(
    const element<T>& method,
    Function f,
    typename geo_element::size_type loc_idof,
    const geo_element& K,
    const geo_basic<T,M>& omega)
{
    typedef typename geo_element::size_type size_type;
    if (method.name() == "P1") {
      // TODO: P1 Lagrange only ; more efficient than the general case...
      size_type dis_idof = K [loc_idof];
      const basic_point<T>& x = omega.dis_vertex (dis_idof);
    } else {
      basic_point<T> x = method.x_dof (K, loc_idof, omega);
      return f(x);
    }
}
#endif // TO_CLEAN

} // namespace rheolef
#endif // _RHEOLEF_ELEMENT_H
