# ifndef _RHEO_PROJECT_H
# define _RHEO_PROJECT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
namespace rheolef { 
/*!
// L2-Projection toolkit
//
// contents:
//	* Riesz representer of a function,
//	* Riesz representer of a composed function.
//	* Euler quadrature formulae
//
// authors:
//	Jocelyn.Etienne@imag.fr
//	Pierre.Saramito@imag.fr
//	Guillaume Allegre for indices trick.
//
// date: 2 may 2003	update: 2 may 2003
 */

//! Gives Euler quadrature formula over a triangle
/*! Returns a list of points and weights such that Int(f)=Sigma(w_i f(p_i))
 *! The points are listed according to their position in the triangle: 
 *! vertices, then edge points, then face points, (then volume points)
 */
inline
void
get_quadrature (std::string quadrature, size_t order, 
	geo_element::enum_type element_type,
	std::vector<point>& qpoint, std::vector<Float>& weight,
	tiny_vector<size_t> ppd)
{
 	if (element_type!=geo_element::t) 
		error_macro("Only quadrature on triangles supported yet.");
 	if (quadrature!="Euler") 
		error_macro(quadrature+" quadrature is not implemented.");

	ppd =tiny_vector<size_t> (5);
	// Total number of points:
	ppd[4] =(order+1)*(order+2)/2;
	// Number of summit points :
	ppd[0] =3;
	// Number of edge points :
	ppd[1] =3*(order-1);
	// Number of face points :
	ppd[2] =(order*order -3*order +2)/2;
	// Number of volume points
	ppd[3] =0;

	// total weight is
	size_t tot_w =ppd[0] +ppd[1]*3 +ppd[2]*6;

	// face points are weighted 6
	weight =std::vector<Float>(ppd[4], 6./tot_w);
	qpoint =std::vector<point>(ppd[4]);

	// summit points are weighted 1
	for (size_t i=0; i<ppd[0]; i++) 
	 {
	 	weight[i]=1./tot_w;
		qpoint[i]=point(i%2,i/2,0);
	 }
	
	// edge points are weighted 3
	for (size_t i=0; i<order-1; i++)
	 {
		weight[ppd[0]+i] =3./tot_w;
		weight[ppd[0]+order-1+i] =3./tot_w;
		weight[ppd[0]+2*(order-1)+i] =3./tot_w;
		qpoint[ppd[0]+i] =point((i+1.)/order, 0,0);
		qpoint[ppd[0]+order-1+i] =point((order-1.-i)/order, (i+1.)/order, 0);
		qpoint[ppd[0]+2*(order-1)+i] =point(0, (order-1.-i)/order, 0);
	 }

	// face points
	size_t t=ppd[0]+ppd[1];
	for (size_t i=1; i<=order-1; i++)
	    for (size_t j=1; j<=order-1-i; j++)
	 	qpoint[t++] =point(Float(i)/Float(order), Float(j)/Float(order), 0);
	
	// no volume points

	return;
}
inline
void
get_quadrature (std::string quadrature, size_t order, 
	geo_element::enum_type element_type,
	std::vector<point>& qpoint, std::vector<Float>& weight)
{
 	tiny_vector<size_t> ppd;
	get_quadrature(quadrature, order, element_type, qpoint, weight, ppd);
}
}// namespace rheolef
#endif // _RHEO_PROJECT_H
