///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo-interface : provides normal, tangent to subdomains
//
// author:
//      J.Etienne@damtp.cam.ac.uk
// contribution : splines by B. Balde
//
// date:
//	26/09/06
//
#include "georep.h"
#include "geo-connectivity.h"
#include "field.h"
#include <string>
using namespace std;
namespace rheolef { 

field
geo::normal (const class space& Nh, const domain& d, const std::string& region) const
 {
    interface bc(region);
    return data().get_normal(Nh, d, false, bc);
 }
class field
geo::normal (const space& Nh, const domain& d, const interface& bc) const
 {
    return data().get_normal(Nh, d, false, bc);
 }
class field
geo::tangent (const space& Nh, const domain& d, const string& region) const
 {
    interface bc(region);
    return tangent(Nh, d, bc);
 }
class field
geo::tangent (const space& Nh, const domain& d, const interface& bc) const
 {
    field n=data().get_normal(Nh, d, false, bc);
    field t(n.get_space());
    t[0]=n[1]; t[1]=-1.*n[0];
    return t;
 }
class field
geo::axisymmetric_curvature (const space& Nh, const domain& d) const
 {
    interface bc;
    return data().axi_curvature(Nh, d, bc);
 }
class field
geo::axisymmetric_curvature (const space& Nh, const domain& d,
	const interface& bc) const
 {
    return data().axi_curvature(Nh, d, bc);
 }
class field
geo::plane_curvature (const space& Nh, const domain& d,
        const interface& bc) const
 {
    return data().plane_curvature(Nh, d, bc);
 }
field
geo::tangent_spline (const space& Nh, const domain& d, 
	const interface& bc) const
 {
    return data().tangent_spline(Nh, d, bc);
 }
class field
geo::plane_curvature_spline (const space& Nh, const domain& d,
        const interface& bc) const
 {
    return data().plane_curvature_spline(Nh, d, bc);
 }
class field
geo::plane_curvature_quadratic (const space& Nh, const domain& d,
        const interface& bc) const
 {
    return data().plane_curvature_quadratic(Nh, d, bc);
 }

void
georep::set_interface_orientation(const domain& d, const interface& bc) const
{
    // set of the vertices of domain d
    map< size_type, size_type > domain_element;
    domain::const_iterator iE=d.begin();
    domain::const_iterator eE=d.end();
    for (size_type i=0 ; iE!=eE; iE++, i++)
      { domain_element.insert(pair<size_type,size_type>((*iE).index(),i)); }

    // find an element of the inside-region with a side on the interface
    Vector<geo_element>::const_iterator iK, eK;
    if (bc.region=="")
     { iK=begin(); eK=end(); }
    else 
     { iK=operator[](bc.region).begin(); eK=operator[](bc.region).end(); }
    map<size_type,size_type>::const_iterator side;
    map<size_type,size_type>::const_iterator not_found=domain_element.end();
    bool found=false;
    for ( ; !found && iK != eK ; iK++ )
      for (size_type i=0; i<(*iK).n_subgeo((*iK).dimension()-1); i++)
        {
	  side=domain_element.find((*iK).side(i));
	  if (side!=not_found)
	  {
	    // Get the outward normal to element iK, whose side is on the interface
	    point n = normal ((*iK), i);
	    // remember that this element has this outward normal
	    bc.set_known_normal((*side).second,n);
	    found=true; break;
	  }
	}
    check_macro(found, "Interface is not on the boundary of region " << bc.region); 
}

// computes a P0 normal (no inverse connectivity reconstructed)
class field
georep::get_normal_P0 (const space& Nh, const domain& d, 
	bool axisymmetric_curvature, const interface& bc) const
{
    point er,ez;
    if (axisymmetric_curvature) 
     {
    	check_macro(Nh.coordinate_system()=="rz"||Nh.coordinate_system()=="zr",
		"Non-axisymmetric geometry");
	if (Nh.coordinate_system()=="rz") { er=point(1,0); ez=point(0,1); }
	else { er=point(0,1); ez=point(1,0); }
     }
    // fill in with zeros in case only part of domain d is bounding region
    field nh(Nh, Float(0));
   
    // set of the vertices of domain d
    map< size_type, size_type > domain_element;
    domain::const_iterator iE=d.begin();
    domain::const_iterator eE=d.end();
    for (size_type i=0 ; iE!=eE; iE++, i++)
      { //cerr << (*iE).index() <<":"<<(*iE)[0]<<"-"<<(*iE)[1] << " ";
        domain_element.insert(pair<size_type,size_type>((*iE).index(),i)); }

    // go through all elements in region (or whole mesh) and build normal
    Vector<geo_element>::const_iterator iK, eK;
    if (bc.region=="")
     { iK=begin(); eK=end(); }
    else 
     { iK=operator[](bc.region).begin(); eK=operator[](bc.region).end(); }
    tiny_vector<size_type> dofs;
    map<size_type,size_type>::const_iterator side;
    map<size_type,size_type>::const_iterator not_found=domain_element.end();
    bool first_time=true;
    for ( ; iK != eK ; iK++ )
      for (size_type i=0; i<(*iK).n_subgeo((*iK).dimension()-1); i++)
        {
	  side=domain_element.find((*iK).side(i));
	  if (side!=not_found)
	  {
	    // Get the outward normal to element iK, whose side is on the interface
	    // WARNING: straight sided elements assumed, and P0 normal ...!
	    point n = normal ((*iK), i);
	    if (first_time) 
	    { // remember that this element has this outward normal for
	      // use in plane_curvature (gives order on interface)
	      //std::cerr << " $ elenorm: " << _x.begin()[(*iK)[i]] << " || " 
	      //	<< _x.begin()[(*iK)[(i+1)%3]] << " || " << _x.begin()[(*iK)[(i+2)%3]]
	      //	<< std::endl;
	      bc.set_known_normal((*side).second,n);
	      first_time=false;
	    }
	    if (axisymmetric_curvature) 
	    {
	      // n.er gives the tan of angle of edge to horizontal,
	      // then curvature is calculated as the inverse of the distance
	      // form middle of edge to a point on the axis s.t....
	      point Me=0.5*(_x.begin()[(*iK)[i]]+_x.begin()[(*iK)[(i+1)%3]]); 
	      n= -dot(n,er)/dot(Me,er) *n;
	    }
	    
	    for (size_type id=0; id<Nh.n_component(); id++) 
	    {
	      // get nh dofs corresponding
	      Nh.set_dof((Nh.get_geo().begin())[(*side).second],dofs,id);
	      // set dofs
	      for (size_type idof=0; idof<dofs.size(); idof++)
	        nh.at(dofs[idof])=n[id];
	    }
	  }
	}
    return nh;
}
// computes a P1 or P2 normal after reordering the elements of the interface (required to be open
// because of the reordering)
class field
georep::get_normal_P1P2 (const space& Nh, const domain& d, const interface& bc) const
{
  string approx = Nh.get_approx();
  // fill in with zeros in case only part of domain d is bounding region
  field nh(Nh, Float(0));
  const_iterator_node sommets = begin_node();
  (*this).sort_interface(d, bc);
  Vector<size_t>::const_iterator bI = bc.begin();
  Vector<size_t>::const_iterator eI = bc.end();
  //we take care of the first two edges
  geo_element e0 = d.at(*bI++);
  geo_element e1 = d.at(*bI++);
  //the edges are sorted, but not necessarily the nodes
  size_type OrderedLocalIndex0 [2];
  size_type OrderedLocalIndex1 [2];
  if (e0[0]==e1[1]) {
    OrderedLocalIndex0[0] = 1;
    OrderedLocalIndex0[1] = 0;
    OrderedLocalIndex1[0] = 1;
    OrderedLocalIndex1[1] = 0;
  } 
  else if (e0[1]==e1[1]) {
    OrderedLocalIndex0[0] = 0;
    OrderedLocalIndex0[1] = 1;
    OrderedLocalIndex1[0] = 1;
    OrderedLocalIndex1[1] = 0;
  }
  else if (e0[1]==e1[0]) {
    OrderedLocalIndex0[0] = 0;
    OrderedLocalIndex0[1] = 1;
    OrderedLocalIndex1[0] = 0;
    OrderedLocalIndex1[1] = 1;
  }
  else /* e0[0]==e1[0] */ {
    OrderedLocalIndex0[0] = 1;
    OrderedLocalIndex0[1] = 0;
    OrderedLocalIndex1[0] = 0;
    OrderedLocalIndex1[1] = 1;
  }
  Float coord0[2][2]; 
  Float coord1[2][2]; 
  for (size_type i = 0 ; i < 2 ; i++) {
    for (size_type j = 0 ; j < 2 ; j++) {
      coord0[i][j] = sommets[e0[OrderedLocalIndex0[i]]][j];
      coord1[i][j] = sommets[e1[OrderedLocalIndex1[i]]][j];
    }
  }
  point n0(-(coord0[1][1]-coord0[0][1]), coord0[1][0]-coord0[0][0]);
  point n1(-(coord1[1][1]-coord1[0][1]), coord1[1][0]-coord1[0][0]);
  point n_avrg = n0 + n1;
  n_avrg = n_avrg/norm(n_avrg);
  Float edge0 = norm(n0);
  Float edge1 = norm(n1);
  tiny_vector<size_t> dof;
  size_type d0,d1;
  for (size_type id = 0 ; id < Nh.n_component(); id++){
    Nh.set_global_dof(e0, dof, id);
    d0 = dof[OrderedLocalIndex0[0]];
    d1 = dof[OrderedLocalIndex0[1]];
    if (bc.first_normal_imposed()||bc.first_tangent_imposed())
      nh.at(Nh.domain_index(d0)) = (bc.first_normal())(id);
    else {
      nh.at(Nh.domain_index(d0)) = n0(id)/edge0;
      //warning_macro("first normal is not imposed. Use set_first_normal() if you want to impose it");
    }
    if (approx == "P2")
      nh.at(Nh.domain_index(dof[2])) = n0(id)/edge0;
    nh.at(Nh.domain_index(d1)) = n_avrg(id);
    if (approx == "P2") {
      Nh.set_global_dof(e1, dof, id);
      nh.at(Nh.domain_index(dof[2])) = n1(id)/edge1;
    }
  }
  n0 = n1;
  size_type indexp = e1[OrderedLocalIndex1[1]];
  for( ; bI != eI ; bI++) {
    geo_element e = d.at(*bI);
    if (indexp == e[0]) {
      OrderedLocalIndex1[0] = 0;
      OrderedLocalIndex1[1] = 1;
    } else {
      OrderedLocalIndex1[0] = 1;
      OrderedLocalIndex1[1] = 0;
    }
    for (size_type i = 0 ; i < 2 ; i++) {
      for (size_type j = 0 ; j < 2 ; j++) {
	coord1[i][j] = sommets[e[OrderedLocalIndex1[i]]][j];
      }
    }
    n1 = point(-(coord1[1][1]-coord1[0][1]), (coord1[1][0]-coord1[0][0]));
    n_avrg = n0 + n1;
    edge0 = norm(n0);
    edge1 = norm(n1);
    n_avrg = n_avrg/::sqrt(dot(n_avrg,n_avrg));
    for (size_type id = 0 ; id < Nh.n_component(); id++){
      Nh.set_global_dof(e, dof, id);
      d0 = dof[OrderedLocalIndex1[0]];
      d1 = dof[OrderedLocalIndex1[1]];
      nh.at(Nh.domain_index(d0)) = n_avrg(id);
      if (approx == "P2")
	nh.at(Nh.domain_index(dof[2])) = n1(id)/edge1;
    }
    indexp = e[OrderedLocalIndex1[1]];
    n0 = n1;
  }
  geo_element e = d.at(*(--bI));
  for (size_type id = 0 ; id < Nh.n_component(); id++){
    Nh.set_global_dof(e, dof, id);
    d1 = dof[OrderedLocalIndex1[1]];
    if (bc.last_normal_imposed()||bc.last_tangent_imposed())
      nh.at(Nh.domain_index(d1)) = (bc.last_normal())(id);
    else {
      nh.at(Nh.domain_index(d1)) = n1(id)/edge1;
      //warning_macro("last normal is not imposed. Use set_last_normal() if you want to impose it");
    }
  }
  return nh;
}
// computes a P0, P1, or P2 normal 
class field
georep::get_normal (const space& Nh, const domain& d, 
	bool axisymmetric_curvature, const interface& bc) const
{
  string approx = Nh.get_approx();
  check_macro((approx=="P0"||approx=="P1"||approx=="P2"), 
	      approx + " normal not implemented.");
  if (approx=="P0") {
    return get_normal_P0 (Nh, d, axisymmetric_curvature, bc);
  }
  check_macro(!axisymmetric_curvature, "axisymetric curvature not implemented for " + approx + " normals.");
  return get_normal_P1P2 (Nh, d, bc);
}
inline
point ortho2d(const point& AB)
{
    return point(-AB[1],AB[0]);
}
inline
point normal2d(const point& AB)
{
    return point(-AB[1],AB[0])/norm(AB);
}

inline
Float curvature2d(const point& AB, const point& BC)
// calculates the curvature at B using a quadratic approximation
// with sA=-|AB|, sB=0, sC=|BC|, we get
//  x'y''-x''y' = -2 |AB x BC| / ( sA sC (sC-sA) )
// (which is indep of s), and
//  (x'^2+y'^2)(s=0) = |AC|^2 /(sC-sA)^2
 {
   Float sA=-norm(AB); 
   Float sC=norm(BC);
   return -2*(sC-sA)*vect2d(AB,BC) / sA / sC / norm2(AB+BC);
 }
inline
point vcurvature2d(const point& AB, const point& BC)
// calculates the curvature at B using a quadratic approximation
 {
   Float sA2=norm2(AB); Float sA=-::sqrt(sA2); 
   Float sC2=norm2(BC); Float sC=::sqrt(sC2); 
   return 2*(sC-sA)*vect2d(AB,BC) / sA2 / sC2 / ::pow(norm2(AB+BC), Float(1.5))
   		*(sA2*ortho2d(BC)+sC2*ortho2d(AB));
 }
inline
point normal2d(const point& AB, const point& BC)
// calculates the normal at B using a quadratic approximation
 {
   Float sA2=norm2(AB); Float sA=-::sqrt(sA2); 
   Float sC2=norm2(BC); Float sC=::sqrt(sC2); 
   return -1 / sA / sC / norm(AB+BC) *(sA2*ortho2d(BC)+sC2*ortho2d(AB));
 }
inline
Float curvature2d_axi(const point& M, const point& AB, const point& er)
// n.er = |AB x er|/|AB| gives the tan of angle of edge to horizontal,
// then curvature is calculated as the inverse of the distance
// form middle of edge to a point on the axis s.t....
 {
   return -vect2d(AB,er)/dot(M,er)/norm(AB);
 }

/*
// calculates the inverse of the radius of a circle going through A, B and C
// curvature sign is determined independently
{
    if (abs(vect2d(AB,BC))<sqrt(numeric_limits<Float>::epsilon()))
        // points are aligned
    	return 0;
    Float sign=1;
    if (vect2d(AB,BC)<0) sign=-1;
    // let M be the middle of BC:
    Float dist_OM=-.5*dot(AB+BC,AB)/dot(normal2d(BC),AB);
    return sign/sqrt(sqr(dist_OM)+norm2(BC)/4.);
}
inline
point vcurvature2d(const point& AB, const point& BC)
{
    if (abs(vect2d(AB,BC))<sqrt(numeric_limits<Float>::epsilon()))
        // points are aligned
    	return point(0);
    Float signed_dist_OM=-.5*dot(AB+BC,AB)/dot(normal2d(BC),AB);
    return .5*BC+signed_dist_OM*normal2d(BC);
}
*/

void
georep::sort_interface (const domain& d, const interface& bc) const
 {
    if (bc.is_sorted()) return;
    // Initialize interface boundary condition 
    if (!bc.initialized())
	set_interface_orientation(d, bc);
    // Sort contiguous elements of the interface
    //  first, group them in contiguous stubs listing element S that are merged
    //  together as the jigsaw progresses...
    typedef list< size_type > stub;
    typedef list< stub > set_of_stubs;
    set_of_stubs S;
    set_of_stubs::iterator iS;
    set_of_stubs::iterator eS;
    set_of_stubs::iterator iS_attached;
    // lists of the first and last vertices in stubs -- misleadingly of type 'stub' as
    // well, these contain *vertex indices*, not *element* ones. 
    stub F, L; 
    stub::iterator iF, iL, iF_attached, iL_attached;
    domain::const_iterator bd=d.begin();
    domain::const_iterator ed=d.end();
    size_type d_index=0;
    bool actual_order_is_set=false;
    for (domain::const_iterator id=bd; id!=ed; id++, d_index++)
    {
        bool attached=false;
	bool over=false;
	int attached_dir=-1;
	for (size_type i=0; i<2; i++) // examine both orders for attachment
	for (iS=S.begin(), eS=S.end(),
	     iF=F.begin(),
	     iL=L.begin();
	     !over && iS!=eS; iS++, iF++, iL++)
	{
	    if (*iF==(*id)[i]) 
	    {
	    // first element of the stub has a common vertex with *id, then
	    // attach *id to front of stub *iS
	        if (!attached) 
		// other end of id is still free
		{
		  (*iS).push_front(d_index);
		  *iF=((*id)[(i+1)%2]);
		  attached=true;
		  iS_attached=iS;
		  iF_attached=iF;
		  iL_attached=iL;
		  attached_dir=i;
		}
		else
		if (iS_attached!=iS) 
		// other end is already attached to a stub: merge them
		{
  		  if (i==0) // the other stub is in the other order
		  {
		    *iF=*iL_attached;
		    (*iS_attached).reverse();
		  }
		  else *iF=*iF_attached;   
		  (*iS).splice((*iS).begin(),*iS_attached);
		  (*iS_attached).~stub();
		  F.erase(iF_attached);
		  L.erase(iL_attached);
		  S.erase(iS_attached);
		  over=true;
		}
	    }
//	    if (bd[*(*iS).rbegin()][1]==(*id)[i])
	    if (*iL==(*id)[i])
	    {
	    // last element of the stub has a common vertex with *id, then
	    // attach *id to back of stub *iS 
	        if (!attached)
		{
		  (*iS).push_back(d_index);
		  *iL=((*id)[(i+1)%2]);
		  attached=true;
		  iS_attached=iS;
		  iF_attached=iF;
		  iL_attached=iL;
		  attached_dir=i;
		}
		else
		if (iS_attached!=iS) 
		{
  		  if (i==1) // the other stub is in the other order
		  {
		    *iL=*iF_attached;
		    (*iS_attached).reverse();
		  }
		  else *iL=*iL_attached;
		  (*iS_attached).splice((*iS_attached).begin(), *iS);
		  (*iS).~stub();
		  F.erase(iF_attached);
		  L.erase(iL_attached);
		  S.erase(iS);
		  over=true;
		}
	    }
	}
	if (!attached) // we have to create a new stub for this element
        {
            stub* newstub =new stub;
            (*newstub).push_front(d_index);
            S.push_front(*newstub);
	    F.push_front((*id)[0]);
	    L.push_front((*id)[1]);
	    attached_dir=0;
        }
	if (d_index==bc.element_of_known_normal)
	{
	    check_macro(attached_dir!=-1,"Attached direction not set !" << S.size());
	    bc.set_actual_order( ((attached_dir==0)?1:-1)
	    	*(_x.begin()[(*id)[1]] - _x.begin()[(*id)[0]]));
	    //	    cerr << "$$ Actual order : " << ((attached_dir==0)?1:-1)*(_x.begin()[(*id)[1]] - _x.begin()[(*id)[0]]) << "( dir " << ((attached_dir==0)?1:-1) << " )"<< endl;
	    actual_order_is_set=true;
	}
    }
    check_macro (actual_order_is_set, "Element whose normal is marked " 
    	<< bc.element_of_known_normal << " not found!");
    check_macro (S.size()==1, "The interface has to be a connected set!");
    if ((*F.begin())!=(*L.begin())) { warning_macro( "Closed interfaces not tested yet! Weird things might happen... \nIn particular, tangents/normals at 'end' points...") ;}
    check_macro ((*S.begin()).size()==d.size(), "Not all elements found?");
    if (bc.is_reversed())
      bc.set_sorted( (*S.begin()).size(), (*S.begin()).rbegin(), (*S.begin()).rend());
    else
      bc.set_sorted( (*S.begin()).size(), (*S.begin()).begin(), (*S.begin()).end());
    
    // get global dof number for first and last dof
    Vector<size_t>::const_iterator iAB=bc.begin();
    Vector<size_t>::const_iterator iBC=bc.begin(); iBC++;
    size_t AB0=bd[*iAB][0];
    size_t AB1=bd[*iAB][1];
    size_t BC0=bd[*iBC][0];
    size_t BC1=bd[*iBC][1];
    // check that nodes are in the same order as elements, otherwise reverse
    if (AB1==BC0||AB1==BC1) { bc.set_first_node(*iAB, 0); }
    else
    if (AB0==BC0||AB0==BC1) { bc.set_first_node(*iAB, 1); }
    else error_macro("Fatal: non consecutive edges!");

    iAB=bc.end(); iAB--;
    iBC=iAB; iBC--;
    AB0=bd[*iAB][0];
    AB1=bd[*iAB][1];
    BC0=bd[*iBC][0];
    BC1=bd[*iBC][1];
    // check that nodes are in the same order as elements, otherwise reverse
    if (AB1==BC0||AB1==BC1) { bc.set_last_node(*iAB,0); }
    else
    if (AB0==BC0||AB0==BC1) { bc.set_last_node(*iAB,1); }
    else error_macro("Fatal: non consecutive edges!");
 }
field
georep::plane_curvature (const space& Nh, const domain& d, 
	const interface& bc) const
{
    return plane_curvature_spline (Nh, d, bc);
}

field
georep::tangent_spline (const space& Nh, const domain& d, 
	const interface& bc) const
{
  check_macro(Nh.get_geo() .dimension()==2,"plane curvature is for 2D geometries only");
  check_macro(Nh.n_component ()==2,"2-component vector space needed");
  // Sort contiguous elements of the interface
  // first ,group them in contiguous stubs listing element #s that are merged
  // together as the jidsaw progresses....
  sort_interface(d,bc);
  // get coodinate of lagrange interpolation points on edge elements :
  vector<point> hat_node;
  domain::const_iterator bd=d.begin () ;
  Nh.get_basis().hat_node(*bd,hat_node) ;

  Vector<size_t>::const_iterator iAB=bc.begin() ;
  Vector<size_t>::const_iterator iBC=bc.begin() ;  iBC++;
  Vector<size_t>::const_iterator the_end=bc.end() ;

  // calculate interface curvature using a parametric spline
  
  field kh(Nh,float(0));
  tiny_vector<size_type> dofs;
 
  point AB0=_x.begin()[bd[*iAB][0]];
  point AB1=_x.begin()[bd[*iAB][1]];
  point BC0=_x.begin()[bd[*iBC][0]];
  point BC1=_x.begin()[bd[*iBC][1]];
  point AB,BC;
  bool rev_AB;

  /* calulate zi
     compute the hi and the bi */
  size_type n=d.size();
  vector<Float> u(n+1),h(n);
  vector<point> v(n+1),b(n+1),z(n+1) ,y(n+1),w(n+1);
  size_type i;
  for (i=0 ; iBC!=the_end; iAB++,iBC++,i++)
    {
 	// check that nodes are in the same order as elements ,otherwise reverse 
	BC0=_x.begin()[bd[*iBC][0]];
   	BC1=_x.begin()[bd[*iBC][1]];
 	if(AB1==BC0) { rev_AB=false; AB=AB1-AB0 ; BC=BC1-BC0; }
 	else
 	if(AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0 ; }
 	else
   	if(AB1==BC1) { rev_AB=false;AB=AB1-AB0; BC=BC0-BC1 ; }
   	else
    	{
     		check_macro (AB0==BC1, "Fatal :non consecutive edges!");
  		rev_AB=true; AB=AB0-AB1; BC=BC0-BC1;
    	}
       	h[i]=norm(AB);
       	b[i]=1./h[i]*AB;
     	AB0=BC0; AB1=BC1 ;AB=BC;
     	 
    }
   
   h[n-1]=norm(BC); b[n-1]=1./h[n-1]*BC;
  // GAUSSIAN ELIMINATION 
  size_type n_loop=n;
  size_type start_loop=1;
  point t0;
  point tn;
  if (bc.first_tangent_imposed())
    {
     t0=bc.first_tangent();
     u[0]=2*(h[0]);
     v[0]=6*(b[0]-t0);
     u[1]=2*(h[0]+h[1])-sqr(h[0])/u[0];
     v[1]=6*(b[1]-b[0])- (h[0]/u[0])*v[0];
     start_loop =0;
    }
  else 
    {
     u[1]=2*(h[0]+h[1]);
     v[1]=6*(b[1]-b[0]);
     start_loop=1 ;
    }
  for (i=2; i<=(n-1); i++)
     { 
       u[i]=2.*(h[i-1]+h[i])-sqr(h[i-1])/u[i-1];
       v[i]=6.*(b[i]-b[i-1])-h[i-1]/u[i-1]*v[i-1];
     }
  if (bc.last_tangent_imposed())
   { 
    tn=bc.last_tangent();
    n_loop=n ;
    u[n]=2*h[n-1]-sqr(h[n-1])/u[n-1] ;
    v[n]= 6*(tn-b[n-1])-h[n-1]/u[n-1]*v[n-1];
   }
  else 
   { 
    n_loop=n-1;
    u[n]=2*h[n-1]-sqr(h[n-1])/u[n-1] ;
    v[n]= 6*(tn-b[n-1])-h[n-1]/u[n-1]*v[n-1];
   }
      

  //back substitution 
  if (!bc.last_tangent_imposed()) {z[n]=point(0,0);}
  else z[n]=1./u[n]*v[n];
     for (int j=n-1; j>=int(start_loop); j--)
       {
       z[j]=1./u[j]*( v[j] -h[j]*z[j+1] );
     
       w[j]=-h[j]/6 *(z[j+1]-z[j]) +b[j];
      
       }
  if (!bc.first_tangent_imposed()){z[0]=point(0,0);w[0]=-h[0]/6 *(z[1]-z[0]) +b[0];}
   
  // curvature calculation
  iAB=bc.begin() ;
  iBC=bc.begin() ;  iBC++;
  AB0=_x.begin()[bd[*iAB][0]];
  AB1=_x.begin()[bd[*iAB][1]];
 
  point tang;
  for (i=0 ; iBC!=the_end; iAB++,iBC++,i++)
   {
	BC0=_x.begin()[bd[*iBC][0]];
	BC1=_x.begin()[bd[*iBC][1]];
 	// check that nodes are in the same order as elements ,otherwise reverse 

 	if(AB1==BC0) { rev_AB=false; AB=AB1-AB0 ; BC=BC1-BC0; }
 	else
 	if(AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0 ; }
 	else
   	if(AB1==BC1) { rev_AB=false;AB=AB1-AB0; BC=BC0-BC1 ; }
   	else
    	{
     		check_macro (AB0==BC1, "Fatal :non consecutive edges!");
  		rev_AB=true; AB=AB0-AB1; BC=BC0-BC1;
    	}
  	for (size_type j=0;j<Nh.n_component();j++)
   	  {
   	        // get nh dofs corresponding

   	        Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,j);
   	        //set dofs

  	        for (size_type idof=0;idof<dofs.size();idof++)
		  {
   		   if(!rev_AB)
   		     {
               		tang=-sqr(1-hat_node[idof][0])*z[i]/2*h[i]  
				    +sqr(hat_node[idof][0])*z[i+1]/2*h[i] 
				    +w[i];
  		     }
   		   else
    		     {
   		        //cerr << (1-hat_node[idof][0]) << "*" << z[i][j] 
   		            // << " + " << hat_node[idof][0] << "*" << z[i+1][j] << endl;
                        tang=sqr(1-hat_node[idof][0])* z[i+1]/2*h[i] 
				   -sqr(hat_node[idof][0])*z[i]/2*h[i] 
				   +w[i];
   		     }
                   kh.at(dofs[idof])= tang[j];
                 
		 } // idof
  	  } // j<n_compo
     	AB0=BC0; AB1=BC1 ;AB=BC;
  } // iAB
 // don't need to check that nodes are in the same order as elements,as BC was

 rev_AB=(AB==AB0 -AB1);
 i=n-1;
 for (size_type j=0;j<Nh.n_component();j++)
 {
   	        // get nh dofs corresponding

   	        Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,j);
   	        //set dofs
              
  	        for (size_type idof=0;idof<dofs.size();idof++)
		 {
		
   		   if(!rev_AB)
   		     {
               		tang=-sqr(1-hat_node[idof][0])*z[i]/2*h[i] 
				    +sqr(hat_node[idof][0])* z[i+1]/2*h[i] 
				    +w[i];
  		     }
   		   else
    		     {
   		        //cerr << (1-hat_node[idof][0]) << "*" << z[i][j] 
   		            // << " + " << hat_node[idof][0] << "*" << z[i+1][j] << endl;
                        tang=sqr(1-hat_node[idof][0])*z[i+1]/2*h[i] 
				   -sqr(hat_node[idof][0])*z[i]/2*h[i] 
				   +w[i];
   		     }
                   kh.at(dofs[idof])= tang[j];
                  
		 }
  }
  return kh;
}

field
georep::plane_curvature_spline (const space& Nh, const domain& d, 
	const interface& bc) const
{
  check_macro(Nh.get_geo() .dimension()==2,"plane curvature is for 2D geometries only");
  check_macro(Nh.n_component ()==2,"2-component vector space needed");
  // Sort contiguous elements of the interface
  // first ,group them in contiguous stubs listing element #s that are merged
  // together as the jidsaw progresses....
  sort_interface(d,bc);
  // get coodinate of lagrange interpolation points on edge elements :
  vector<point> hat_node;
  domain::const_iterator bd=d.begin () ;
  Nh.get_basis().hat_node(*bd,hat_node) ;

  Vector<size_t>::const_iterator iAB=bc.begin() ;
  Vector<size_t>::const_iterator iBC=bc.begin() ;  iBC++;
  Vector<size_t>::const_iterator the_end=bc.end() ;

  // calculate interface curvature using a parametric spline
  
  field kh(Nh,float(0));
  tiny_vector<size_type> dofs;
 
  point AB0=_x.begin()[bd[*iAB][0]];
  point AB1=_x.begin()[bd[*iAB][1]];
  point BC0=_x.begin()[bd[*iBC][0]];
  point BC1=_x.begin()[bd[*iBC][1]];
  point AB,BC;
  bool rev_AB;

  /* calulate zi
     compute the hi and the bi */
  size_type n=d.size();
  vector<Float> u(n+1),h(n);
  vector<point> v(n+1),b(n+1),z(n+1) ,y(n+1),w(n+1);
  size_type i;
  for (i=0 ; iBC!=the_end; iAB++,iBC++,i++)
    {
 	// check that nodes are in the same order as elements ,otherwise reverse 
	BC0=_x.begin()[bd[*iBC][0]];
   	BC1=_x.begin()[bd[*iBC][1]];
 	if(AB1==BC0) { rev_AB=false; AB=AB1-AB0 ; BC=BC1-BC0; }
 	else
 	if(AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0 ; }
 	else
   	if(AB1==BC1) { rev_AB=false;AB=AB1-AB0; BC=BC0-BC1 ; }
   	else
    	{
     		check_macro (AB0==BC1, "Fatal :non consecutive edges!");
  		rev_AB=true; AB=AB0-AB1; BC=BC0-BC1;
    	}
       	h[i]=norm(AB);
       	b[i]=1./h[i]*AB;
     	AB0=BC0; AB1=BC1 ;AB=BC;
     	 
    }
   
   h[n-1]=norm(BC); b[n-1]=1./h[n-1]*BC;
  // GAUSSIAN ELIMINATION 
  size_type n_loop=n;
  size_type start_loop=1;
  point t0;
  point tn;
  if (bc.first_tangent_imposed())
    {
     t0=bc.first_tangent();
     u[0]=2*(h[0]);
     v[0]=6*(b[0]-t0);
     u[1]=2*(h[0]+h[1])-sqr(h[0])/u[0];
     v[1]=6*(b[1]-b[0])- (h[0]/u[0])*v[0];
     start_loop =0;
    }
  else 
    {
     u[1]=2*(h[0]+h[1]);
     v[1]=6*(b[1]-b[0]);
     start_loop=1 ;
    }
  for (i=2; i<=(n-1); i++)
     { 
       u[i]=2.*(h[i-1]+h[i])-sqr(h[i-1])/u[i-1];
       v[i]=6.*(b[i]-b[i-1])-h[i-1]/u[i-1]*v[i-1];
     }
  if (bc.last_tangent_imposed())
   { 
    tn=bc.last_tangent();
    n_loop=n ;
    u[n]=2*h[n-1]-sqr(h[n-1])/u[n-1] ;
    v[n]= 6*(tn-b[n-1])-h[n-1]/u[n-1]*v[n-1];
   }
  else 
   { 
    n_loop=n-1;
    u[n]=2*h[n-1]-sqr(h[n-1])/u[n-1] ;
    v[n]= 6*(tn-b[n-1])-h[n-1]/u[n-1]*v[n-1];
   }
      

  //back substitution 
  if (!bc.last_tangent_imposed()) {z[n]=point(0,0);}
  else z[n]=1./u[n]*v[n];
     for (int j=n-1; j>=int(start_loop); j--)
       {
       z[j]=1./u[j]*( v[j] -h[j]*z[j+1] );
     
       w[j]=-h[j]/6 *(z[j+1]-z[j]) +b[j];
      
       }
  if (!bc.first_tangent_imposed()){z[0]=point(0,0);w[0]=-h[0]/6 *(z[1]-z[0]) +b[0];}
   
  // curvature calculation
  iAB=bc.begin() ;
  iBC=bc.begin() ;  iBC++;
  AB0=_x.begin()[bd[*iAB][0]];
  AB1=_x.begin()[bd[*iAB][1]];
 
  point z_at_dof, devy_at_dof, n1;
  for (i=0 ; iBC!=the_end; iAB++,iBC++,i++)
   {
	BC0=_x.begin()[bd[*iBC][0]];
	BC1=_x.begin()[bd[*iBC][1]];
 	// check that nodes are in the same order as elements ,otherwise reverse 

 	if(AB1==BC0) { rev_AB=false; AB=AB1-AB0 ; BC=BC1-BC0; }
 	else
 	if(AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0 ; }
 	else
   	if(AB1==BC1) { rev_AB=false;AB=AB1-AB0; BC=BC0-BC1 ; }
   	else
    	{
     		check_macro (AB0==BC1, "Fatal :non consecutive edges!");
  		rev_AB=true; AB=AB0-AB1; BC=BC0-BC1;
    	}
  	for (size_type j=0;j<Nh.n_component();j++)
   	  {
   	        // get nh dofs corresponding

   	        Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,j);
   	        //set dofs

  	        for (size_type idof=0;idof<dofs.size();idof++)
		  {
   		   if(!rev_AB)
   		     {
  	       		z_at_dof= (1-hat_node[idof][0])* z[i] + hat_node[idof][0]*z[i+1] ;
               		devy_at_dof=-sqr(1-hat_node[idof][0])*z[i]/2*h[i]  
				    +sqr(hat_node[idof][0])*z[i+1]/2*h[i] 
				    +w[i];
                        n1=point(devy_at_dof[1],-devy_at_dof[0]);
  		     }
   		   else
    		     {
   		        //cerr << (1-hat_node[idof][0]) << "*" << z[i][j] 
   		            // << " + " << hat_node[idof][0] << "*" << z[i+1][j] << endl;
   			z_at_dof=(1-hat_node[idof][0])* z[i+1] + hat_node[idof][0] *z[i];
                        devy_at_dof=sqr(1-hat_node[idof][0])* z[i+1]/2*h[i] 
				   -sqr(hat_node[idof][0])*z[i]/2*h[i] 
				   +w[i];
                        n1=point(devy_at_dof[1],-devy_at_dof[0]);
   		     }
                   kh.at(dofs[idof])= -1./(sqr(sqr(devy_at_dof[0])+sqr(devy_at_dof[1])))
				*(z_at_dof[1]*devy_at_dof[0]-z_at_dof[0]*devy_at_dof[1])
				*n1[j];
                 
		 } // idof
  	  } // j<n_compo
     	AB0=BC0; AB1=BC1 ;AB=BC;
  } // iAB
 // don't need to check that nodes are in the same order as elements,as BC was

 rev_AB=(AB==AB0 -AB1);
 i=n-1;
 for (size_type j=0;j<Nh.n_component();j++)
 {
   	        // get nh dofs corresponding

   	        Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,j);
   	        //set dofs
              
  	        for (size_type idof=0;idof<dofs.size();idof++)
		 {
		
   		   if(!rev_AB)
   		     {
  	       		z_at_dof= (1-hat_node[idof][0])* z[i] + hat_node[idof][0] *z[i+1];
               		devy_at_dof=-sqr(1-hat_node[idof][0])*z[i]/2*h[i] 
				    +sqr(hat_node[idof][0])* z[i+1]/2*h[i] 
				    +w[i];
                        n1=point(devy_at_dof[1],-devy_at_dof[0]);
  		     }
   		   else
    		     {
   		        //cerr << (1-hat_node[idof][0]) << "*" << z[i][j] 
   		            // << " + " << hat_node[idof][0] << "*" << z[i+1][j] << endl;
   			z_at_dof=(1-hat_node[idof][0])* z[i+1] + hat_node[idof][0] *z[i];
                        devy_at_dof=sqr(1-hat_node[idof][0])*z[i+1]/2*h[i] 
				   -sqr(hat_node[idof][0])*z[i]/2*h[i] 
				   +w[i];
                        n1=point(devy_at_dof[1],-devy_at_dof[0]);
   		     }
                   kh.at(dofs[idof])= -1./(sqr(sqr(devy_at_dof[0])+sqr(devy_at_dof[1])))
				*(z_at_dof[1]*devy_at_dof[0]-z_at_dof[0]*devy_at_dof[1])
				*n1[j];
                  
		 }
  }
  return kh;
}

field
georep::plane_curvature_quadratic (const space& Nh, const domain& d, 
	const interface& bc) const
{
    check_macro(Nh.get_geo().dimension()==2,"Plane curvature is for 2D geometries only");
    check_macro(Nh.n_component()==2,"2-component vector space needed");
    // Sort contiguous elements of the interface
    //  first, group them in contiguous stubs listing element #s that are merged
    //  together as the jigsaw progresses...
    sort_interface(d,bc);

    // get coordinates of Lagrange interpolation points on edge elements:
    vector<point> hat_node;
    domain::const_iterator bd=d.begin();
    Nh.get_basis().hat_node(*bd, hat_node);

    Vector<size_t>::const_iterator iAB=bc.begin();
    Vector<size_t>::const_iterator iBC=bc.begin(); iBC++;
    Vector<size_t>::const_iterator the_end=bc.end();
    // Calculate interface curvature using the mean of the curvature of
    // circles passing through points e--,e-,e+ (curv1) and e-,e+,e++ (curv2)
    field kh(Nh, Float(0));
    tiny_vector<size_type> dofs;
    point vcurv1, vcurv2;
    point AB0=_x.begin()[bd[*iAB][0]];
    point AB1=_x.begin()[bd[*iAB][1]];
    point BC0=_x.begin()[bd[*iBC][0]];
    point BC1=_x.begin()[bd[*iBC][1]];
    point AB, BC;
    bool rev_AB;
    // check that nodes are in the same order as elements, otherwise reverse
    if (AB1==BC0) { rev_AB=false; AB=AB1-AB0; BC=BC1-BC0; }
    else
      if (AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0; }
      else
        if (AB1==BC1) { rev_AB=false; AB=AB1-AB0; BC=BC0-BC1;  }
        else
         {
    	    check_macro(AB0==BC1, "Fatal: non consecutive edges!");
	    rev_AB=true; AB=AB0-AB1; BC=BC0-BC1; 
         }
    if (bc.first_tangent_imposed())
    {
     	vcurv1=vcurvature2d(AB-2*dot(AB,bc.first_normal())*bc.first_normal(), AB);
    }
    else
    	vcurv1=vcurvature2d(AB,BC);
    for ( ; iBC!=the_end; iAB++, iBC++) 
    {
        BC0=_x.begin()[bd[*iBC][0]];
        BC1=_x.begin()[bd[*iBC][1]];
        // check that nodes are in the same order as elements, otherwise reverse
        if (AB1==BC0) { rev_AB=false; AB=AB1-AB0; BC=BC1-BC0; }
        else
          if (AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0; }
          else
            if (AB1==BC1) { rev_AB=false; AB=AB1-AB0; BC=BC0-BC1;  }
	    else
	     {
	        check_macro(AB0==BC1, "Fatal: non consecutive edges!");
                rev_AB=true; AB=AB0-AB1; BC=BC0-BC1; 
	     }
        vcurv2=vcurvature2d(AB,BC);
        /*
	check_macro(abs(norm(vcurv2)-abs(curvature2d(AB,BC)))<1e-12,
		"curvature error " << AB << " U " << BC << " " << norm(vcurv2) 
		<< "-" << curvature2d(AB,BC) << "=" << norm(vcurv2)-abs(curvature2d(AB,BC)));
	*/
	for (size_type i=0; i<Nh.n_component(); i++) 
	{
	    // get nh dofs corresponding
	    Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,i);
	    // set dofs
	    for (size_type idof=0; idof<dofs.size(); idof++)
	    if (!rev_AB)
	      kh.at(dofs[idof])=(1-hat_node[idof][0])*vcurv1[i] + hat_node[idof][0]*vcurv2[i];
	    else
	      kh.at(dofs[idof])=(1-hat_node[idof][0])*vcurv2[i] + hat_node[idof][0]*vcurv1[i];
	}
	vcurv1=vcurv2;
	AB0=BC0; AB1=BC1; AB=BC;
    }
    // don't need to check that nodes are in the same order as elements, as BC was
    rev_AB=(AB==AB0-AB1);
    if (bc.last_tangent_imposed())
    {
     	vcurv2=vcurvature2d(AB, AB-2*dot(AB,bc.last_normal())*bc.last_normal());
    }
    else vcurv2=vcurv1;
    for (size_type i=0; i<Nh.n_component(); i++)
    {
	// get nh dofs corresponding
	Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,i);
	// set dofs
	for (size_type idof=0; idof<dofs.size(); idof++)
	if (!rev_AB)
	  kh.at(dofs[idof])=(1-hat_node[idof][0])*vcurv1[i] + hat_node[idof][0]*vcurv2[i];
	else
	  kh.at(dofs[idof])=(1-hat_node[idof][0])*vcurv2[i] + hat_node[idof][0]*vcurv1[i];
    }
    return kh;
}


field
georep::axi_curvature (const space& Nh, const domain& d, 
	const interface& bc) const
{
    check_macro(Nh.coordinate_system()=="rz"||Nh.coordinate_system()=="zr",
    	"Axisymmetric curvature is for 2D-axisymmetric geometries only");
    check_macro(Nh.n_component()==2,"2-component vector space needed");
    // Sort contiguous elements of the interface
    //  first, group them in contiguous stubs listing element #s that are merged
    //  together as the jigsaw progresses...
    sort_interface(d,bc);

    point er;
    if (Nh.coordinate_system()=="rz") { er=point(1,0);  }
	else { er=point(0,1);  }
    // fill in with zeros in case only part of domain d is bounding region
    field nh(Nh, Float(0));

    // get coordinates of Lagrange interpolation points on edge elements:
    vector<point> hat_node;
    domain::const_iterator bd=d.begin();
    Nh.get_basis().hat_node(*bd, hat_node);

    Vector<size_t>::const_iterator iAB=bc.begin();
    Vector<size_t>::const_iterator iBC=bc.begin(); iBC++;
    Vector<size_t>::const_iterator the_end=bc.end();
    field kh(Nh, Float(0));
    tiny_vector<size_type> dofs;
    Float curv0, curv1, curv2;
    point nA, nB;
    point AB0=_x.begin()[bd[*iAB][0]];
    point AB1=_x.begin()[bd[*iAB][1]];
    point BC0=_x.begin()[bd[*iBC][0]];
    point BC1=_x.begin()[bd[*iBC][1]];
    point AB, BC;
    bool rev_AB=false;
    // check that nodes are in the same order as elements, otherwise reverse
    if (AB1==BC0) { rev_AB=false; AB=AB1-AB0; BC=BC1-BC0; }
    else
      if (AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0; }
      else
        if (AB1==BC1) { rev_AB=false; AB=AB1-AB0; BC=BC0-BC1;  }
	else
	 {
	    check_macro(AB0==BC1, "Fatal: non consecutive edges!");
            rev_AB=true; AB=AB0-AB1; BC=BC0-BC1; 
	 }
    curv0=curv1=curvature2d_axi((AB0+AB1)/2, AB, er);
    if (bc.first_tangent_imposed())
    	nA=normal2d(AB-2*dot(AB,bc.first_normal())*bc.first_normal(), AB);
    else
	nA=normal2d(AB,BC);
    for ( ; iBC!=the_end; iAB++, iBC++) 
    {
        BC0=_x.begin()[bd[*iBC][0]];
        BC1=_x.begin()[bd[*iBC][1]];
        // check that nodes are in the same order as elements, otherwise reverse
        if (AB1==BC0) { rev_AB=false; AB=AB1-AB0; BC=BC1-BC0; }
        else
          if (AB0==BC0) { rev_AB=true; AB=AB0-AB1; BC=BC1-BC0; }
          else
            if (AB1==BC1) { rev_AB=false; AB=AB1-AB0; BC=BC0-BC1;  }
	    else
	     {
	        check_macro(AB0==BC1, "Fatal: non consecutive edges!");
                rev_AB=true; AB=AB0-AB1; BC=BC0-BC1; 
	     }
    	nB=normal2d(AB, BC);
    	curv2=curvature2d_axi((BC0+BC1)/2, BC, er);
	for (size_type i=0; i<Nh.n_component(); i++) 
	{
	    // get nh dofs corresponding
	    Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,i);
	    // set dofs
	    for (size_type idof=0; idof<dofs.size(); idof++)
	    if (!rev_AB)
	      kh.at(dofs[idof])=.5*(1-hat_node[idof][0])*(curv0+curv1)*nA[i]
	      	+ .5*hat_node[idof][0]*(curv1+curv2)*nB[i];
	    else
	      kh.at(dofs[idof])=.5*hat_node[idof][0]*(curv0+curv1)*nA[i]
	      	+ .5*(1-hat_node[idof][0])*(curv1+curv2)*nB[i];
	}
	curv0=curv1;
	curv1=curv2;
	AB0=BC0; AB1=BC1; AB=BC;
	nA=nB;
    }
    // don't need to check that nodes are in the same order as elements, as BC was
    rev_AB=(AB==AB0-AB1);
    curv2=curv1;
    if (bc.last_tangent_imposed())
    	nB=normal2d(AB, AB-2*dot(AB,bc.last_normal())*bc.last_normal());
    else 
	nB=nA;
    for (size_type i=0; i<Nh.n_component(); i++)
    {
	// get nh dofs corresponding
	Nh.set_dof((Nh.get_geo().begin())[*iAB],dofs,i);
	// set dofs
	for (size_type idof=0; idof<dofs.size(); idof++)
	    if (!rev_AB)
	      kh.at(dofs[idof])=.5*(1-hat_node[idof][0])*(curv0+curv1)*nA[i]
	      	+ .5*hat_node[idof][0]*(curv1+curv2)*nB[i];
	    else
	      kh.at(dofs[idof])=.5*hat_node[idof][0]*(curv0+curv1)*nA[i]
	      	+ .5*(1-hat_node[idof][0])*(curv1+curv2)*nB[i];
    }
    return kh;
}

void
geo::interface_process (const domain& d, const interface& bc,
	geometric_event& event) const
 {
    if (!bc.initialized())
     {
        space Nh(*this, d, "P0");
	data().get_normal(Nh, d, false, bc);
     }
    data().interface_process(d, bc, event);
 }
void
georep::interface_process (const domain& d, const interface& bc,
	geometric_event& event) const
 {
    sort_interface(d, bc);
    vector<point> hat_node;
    domain::const_iterator bd=d.begin();

    Vector<size_t>::const_iterator iAB=bc.begin();
    Vector<size_t>::const_iterator iBC=bc.begin(); iBC++;
    Vector<size_t>::const_iterator the_end=bc.end();
    
    point AB0=_x.begin()[bd[*iAB][0]];
    point AB1=_x.begin()[bd[*iAB][1]];
    point BC0=_x.begin()[bd[*iBC][0]];
    point BC1=_x.begin()[bd[*iBC][1]];
    point A, B;
    cerr << "!\tA=" << AB0;
    for ( ; iBC!=the_end; iAB++, iBC++) 
    {
        BC0=_x.begin()[bd[*iBC][0]];
        BC1=_x.begin()[bd[*iBC][1]];
        // check that nodes are in the same order as elements, otherwise reverse
        if (AB1==BC0) { A=AB0; B=AB1; }
        else
          if (AB0==BC0) { A=AB1; B=AB0; }
          else
            if (AB1==BC1) { A=AB0; B=AB1;  }
	    else
	     {
	        check_macro(AB0==BC1, "Fatal: non consecutive edges!");
                A=AB1; B=AB0; 
	     }
        event(A,B); 
	AB0=BC0; AB1=BC1; A=B;
    }
    cerr << " B=" << BC1 << endl;
cerr << "@";
    if (BC0==B) event(A,BC1); else event(A,BC0);
 }
}// namespace rheolef
