/**********************************************************************
* based on:   testpcap3.c
* date:   Sat Apr 07 23:23:02 PDT 2001
* Author: Martin Casado
*
* DLM additions: Christine Caulfield <ccaulfie@redhat.com> Oct 2008
*
* v0.2   Show unlock and lock held times
* v0.3   Show conversions and all locks in one packet
* v0.4   Add RHEL5 support(untested)
* v0.5   Fix bug that caused us to loop if length was zero (why??)
* v0.6   Show full command-line and timestamps (djuran)
*
**********************************************************************/

#define RHEL4

#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <search.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/if_ether.h>
#include <time.h>

static char *dlm_filter = "port 21064";

#ifdef RHEL4
#define DLM_LVB_LEN            (32)

struct dlm_message {
	uint32_t		m_version;
	uint8_t			m_cmd;		/* What we are */
	uint8_t			m_flags;	/* maybe just a pad */
	uint16_t		m_length;	/* Length of struct (so we can
						   send many in 1 message) */
	uint32_t		m_lkid;     	/* Lock ID tag: ie the local
						   (requesting) lock ID */
	uint32_t		m_lockspace;	/* Lockspace ID */
	uint32_t		m_remlkid;	/* Remote lock ID */
	uint32_t		m_remparid;	/* Parent's remote lock ID */
	uint32_t		m_lflags;	/* Flags from lock/convert req*/
	uint64_t		m_range_start; /* Yes, these are in the right
						   place... */
	uint64_t		m_range_end;
	uint32_t		m_status;	/* Status to return if this is
						   an AST request */
        uint32_t                m_pid;         /* Owner PID of lock */
	uint32_t		m_lvbseq;
	uint8_t			m_rqmode;	/* Requested lock mode */
	uint8_t			m_asts;	/* Whether the LKB has ASTs */
	char			m_lvb[DLM_LVB_LEN];
	char			m_name[1];	/* As long as needs be. Only
						   used for directory lookups.
						   The length of this can be
						   worked out from the packet
						   length */
} __attribute__((packed));

#define DLM_MSG_LOCK     2
#define DLM_MSG_UNLOCK   3
#define DLM_MSG_CONVERT  4

#define DLM_VERSION      0x00010001

#endif

#ifdef RHEL5

struct dlm_message {
	uint32_t		m_version;
	uint32_t		m_lockspace;
	uint32_t		m_nodeid;	/* nodeid of sender */
	uint16_t		m_length;
	uint8_t			m_msg;		/* DLM_MSG, DLM_RCOM */
	uint8_t			m_pad;
	uint32_t		m_msg;		/* DLM_MSG_ */
	uint32_t		m_rnodeid;
	uint32_t		m_pid;
	uint32_t		m_lkid;		/* lkid on sender */
	uint32_t		m_remid;	/* lkid on receiver */
	uint32_t		m_parent_lkid;
	uint32_t		m_parent_remid;
	uint32_t		m_exflags;
	uint32_t		m_sbflags;
	uint32_t		m_flags;
	uint32_t		m_lvbseq;
	uint32_t		m_hash;
	int			m_status;
	int			m_grmode;
	int			m_rqmode;
	int			m_bastmode;
	int			m_asts;
	int			m_result;	/* 0 or -EXXX */
	char			m_name[0];	/* name or lvb */
};

#define DLM_MSG_LOCK     1
#define DLM_MSG_UNLOCK   3
#define DLM_MSG_CONVERT  2

#define DLM_VERSION      0x00030000

#endif

#ifndef DLM_MSG_LOCK
#error You must build for RHEL4 or RHEL5
#endif

/* Struct for a lock */
struct saved_lock
{
	char name[65];
	char lkid_string[10];
	uint32_t lkid;
	struct timeval locktime;
};

static int nondlm = 0;
static int printed = 0;
static int packets = 0;
static int locks = 0;
static struct timeval starttime;

static char *lkmode(int mode)
{
	switch (mode) {
	case -1:
		return "IV";
		break;
	case 0:
		return "NL";
		break;
	case 2:
		return "CR";
		break;
	case 3:
		return "CW";
		break;
	case 4:
		return "PR";
		break;
	case 5:
		return "PW";
		break;
	case 6:
		return "EX";
		break;
	}
	return "  ";
}

void finish_up(int sig)
{
	struct timeval now;
	int secs;
	float lockss;

	fflush(stdout);

	gettimeofday(&now, NULL);
	secs = (int)(now.tv_sec - starttime.tv_sec);
	lockss = (float)locks/secs;

	printf("\nSeen %d packets, %d non-dlm, %d printed\n", packets, nondlm, printed);
	printf("%d locks in %d seconds, %.2f locks/s\n", locks, secs, lockss);
	exit(0);
}

/* Assume this is a DLM packet! */
void my_callback(u_char *useless,const struct pcap_pkthdr* pkthdr,
		 const u_char* packet)
{
	struct dlm_message *req;
	char lockname[65];
	int namelen;
	unsigned int lock_type;
	unsigned int inode_num;
	static int last_pid = 0;
	static char cmd[1024] = {'\0'};
	struct saved_lock *linfo;
	ENTRY hentry;
	int offset = 0x42;
	struct timeval now;
	struct tm* tmp;
	int i;


	packets++;
	while (offset < pkthdr->caplen) {
		req = (struct dlm_message *)((char *)packet+offset);

		/* Non-dlm packet */
		if (req->m_version != DLM_VERSION) {
			nondlm++;
			return;
		}

		if (req->m_cmd == DLM_MSG_LOCK ||
		    req->m_cmd == DLM_MSG_CONVERT) {
			locks++;

			if (!req->m_pid)
				cmd[0] = '\0';
			if (req->m_pid && last_pid != req->m_pid) {
				char name[1024];
				FILE *f;
				sprintf(name, "/proc/%d/cmdline", req->m_pid);
				f = fopen(name, "r");
				if (f) {
					fgets(cmd, sizeof(cmd), f);

					/*full command line*/
					for(i=0;i<sizeof(cmd);i++)
					  if ((cmd[i]==0)&&
					      (cmd[i+1]!=0))
					    cmd[i]=' ';

					fclose(f);
				}
				last_pid = req->m_pid;
			}


			namelen = req->m_length - sizeof(*req) + 1;
			memcpy(lockname, req->m_name, namelen);
			lockname[namelen] = 0;
			lockname[8] = 0;// Lock type

			lock_type = atoi(lockname);
			inode_num = atoi(lockname+9);

			gettimeofday(&now,NULL);
			tmp=localtime(&(now.tv_sec));

			printf("%i-%i-%i %02i:%02i:%02i   %s %s for pid %8d   lkid %8x  ",
			       tmp->tm_year+1900,tmp->tm_mon+1,tmp->tm_mday,tmp->tm_hour,tmp->tm_min,tmp->tm_sec, (req->m_cmd == 2)?"LOCK":"CONV", lkmode(req->m_rqmode),
			       req->m_pid, req->m_lkid);
			switch (lock_type) {
			case 1:
				printf("nondisk(1) = %8d\n", inode_num);
				break;
			case 2:
				printf("inode(2) = %8d %s\n", inode_num, cmd);
				break;
			case 3:
				printf("rgrp(3)  = %8d %s\n", inode_num, cmd);
				break;
			case 4:
				printf("meta(4)  = %8d %s\n", inode_num, cmd);
				break;
			case 5:
				printf("iopen(5) = %8d %s\n", inode_num, cmd);
				break;
			case 6:
				printf("flock(6) = %8d %s\n", inode_num, cmd);
				break;
			case 7:
				printf("plock(7) = %8d %s\n", inode_num, cmd);
				break;
			case 8:
				printf("quota(8) = %8d %s\n", inode_num, cmd);
				break;
			default:
				lockname[8] = ' ';
				printf("'%s'\n", lockname);
			}
			linfo = malloc(sizeof(struct saved_lock));
			if (linfo) {
				sprintf(linfo->lkid_string, "%x", req->m_lkid);

				lockname[8] = ' ';
				memcpy(linfo->name, lockname, namelen+1);
				linfo->lkid = req->m_lkid;
				gettimeofday(&linfo->locktime, NULL);
				hentry.key = linfo->lkid_string;
				hentry.data = linfo;
				hsearch(hentry, ENTER);
			}
			printed++;
		}

		if (req->m_cmd == DLM_MSG_UNLOCK) {
			char lkid_string[32];
			ENTRY *found_entry;

			sprintf(lkid_string, "%x", req->m_lkid);
			hentry.key = lkid_string;
			hentry.data = NULL;

			found_entry = hsearch(hentry, FIND);
			if (found_entry) {
				struct timeval now;
				int interval;

				linfo = found_entry->data;
				gettimeofday(&now, NULL);
				interval = now.tv_sec - linfo->locktime.tv_sec;

				tmp=localtime(&(now.tv_sec));

				printf("%i-%i-%i %02i:%02i:%02i   UNLOCK for lkid %x, held %d seconds. '%s'\n",
				       tmp->tm_year+1900,tmp->tm_mon+1,tmp->tm_mday,tmp->tm_hour,tmp->tm_min,tmp->tm_sec,
				       req->m_lkid, interval, linfo->name);
			}
			printed++;
		}
#if 0
		if (req->m_cmd == 5) { // lockreply
			if (req->m_status) {
				printf("    ERROR Return %d for lkid %x\n", req->m_status, req->m_remlkid);
			}
		}
#endif
		if (req->m_length <= 0) {
			printf(" ERROR, ignoring packet with illegal length %d\n", req->m_length);
			return;
		}
		offset += req->m_length;
	}
}

int main(int argc,char **argv)
{
    char *dev;
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_t* descr;
    struct bpf_program fp;      /* hold compiled program     */
    bpf_u_int32 maskp;          /* subnet mask               */
    bpf_u_int32 netp;           /* ip                        */

    signal(SIGINT, finish_up);

    /* has table for lock info */
    hcreate(100000);

    /* grab a device to peak into... */
    dev = pcap_lookupdev(errbuf);
    if(dev == NULL)
    { fprintf(stderr,"%s\n",errbuf); exit(1); }

    /* ask pcap for the network address and mask of the device */
    pcap_lookupnet(dev,&netp,&maskp,errbuf);

    /* open device for reading this time lets set it in promiscuous
     * mode so we can monitor traffic to another machine             */
    descr = pcap_open_live(dev,65536,1,-1,errbuf);
    if(descr == NULL)
    { printf("pcap_open_live(): %s\n",errbuf); exit(1); }

    /* Lets try and compile the program.. non-optimized */
    if(pcap_compile(descr,&fp,dlm_filter,0,-1/*netp*/) == -1)
    { fprintf(stderr,"Error calling pcap_compile\n"); exit(1); }

    /* set the compiled program as the filter */
    if(pcap_setfilter(descr,&fp) == -1)
    { fprintf(stderr,"Error setting filter\n"); exit(1); }

    /* ... and loop */
	gettimeofday(&starttime, NULL);
    fprintf(stderr, "Listening for DLM packets ...\n");
    pcap_loop(descr,-1,my_callback,NULL);

    return 0;
}
