/* -*- mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 Copyright (C) 2008 Ferdinando Ametrano

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it
 under the terms of the QuantLib license.  You should have received a
 copy of the license along with this program; if not, please email
 <quantlib-dev@lists.sf.net>. The license is also available online at
 <http://quantlib.org/license.shtml>.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

/*! \file sensitivityanalysis.hpp
    \brief sensitivity analysis function
*/

#ifndef quantlib_sensitivity_analysis_hpp
#define quantlib_sensitivity_analysis_hpp

#include <ql/types.hpp>
#include <ql/utilities/null.hpp>
#include <boost/shared_ptr.hpp>
#include <vector>

namespace QuantLib {

    template <class T>
    class Handle;
    class SimpleQuote;
    class Instrument;

    //! Finite differences calculation
    enum SensitivityAnalysis {
        OneSide, /*!<  */
        Centered /*!<  */
    };

    /*! \relates SensitivityAnalysis */
    std::ostream& operator<<(std::ostream&,
                             SensitivityAnalysis);

    //! utility fuction for weighted sum of NPVs
    Real aggregateNPV(const std::vector<boost::shared_ptr<Instrument> >&,
                      const std::vector<Real>& quantities);

    //! parallel shift sensitivity analysis for a SimpleQuote vector
    /*! returns a pair of first and second derivative values calculated as
        prescribed by SensitivityAnalysis. Second derivative might not be
        available depending on SensitivityAnalysis value.

        Empty quantities vector is considered as unit vector. The same if
        the vector is just one single element equal to one.

        All SimpleQuotes are tweaked together in a parallel fashion.
    */
    std::pair<Real, Real>
    parallelAnalysis(const std::vector<Handle<SimpleQuote> >&,
                     const std::vector<boost::shared_ptr<Instrument> >&,
                     const std::vector<Real>& quantities,
                     Real shift = 0.0001,
                     SensitivityAnalysis type = Centered,
                     Real referenceNpv = Null<Real>());

    //! parallel shift sensitivity analysis for a SimpleQuote matrix
    /*! returns a pair of first and second derivative values calculated as
        prescribed by SensitivityAnalysis. Second derivative might not be
        available depending on SensitivityAnalysis value.

        Empty quantities vector is considered as unit vector. The same if
        the vector is of size one.

        All SimpleQuotes are tweaked together in a parallel fashion.
    */
    std::pair<Real, Real>
    parallelAnalysis(const std::vector<std::vector<Handle<SimpleQuote> > >&,
                     const std::vector<boost::shared_ptr<Instrument> >&,
                     const std::vector<Real>& quantities,
                     Real shift = 0.0001,
                     SensitivityAnalysis type = Centered,
                     Real referenceNpv = Null<Real>());

    //! (bucket) sensitivity analysis for a (single) SimpleQuote
    /*! returns a pair of first and second derivative values calculated as
        prescribed by SensitivityAnalysis. Second derivative might not be
        available depending on SensitivityAnalysis value.

        Empty quantities vector is considered as unit vector. The same if
        the vector is of size one.
    */
    std::pair<Real, Real>
    bucketAnalysis(Handle<SimpleQuote> quote,
                   const std::vector<boost::shared_ptr<Instrument> >&,
                   const std::vector<Real>& quantities,
                   Real shift = 0.0001,
                   SensitivityAnalysis type = Centered,
                   Real referenceNpv = Null<Real>());

    //! bucket sensitivity analysis for a SimpleQuote vector
    /*! returns a pair of first and second derivative vectors calculated as
        prescribed by SensitivityAnalysis. Second derivative might not be
        available depending on SensitivityAnalysis value.

        Empty quantities vector is considered as unit vector. The same if
        the vector is of size one.

        The (bucket) SimpleQuotes are tweaked one by one separately.
    */
    std::pair<std::vector<Real>, std::vector<Real> >
    bucketAnalysis(const std::vector<Handle<SimpleQuote> >&,
                   const std::vector<boost::shared_ptr<Instrument> >&,
                   const std::vector<Real>& quantities,
                   Real shift = 0.0001,
                   SensitivityAnalysis type = Centered);

    //! bucket sensitivity analysis for a SimpleQuote matrix
    /*! returns a pair of first and second derivative metrices calculated as
        prescribed by SensitivityAnalysis. Second derivative might not be
        available depending on SensitivityAnalysis value.

        Empty quantities vector is considered as unit vector. The same if
        the vector is of size one.

        The (bucket) SimpleQuotes are tweaked one by one separately.
    */
    std::pair<std::vector<std::vector<Real> >, std::vector<std::vector<Real> > >
    bucketAnalysis(const std::vector<std::vector<Handle<SimpleQuote> > >&,
                   const std::vector<boost::shared_ptr<Instrument> >&,
                   const std::vector<Real>& quantities,
                   Real shift = 0.0001,
                   SensitivityAnalysis type = Centered);

}

#endif
