//
// C++ Implementation:
//
// Description:
//
//
// Author: Sebastian Holtermann <sebholt@xwmw.org>, (C) 2011
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "controls_view.hpp"

#include <mwdg/event_types.hpp>
#include <mwdg/controls_view_setup.hpp>

#include <QCoreApplication>
#include <QScrollBar>

#include <iostream>


namespace MWdg
{


Controls_View::Controls_View (
	QWidget * parent ) :
QTreeView ( parent ),
_view_setup ( 0 ),
_show_rows_min ( 3 ),
_show_rows_avrg ( 10 ),
_min_chars_vis ( 18 ),
_maximum_update_requested ( false ),
_restoring_state ( false )
{
	setHeaderHidden ( true );
	{
		QSizePolicy policy ( sizePolicy() );
		policy.setVerticalPolicy ( QSizePolicy::Expanding );
		setSizePolicy ( policy );
	}

	connect ( this, SIGNAL ( expanded ( const QModelIndex & ) ),
		this, SLOT ( item_expanded() ) );

	connect ( this, SIGNAL ( collapsed ( const QModelIndex & ) ),
		this, SLOT ( item_expanded() ) );
}


Controls_View::~Controls_View ( )
{
}


QSize
Controls_View::minimumSizeHint ( ) const
{
	ensurePolished();

	QSize res;
	int & ww ( res.rwidth() );
	int & hh ( res.rheight() );

	QSize vsb_msh ( verticalScrollBar()->minimumSizeHint() );
	QSize vsb_sh ( verticalScrollBar()->sizeHint() );

	// Width
	{
		ww = fontMetrics().averageCharWidth() * _min_chars_vis;
		ww = qMax ( 16, ww );
		ww += indentation();
		if ( vsb_msh.width() > 0 ) {
			ww += vsb_msh.width();
		} else if ( vsb_sh.width() > 0 ) {
			ww += vsb_sh.width();
		}
	}


	// Height
	{
		int rh0 ( sizeHintForRow ( 0 ) );
		rh0 = qMax ( rh0,
			itemDelegate()->sizeHint ( viewOptions(), QModelIndex() ).height() );
		hh = rh0 * _show_rows_min;
		hh = qMax ( vsb_msh.height(), hh );
	}

	QMargins mg ( contentsMargins() );
	ww += mg.left() + mg.right();
	hh += mg.top() + mg.bottom();

	return res;
}


QSize
Controls_View::sizeHint ( ) const
{
	ensurePolished();

	QSize res;
	int & ww ( res.rwidth() );
	int & hh ( res.rheight() );

	QSize vsb_sh ( verticalScrollBar()->sizeHint() );

	// Width
	{
		ww = fontMetrics().averageCharWidth() * _min_chars_vis;
		ww += indentation();
		if ( vsb_sh.width() > 0 ) {
			ww += vsb_sh.width();
		}
	}

	// Height
	{
		int rh0 ( sizeHintForRow ( 0 ) );
		rh0 = qMax ( rh0,
			itemDelegate()->sizeHint ( viewOptions(), QModelIndex() ).height() );
		hh = rh0 * _show_rows_avrg;
		hh = qMax ( vsb_sh.height(), hh );
	}

	QMargins mg ( contentsMargins() );
	ww += mg.left() + mg.right();
	hh += mg.top() + mg.bottom();

	return res;
}


int
Controls_View::sizeHintForColumn (
	int column_n ) const
{
	int ww ( 16 );
	ww = qMax ( fontMetrics().averageCharWidth() * _min_chars_vis, ww );
	ww = qMax ( QTreeView::sizeHintForColumn ( column_n ), ww );
	return ww;
}


void
Controls_View::set_view_setup (
	::MWdg::Controls_View_Setup * setup_n )
{
	_view_setup = setup_n;

	if ( _view_setup != 0 ) {
		restore_state();
	}
}


void
Controls_View::setModel (
	QAbstractItemModel * model_n )
{
	if ( model() != 0 ) {
		disconnect ( model(), 0, this, 0 );
	}

	QTreeView::setModel ( model_n );

	if ( model() != 0 ) {
		maximum_height_update_request();
		restore_state();

		connect ( model(), SIGNAL ( modelReset() ),
			this, SLOT ( restore_state() ) );

		connect ( model(), SIGNAL ( layoutChanged() ),
			this, SLOT ( maximum_height_update_request() ) );

		connect ( model(), SIGNAL ( rowsInserted ( const QModelIndex &, int , int )  ),
			this, SLOT ( maximum_height_update_request() ) );

		connect ( model(), SIGNAL ( rowsRemoved ( const QModelIndex &, int , int )  ),
			this, SLOT ( maximum_height_update_request() ) );
	}
}


void
Controls_View::save_state ( )
{
	if ( ( model() == 0 ) || ( _view_setup == 0 ) ) {
		return;
	}

	_view_setup->expanded_items.clear();

	const unsigned int num_rows ( model()->rowCount() );
	for ( unsigned int ii=0; ii < num_rows; ++ii ) {
		const QModelIndex idx ( model()->index ( ii, 0 ) );
		if ( isExpanded ( idx ) ) {
			_view_setup->expanded_items.append (
				idx.data ( Qt::DisplayRole ).toString() );
		}
	}
}


void
Controls_View::restore_state ( )
{
	if ( ( model() == 0 ) || ( _view_setup == 0 ) ) {
		return;
	}
	_restoring_state = true;

	const unsigned int num_rows ( model()->rowCount() );
	for ( unsigned int ii=0; ii < num_rows; ++ii ) {
		const QModelIndex idx ( model()->index ( ii, 0 ) );
		const QString txt ( idx.data ( Qt::DisplayRole ).toString() );
		if ( _view_setup->expanded_items.contains ( txt ) ) {
			expand ( idx );
		}
	}

	_restoring_state = false;
}


void
Controls_View::currentChanged (
	const QModelIndex & current,
	const QModelIndex & previous )
{
	QTreeView::currentChanged ( current, previous );
	emit activated ( current );
}


void
Controls_View::item_expanded ( )
{
	if ( !_restoring_state ) {
		save_state();
	}
}


void
Controls_View::maximum_height_update_request ( )
{
	if ( !_maximum_update_requested ) {
		_maximum_update_requested = true;

		QCoreApplication::postEvent (
			this, new QEvent ( ::MWdg::evt_refresh_data ) );
	}
}


bool
Controls_View::event (
	QEvent * event_n )
{
	bool res ( false );
	if ( event_n->type() == ::MWdg::evt_refresh_data ) {
		_maximum_update_requested = false;
		maximum_height_update();
		res = true;
	} else {
		res = QTreeView::event ( event_n );
	}
	return res;
}


void
Controls_View::maximum_height_update ( )
{
	unsigned int hsum ( 16000 );
	if ( model() != 0 ) {
		hsum = 0;
		hsum += frameWidth()*2;
		hsum += height_sum_recursive ( QModelIndex() );

		{
			const QModelIndex idx_cur ( model()->index ( 0, 0 ) );
			const int hrow ( rowHeight ( idx_cur ) );
			if ( hrow > 0 ) {
				hsum += hrow;
			}
		}
	}
	//std::cout << "Controls_View::maximum_height_update " << hsum << "\n";

	setMaximumHeight ( hsum );
	updateGeometry();
}


unsigned int
Controls_View::height_sum_recursive (
	const QModelIndex & idx_n )
{
	unsigned int hsum ( 0 );
	for ( int ii=0; ii < model()->rowCount ( idx_n ); ++ii ) {
		const QModelIndex idx_cur ( model()->index ( ii, 0, idx_n ) );
		int hrow ( rowHeight ( idx_cur ) );
		if ( hrow <= 0 ) {
			hrow = rowHeight ( model()->index ( 0, 0 ) );
		}
		if ( hrow > 0 ) {
			hsum += hrow;
		}
		hsum += height_sum_recursive ( idx_cur );
	}
	return hsum;
}


} // End of namespace
