/***************************************************************************
 *   Copyright (C) 2005 by Niklas Knutsson                                 *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "qalculateeditfunctiondialog.h"
#include "qalculate_kde_utils.h"
#include "qalculateeditnamesdialog.h"

#include <qgrid.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kcombobox.h>
#include <qlabel.h>
#include <klocale.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qcheckbox.h>
#include <kmessagebox.h>
#include <qlayout.h>
#include <kdeversion.h>
#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 1
#include <qtextedit.h>
#else
#include <ktextedit.h>
#endif
#include <klistview.h>
#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 2
#include <qtabwidget.h>
#else
#include <ktabwidget.h>
#endif
#include <qheader.h>
#include <kapplication.h>
#include <kstdguiitem.h>

extern PrintOptions printops;
extern EvaluationOptions evalops;
extern tree_struct function_cats;

QalculateEditFunctionDialog::QalculateEditFunctionDialog(QWidget *parent, const char *name) : KDialogBase(parent, name, true, i18n("Edit Function"), Ok | Cancel | Help, Ok, false) {

	names_edit_dialog = NULL;
	subfunctions_edit_dialog = NULL;
	edited_function = NULL;
	argument_edit_dialog = NULL;

#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 2
	tabs = new QTabWidget(this);
#else
	tabs = new KTabWidget(this);
#endif
	setMainWidget(tabs);
	QWidget *page1 = new QWidget(this);
	QWidget *page2 = new QWidget(this);
	tabs->addTab(page1, i18n("General"));
	tabs->addTab(page2, i18n("Function"));
	QGridLayout *grid = new QGridLayout(page1, 1, 1, spacingHint());
	QVBoxLayout *vbox = new QVBoxLayout(page2, spacingHint());
	grid->addWidget(new QLabel(i18n("Name:"), page1), 0, 0);
	nameEdit = new KLineEdit(page1);
	grid->addWidget(nameEdit, 0, 1);
	QHBoxLayout *hbox = new QHBoxLayout(0, 0, spacingHint());
	grid->addMultiCellLayout(hbox, 1, 1, 0, 1);
	namesLabel = new QLabel(page1);
	namesLabel->setAlignment(Qt::AlignRight);
	namesLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
	hbox->addWidget(namesLabel);
	namesButton = new QPushButton(i18n("Names Details"), page1);
	hbox->addWidget(namesButton);
	grid->addWidget(new QLabel(i18n("Category:"), page1), 2, 0);
	categoryCombo = new KComboBox(page1);
	categoryCombo->setMaximumWidth(250);
	categoryCombo->setEditable(true);
	grid->addWidget(categoryCombo, 2, 1);
	grid->addWidget(new QLabel(i18n("Descriptive name:"), page1), 3, 0);
	titleEdit = new KLineEdit(page1);
	grid->addWidget(titleEdit, 3, 1);
	hbox = new QHBoxLayout(0, 0, spacingHint());
	grid->addMultiCellLayout(hbox, 4, 4, 0, 1);
	hbox->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	hideButton = new QCheckBox(i18n("Hide function"), page1);
	hbox->addWidget(hideButton);
	grid->addWidget(new QLabel(i18n("Description:"), page1), 5, 0);	
#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 1
	descriptionEdit = new QTextEdit(page1);
#else
	descriptionEdit = new KTextEdit(page1);
#endif
	descriptionEdit->setTextFormat(QTextEdit::PlainText);
	grid->addMultiCellWidget(descriptionEdit, 6, 6, 0, 1);
	vbox->addWidget(new QLabel(i18n("Expression:"), page2));
#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 1
	expressionEdit = new QTextEdit(page2);
#else	
	expressionEdit = new KTextEdit(page2);
#endif
	expressionEdit->setTextFormat(QTextEdit::PlainText);
	vbox->addWidget(expressionEdit);
	hbox = new QHBoxLayout(vbox, spacingHint());
	hbox->addWidget(new QLabel(i18n("Condition:"), page2));
	conditionEdit = new KLineEdit(page2);
	hbox->addWidget(conditionEdit);
	hbox = new QHBoxLayout(vbox, spacingHint());
	hbox->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	subFunctionsButton = new QPushButton(i18n("Sub-Functions"), page2);
	hbox->addWidget(subFunctionsButton);
	vbox->addWidget(new QLabel(i18n("Arguments:"), page2));
	argumentsView = new KListView(page2);
	argumentsView->addColumn(i18n("Name"));
	argumentsView->addColumn(i18n("Type"));
	argumentsView->setSorting(-1);
	argumentsView->setRootIsDecorated(false);
	argumentsView->setItemsRenameable(true);
	argumentsView->setItemsMovable(false);
	vbox->addWidget(argumentsView);
	hbox = new QHBoxLayout(vbox, spacingHint());
	hbox->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	newArgumentButton = new QPushButton(i18n("New"), page2);
	hbox->addWidget(newArgumentButton);
	editArgumentButton = new QPushButton(i18n("Edit"), page2);
	hbox->addWidget(editArgumentButton);
	deleteArgumentButton = new QPushButton(i18n("Delete"), page2);
	hbox->addWidget(deleteArgumentButton);

	connect(namesButton, SIGNAL(clicked()), this, SLOT(editNames()));
	connect(subFunctionsButton, SIGNAL(clicked()), this, SLOT(editSubFunctions()));
	connect(nameEdit, SIGNAL(textChanged(const QString&)), this, SLOT(nameChanged(const QString&)));
	connect(newArgumentButton, SIGNAL(clicked()), this, SLOT(newArgument()));
	connect(editArgumentButton, SIGNAL(clicked()), this, SLOT(editArgument()));
	connect(deleteArgumentButton, SIGNAL(clicked()), this, SLOT(deleteArgument()));
	connect(argumentsView, SIGNAL(itemRenamed(QListViewItem*, const QString&, int)), this, SLOT(argumentRenamed(QListViewItem*, const QString&, int)));
	connect(argumentsView, SIGNAL(selectionChanged()), this, SLOT(argumentSelectionChanged()));
	
}

QalculateEditFunctionDialog::~QalculateEditFunctionDialog() {}


void QalculateEditFunctionDialog::slotHelp() {
	KApplication::kApplication()->invokeHelp("qalculate-function-creation");
}

void QalculateEditFunctionDialog::argumentSelectionChanged() {
	QListViewItem *i = argumentsView->selectedItem();
	if(i == NULL || (edited_function && edited_function->isBuiltin()) || argumentsItems[i]->type() > ARGUMENT_TYPE_ANGLE) {
		editArgumentButton->setEnabled(false);
		deleteArgumentButton->setEnabled(false);
	} else {
		editArgumentButton->setEnabled(true);
		deleteArgumentButton->setEnabled(true);
	}
}
void QalculateEditFunctionDialog::argumentRenamed(QListViewItem *i, const QString &str, int col) {
	if(!i || col != 0) return;
	if(col == 0) {
		if(!argumentsItemsEdited[i]) {
			argumentsItems[i] = argumentsItems[i]->copy();
			argumentsItemsEdited[i] = true;
		}
		argumentsItems[i]->setName(str.ascii());
	}
}

void QalculateEditFunctionDialog::slotOk() {

	string str = nameEdit->text().ascii();
	remove_blank_ends(str);
	string str2 = CALCULATOR->unlocalizeExpression(expressionEdit->text().ascii(), evalops.parse_options);
	remove_blank_ends(str2);
	gsub("\n", " ", str2);
	if(str.empty() && (!names_edit_dialog || names_edit_dialog->isEmpty())) {
		//no name -- open dialog again
		tabs->setCurrentPage(0);
		nameEdit->setFocus();
		KMessageBox::error(this, i18n("Empty name field"));
		return;
	}
	if(!(edited_function && edited_function->isBuiltin()) && str2.empty()) {
		//no expression/relation -- open dialog again		
		tabs->setCurrentPage(1);
		expressionEdit->setFocus();
		KMessageBox::error(this, i18n("Empty expression field"));
		return;
	}
	//function with the same name exists -- overwrite or open dialog again
	if((!edited_function || !edited_function->hasName(str)) && (!names_edit_dialog || names_edit_dialog->isEmpty()) && CALCULATOR->functionNameTaken(str, edited_function) && KMessageBox::questionYesNo(this, i18n("A function with the same name already exists.\nDo you want to overwrite it?")) == KMessageBox::No) {
		tabs->setCurrentPage(0);
		nameEdit->setFocus();
		return;
	}
	
	MathFunction *f = edited_function;

	bool add_func = false;
	if(f) {
		f->setLocal(true);
		//edited an existing function
		f->setCategory(categoryCombo->lineEdit()->text().ascii());
		f->setTitle(titleEdit->text().ascii());
		f->setDescription(descriptionEdit->text().ascii());
	} else {
		//new function
		f = new UserFunction(categoryCombo->lineEdit()->text().ascii(), "", "", true, -1, titleEdit->text().ascii(), descriptionEdit->text().ascii());
		add_func = true;
	}
	f->setCondition(CALCULATOR->unlocalizeExpression(conditionEdit->text().ascii(), evalops.parse_options));
	size_t index = 1;
	QListViewItemIterator it(argumentsView);
	while(it.current()) {
		if(argumentsItemsEdited[it.current()]) {
			if(f->isBuiltin() && f->getArgumentDefinition(index)) {
				f->getArgumentDefinition(index)->setName(argumentsItems[it.current()]->name());
				delete argumentsItems[it.current()];
			} else {
				f->setArgumentDefinition(index, argumentsItems[it.current()]);
			}
		}
		++it;
		index++;
	}
	if(!f->isBuiltin()) {
		for(; index <= f->lastArgumentDefinitionIndex(); index++) {
			f->setArgumentDefinition(index, NULL);
		}
	}
	argumentsItemsEdited.clear();
	argumentsItems.clear();
	f->setHidden(hideButton->isChecked());
	if(!f->isBuiltin()) {
		if(names_edit_dialog) {
			names_edit_dialog->saveNames(f, str.c_str());
		} else {
			if(f->countNames() == 0) {
				ExpressionName ename(str);
				ename.reference = true;
				f->setName(ename, 1);
			} else {
				f->setName(str, 1);
			}
		}
		if(subfunctions_edit_dialog) {
			subfunctions_edit_dialog->saveSubFunctions((UserFunction*) f);
		}
		((UserFunction*) f)->setFormula(str2);
	}
	if(add_func) {
		CALCULATOR->addFunction(f);
	}
	edited_function = f;
	
	accept();

}

MathFunction *QalculateEditFunctionDialog::editFunction(QString category, MathFunction *f) {

	nameEdit->setFocus();

	tabs->setCurrentPage(0);
	if(names_edit_dialog) {
		delete names_edit_dialog;
		names_edit_dialog = NULL;
	}
	if(subfunctions_edit_dialog) {
		delete subfunctions_edit_dialog;
		subfunctions_edit_dialog = NULL;
	}
	edited_function = f;
	if(f) {
		if(f->isLocal())
			setCaption(i18n("Edit Function"));
		else
			setCaption(i18n("Edit Function (global)"));
	} else {
		setCaption(i18n("New Function"));
	}

	QString catstr;
	categoryCombo->clear();
	tree_struct *item, *item2;
	function_cats.it = function_cats.items.begin();
	if(function_cats.it != function_cats.items.end()) {
		item = &*function_cats.it;
		++function_cats.it;
		item->it = item->items.begin();
	} else {
		item = NULL;
	}

	catstr = "";
	while(item) {
		if(!catstr.isEmpty()) catstr += "/";
		catstr += item->item.c_str();
		categoryCombo->insertItem(catstr);
		while(item && item->it == item->items.end()) {
			int str_i = catstr.findRev("/");
			if(str_i < 0) {
				catstr = "";
			} else {
				catstr.truncate(str_i);
			}
			item = item->parent;
		}
		if(item) {
			item2 = &*item->it;
			++item->it;
			item = item2;
			item->it = item->items.begin();	
		}
	}

	//clear entries
	nameEdit->clear();
	namesLabel->clear();
	conditionEdit->clear();
	nameEdit->setReadOnly(f && f->isBuiltin());
	expressionEdit->setEnabled(!f || !f->isBuiltin());
	categoryCombo->lineEdit()->setText(category);
	titleEdit->clear();
	descriptionEdit->clear();
	expressionEdit->clear();
	hideButton->setChecked(false);
	actionButton(Ok)->setEnabled(true);
	subFunctionsButton->setEnabled(!f || !f->isBuiltin());
	conditionEdit->setReadOnly(f && f->isBuiltin());
	newArgumentButton->setEnabled(!f || !f->isBuiltin());
	if(f) {
		//fill in original paramaters
		set_name_label_and_entry(f, nameEdit, namesLabel);
		if(!f->isBuiltin()) {
			expressionEdit->setText(CALCULATOR->localizeExpression(((UserFunction*) f)->formula()).c_str());
		}
		categoryCombo->lineEdit()->setText(f->category().c_str());
		titleEdit->setText(f->title(false).c_str());
		conditionEdit->setText(CALCULATOR->localizeExpression(f->condition()).c_str());
		hideButton->setChecked(f->isHidden());
		descriptionEdit->setText(f->description().c_str());
		
	}
	argumentsView->clear();
	argumentsItems.clear();
	argumentsItemsEdited.clear();
	editArgumentButton->setEnabled(false);
	deleteArgumentButton->setEnabled(false);
	Argument defarg;
	if(f) {
		Argument *arg;
		int args = f->maxargs();
		if(args < 0) {
			args = f->minargs() + 1;	
		}
		QString str, str2;
		QListViewItem *i_prev = NULL;
		for(int i = 1; i <= args; i++) {
			arg = f->getArgumentDefinition(i);
			if(arg) {
				arg = arg->copy();
				str = arg->printlong().c_str();
				str2 = arg->name().c_str();
			} else {
				arg = &defarg;
				str = defarg.printlong().c_str();
				str2 = "";
			}
			QListViewItem *i = new KListViewItem(argumentsView, i_prev, str2, str);
			i_prev = i;
			argumentsItems[i] = arg;
			argumentsItemsEdited[i] = false;
		}
	}
	argumentsView->setResizeMode(KListView::AllColumns);

	if(exec() == QDialog::Accepted) {
		f = edited_function;
		edited_function = NULL;
		return f;
	}
	QListViewItemIterator it(argumentsView);
	while(it.current()) {
		if(argumentsItemsEdited[it.current()]) {
			delete argumentsItems[it.current()];
		}
		++it;
	}
	argumentsItemsEdited.clear();
	argumentsItems.clear();
	edited_function = NULL;
	return NULL;
}

void QalculateEditFunctionDialog::editNames() {
	if(!names_edit_dialog) {
		names_edit_dialog = new QalculateEditNamesDialog(TYPE_FUNCTION, this);
		names_edit_dialog->setNames(edited_function, nameEdit->text(), false);
	} else {
		names_edit_dialog->setNames(edited_function, nameEdit->text(), true);
	}
	names_edit_dialog->exec();
	names_edit_dialog->setNamesLE(nameEdit, namesLabel);
}
void QalculateEditFunctionDialog::newArgument() {
	if(!argument_edit_dialog) {
		argument_edit_dialog = new QalculateEditArgumentDialog(this);
	}
	Argument *arg = argument_edit_dialog->editArgument();
	if(arg) {
		QListViewItem *i = new KListViewItem(argumentsView, argumentsView->lastChild(), arg->name().c_str(), arg->printlong().c_str());
		argumentsItems[i] = arg;
		argumentsItemsEdited[i] = true;
	}
}
void QalculateEditFunctionDialog::editArgument() {
	QListViewItem *i = argumentsView->selectedItem();
	if(i == NULL) return;
	if(!argument_edit_dialog) {
		argument_edit_dialog = new QalculateEditArgumentDialog(this);
	}
	Argument *arg = argument_edit_dialog->editArgument(argumentsItems[i]);
	if(arg) {
		if(argumentsItemsEdited[i]) {
			delete argumentsItems[i];
		}
		argumentsItemsEdited[i] = true;
		argumentsItems[i] = arg;
		i->setText(0, arg->name().c_str());
		i->setText(1, arg->printlong().c_str());
	}
}
void QalculateEditFunctionDialog::deleteArgument() {
	QListViewItem *i = argumentsView->selectedItem();
	if(i == NULL) return;
	if(argumentsItemsEdited[i]) {
		delete argumentsItems[i];
	}
	argumentsItemsEdited.erase(i);
	argumentsItems.erase(i);
	argumentsView->removeItem(i);
}
void QalculateEditFunctionDialog::editSubFunctions() {
	if(edited_function && edited_function->isBuiltin()) return;
	if(!subfunctions_edit_dialog) {
		subfunctions_edit_dialog = new QalculateEditSubFunctionsDialog(this);
		subfunctions_edit_dialog->setSubFunctions((UserFunction*) edited_function, false);
	} else {
		subfunctions_edit_dialog->setSubFunctions((UserFunction*) edited_function, true);
	}
	subfunctions_edit_dialog->exec();
}

void QalculateEditFunctionDialog::nameChanged(const QString &name) {
	if(name.isNull()) return;
	if(!CALCULATOR->functionNameIsValid(name.ascii())) {
		nameEdit->blockSignals(true);
		nameEdit->setText(CALCULATOR->convertToValidFunctionName(name.ascii()).c_str());
		nameEdit->blockSignals(false);
	}
}

QalculateEditSubFunctionsDialog::QalculateEditSubFunctionsDialog(QWidget *parent, const char *name) : KDialog(parent, name, true) {

	setCaption(i18n("Sub-Functions"));

	QVBoxLayout *layout = new QVBoxLayout(this, marginHint(), spacingHint());

	subFunctionsView = new KListView(this);
	subFunctionsView->addColumn(i18n("Reference"));
	subFunctionsView->addColumn(i18n("Expression"));
	subFunctionsView->addColumn(i18n("Precalculate"));
	subFunctionsView->setSorting(-1);
	subFunctionsView->setItemsRenameable(true);
	subFunctionsView->setRenameable(0, false);
	subFunctionsView->setRenameable(1, true);
	subFunctionsView->setItemsMovable(false);
	layout->addWidget(subFunctionsView);

	QHBoxLayout *buttonLayout = new QHBoxLayout(layout, spacingHint());
	newButton = new QPushButton(i18n("New"), this);
	buttonLayout->addWidget(newButton);
	deleteButton = new QPushButton(i18n("Delete"), this);
	deleteButton->setEnabled(false);
	buttonLayout->addWidget(deleteButton);
	buttonLayout->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	buttonClose = new KPushButton(KStdGuiItem::close(), this);
	buttonClose->setAutoDefault(true);
	buttonClose->setDefault(true);
	buttonLayout->addWidget(buttonClose);

	connect(subFunctionsView, SIGNAL(selectionChanged()), this, SLOT(selectionChanged()));
	connect(buttonClose, SIGNAL(clicked()), this, SLOT(accept()));
	connect(newButton, SIGNAL(clicked()), this, SLOT(newSubFunction()));
	connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteSubFunction()));
	connect(subFunctionsView, SIGNAL(clicked(QListViewItem*, const QPoint&, int)), this, SLOT(itemClicked(QListViewItem*, const QPoint&, int)));
}

QalculateEditSubFunctionsDialog::~QalculateEditSubFunctionsDialog() {}

void QalculateEditSubFunctionsDialog::newSubFunction() {
	ExpressionName name;
	QString str = "\\";
	str += QString::number(subFunctionsView->childCount() + 1);
	QListViewItem *i = new KListViewItem(subFunctionsView, subFunctionsView->lastChild(), str, "", i18n("Yes"));
	subFunctionsView->setSelected(i, true);
	qApp->processEvents();
	subFunctionsView->rename(i, 1);
}

void QalculateEditSubFunctionsDialog::setSubFunctions(UserFunction *f, bool edited) {
	if(!edited) {
		deleteButton->setEnabled(false);
		subFunctionsView->clear();
	}
	if(!edited && f && f->countSubfunctions() > 0) {
		QListViewItem *i_prev = NULL;
		for(size_t index = 1; index <= f->countSubfunctions(); index++) {
			QListViewItem *i = new KListViewItem(subFunctionsView, i_prev);
			i_prev = i;
			QString str = "\\";
			str += QString::number(index);
			i->setText(0, str);
			i->setText(1, f->getSubfunction(index).c_str());
			if(f->subfunctionPrecalculated(index)) i->setText(2, i18n("Yes"));
			else i->setText(2, i18n("No"));
		}
	}
}
void QalculateEditSubFunctionsDialog::saveSubFunctions(UserFunction *f) {
	f->clearSubfunctions();
	QListViewItemIterator it(subFunctionsView);
	while(it.current()) {
		f->addSubfunction(it.current()->text(1).ascii(), it.current()->text(2) == i18n("Yes"));
		++it;
	}
}
bool QalculateEditSubFunctionsDialog::isEmpty() {
	return subFunctionsView->childCount() <= 0;
}

void QalculateEditSubFunctionsDialog::deleteSubFunction() {
	subFunctionsView->removeItem(subFunctionsView->selectedItem());
	QListViewItemIterator it(subFunctionsView);
	int index = 1;
	while(it.current()) {
		QString str = "\\";
		str += QString::number(index);
		it.current()->setText(0, str);
		++it;
		index++;
	}
}

void QalculateEditSubFunctionsDialog::itemClicked(QListViewItem *i, const QPoint &, int c) {
	if(i && c == 2) {
		if(i->text(c) == i18n("Yes"))
			i->setText(c, i18n("No"));
		else
			i->setText(c, i18n("Yes"));
	}
}

void QalculateEditSubFunctionsDialog::selectionChanged() {
	deleteButton->setEnabled(subFunctionsView->selectedItem() != NULL);
}

QalculateEditArgumentDialog::QalculateEditArgumentDialog(QWidget *parent, const char *name) : KDialogBase(parent, name, true, i18n("Edit Argument"), Ok | Cancel, Ok, true) {

	setMainWidget(new QWidget(this));
	QGridLayout *grid = new QGridLayout(mainWidget(), 1, 1, 0, spacingHint());
	grid->addWidget(new QLabel(i18n("Name:"), mainWidget()), 0, 0);
	nameEdit = new KLineEdit(mainWidget());
	grid->addWidget(nameEdit, 0, 1);
	grid->addWidget(new QLabel(i18n("Type:"), mainWidget()), 1, 0);
	typeCombo = new KComboBox(mainWidget());
	typeCombo->setEditable(false);
	typeCombo->insertItem(i18n("Free"));
	typeCombo->insertItem(i18n("Number"));
	typeCombo->insertItem(i18n("Integer"));
	typeCombo->insertItem(i18n("Symbol"));
	typeCombo->insertItem(i18n("Text"));
	typeCombo->insertItem(i18n("Date"));
	typeCombo->insertItem(i18n("Vector"));
	typeCombo->insertItem(i18n("Matrix"));
	typeCombo->insertItem(i18n("Boolean"));
	typeCombo->insertItem(i18n("Object"));
	typeCombo->insertItem(i18n("Function"));
	typeCombo->insertItem(i18n("Unit"));
	typeCombo->insertItem(i18n("Variable"));
	typeCombo->insertItem(i18n("File"));
	typeCombo->insertItem(i18n("Angle"));
	grid->addWidget(typeCombo, 1, 1);
	testButton = new QCheckBox(i18n("Enable rules and type test"), mainWidget());
	grid->addMultiCellWidget(testButton, 2, 2, 0, 1);
	conditionButton = new QCheckBox(i18n("Custom condition:"), mainWidget());
	grid->addWidget(conditionButton, 3, 0);
	conditionEdit = new KLineEdit(mainWidget());
	grid->addWidget(conditionEdit, 3, 1);
	matrixButton = new QCheckBox(i18n("Allow matrix"), mainWidget());
	grid->addMultiCellWidget(matrixButton, 4, 4, 0, 1);
	zeroButton = new QCheckBox(i18n("Forbid zero"), mainWidget());
	grid->addMultiCellWidget(zeroButton, 5, 5, 0, 1);
	complexButton = new QCheckBox(i18n("Allow complex"), mainWidget());
	grid->addMultiCellWidget(complexButton, 6, 6, 0, 1);
	minButton = new QCheckBox(i18n("Min:"), mainWidget());
	grid->addWidget(minButton, 7, 0);
	minEdit = new KLineEdit(mainWidget());
	grid->addWidget(minEdit, 7, 1);
	QHBoxLayout *hbox = new QHBoxLayout(0, 0, spacingHint());
	grid->addMultiCellLayout(hbox, 8, 8, 0, 1);
	hbox->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	minIncludeEqualsButton = new QCheckBox(i18n("Include equals"), mainWidget());
	hbox->addWidget(minIncludeEqualsButton);
	maxButton = new QCheckBox(i18n("Max:"), mainWidget());
	grid->addWidget(maxButton, 9, 0);
	maxEdit = new KLineEdit(mainWidget());
	grid->addWidget(maxEdit, 9, 1);
	hbox = new QHBoxLayout(0, 0, spacingHint());
	grid->addMultiCellLayout(hbox, 10, 10, 0, 1);
	hbox->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	maxIncludeEqualsButton = new QCheckBox(i18n("Include equals"), mainWidget());
	hbox->addWidget(maxIncludeEqualsButton);

	connect(conditionButton, SIGNAL(toggled(bool)), conditionEdit, SLOT(setEnabled(bool)));
	connect(minButton, SIGNAL(toggled(bool)), this, SLOT(minEnabled(bool)));
	connect(maxButton, SIGNAL(toggled(bool)), this, SLOT(maxEnabled(bool)));
	connect(typeCombo, SIGNAL(activated(int)), this, SLOT(typeChanged(int)));
	
}

QalculateEditArgumentDialog::~QalculateEditArgumentDialog() {}

void QalculateEditArgumentDialog::typeChanged(int i) {
	maxButton->setEnabled(i == 1 || i == 2);
	maxEdit->setEnabled(maxButton->isChecked() && (i == 1 || i == 2));
	maxIncludeEqualsButton->setEnabled(maxButton->isChecked() && i == 1);
	maxIncludeEqualsButton->setChecked(i == 1 || i  == 2);
	minButton->setEnabled(i == 1 || i == 2);
	minEdit->setEnabled(maxButton->isChecked() && (i == 1 || i == 2));
	minIncludeEqualsButton->setEnabled(minButton->isChecked() && i == 1);
	minIncludeEqualsButton->setChecked(i == 1 || i  == 2);
	matrixButton->setEnabled(i != 7 && i != 0);
	matrixButton->setChecked(i == 7 || i == 0);
	complexButton->setEnabled(i == 1);
	complexButton->setChecked(i == 1);
}
void QalculateEditArgumentDialog::maxEnabled(bool b) {
	maxEdit->setEnabled(b);
	if(typeCombo->currentItem() == 2) {
		maxIncludeEqualsButton->setEnabled(b);
	}
}
void QalculateEditArgumentDialog::minEnabled(bool b) {
	minEdit->setEnabled(b);
	if(typeCombo->currentItem() == 2) {
		minIncludeEqualsButton->setEnabled(b);
	}
}

Argument *QalculateEditArgumentDialog::editArgument(Argument *arg) {

	if(arg) {
		setCaption(i18n("Edit Argument"));
	} else {
		setCaption(i18n("New Argument"));
	}

	nameEdit->setFocus();

	typeCombo->setCurrentItem(0);
	conditionEdit->clear();
	conditionButton->setChecked(false);
	nameEdit->clear();
	maxEdit->setText("0");
	minEdit->setText("0");
	testButton->setChecked(true);
	zeroButton->setChecked(false);
	matrixButton->setChecked(false);
	complexButton->setChecked(false);
	PrintOptions po;
	po.number_fraction_format = FRACTION_DECIMAL_EXACT;
	if(arg) {
		switch(arg->type()) {
			case ARGUMENT_TYPE_NUMBER: {
				typeCombo->setCurrentItem(1);
				NumberArgument *farg = (NumberArgument*) arg;
				minButton->setChecked(farg->min() != NULL);
				minIncludeEqualsButton->setChecked(farg->includeEqualsMin());
				if(farg->min()) minEdit->setText(farg->min()->print(po).c_str());
				maxButton->setChecked(farg->max() != NULL);
				maxIncludeEqualsButton->setChecked(farg->includeEqualsMax());
				if(farg->max()) maxEdit->setText(farg->max()->print(po).c_str());
				complexButton->setChecked(farg->complexAllowed());
				break;
			}
			case ARGUMENT_TYPE_INTEGER: {
				typeCombo->setCurrentItem(2);
				IntegerArgument *iarg = (IntegerArgument*) arg;
				minButton->setChecked(iarg->min() != NULL);
				if(iarg->min()) minEdit->setText(iarg->min()->print(po).c_str());
				maxButton->setChecked(iarg->max() != NULL);
				if(iarg->max()) maxEdit->setText(iarg->max()->print(po).c_str());
				break;
			}
			case ARGUMENT_TYPE_TEXT: {typeCombo->setCurrentItem(4); break;}
			case ARGUMENT_TYPE_SYMBOLIC: {typeCombo->setCurrentItem(3); break;}
			case ARGUMENT_TYPE_DATE: {typeCombo->setCurrentItem(5); break;}
			case ARGUMENT_TYPE_VECTOR: {typeCombo->setCurrentItem(6); break;}
			case ARGUMENT_TYPE_MATRIX: {typeCombo->setCurrentItem(7); break;}
			case ARGUMENT_TYPE_EXPRESSION_ITEM: {typeCombo->setCurrentItem(9); break;}
			case ARGUMENT_TYPE_FUNCTION: {typeCombo->setCurrentItem(10); break;}
			case ARGUMENT_TYPE_UNIT: {typeCombo->setCurrentItem(11); break;}
			case ARGUMENT_TYPE_VARIABLE: {typeCombo->setCurrentItem(12); break;}
			case ARGUMENT_TYPE_FILE: {typeCombo->setCurrentItem(13); break;}
			case ARGUMENT_TYPE_BOOLEAN: {typeCombo->setCurrentItem(8); break;}
			case ARGUMENT_TYPE_ANGLE: {typeCombo->setCurrentItem(14); break;}
		}
		nameEdit->setText(arg->name().c_str());
		matrixButton->setChecked(arg->matrixAllowed());
		zeroButton->setChecked(arg->zeroForbidden());
		testButton->setChecked(arg->tests());
		if(!arg->getCustomCondition().empty()) {
			conditionButton->setChecked(true);
			conditionEdit->setText(arg->getCustomCondition().c_str());
		}
	}
	typeChanged(typeCombo->currentItem());

	if(exec() == QDialog::Accepted) {
		switch(typeCombo->currentItem()) {
			case 1: {
				arg = new NumberArgument();
				if(minButton->isChecked()) {
					Number nr(minEdit->text().ascii());
					((NumberArgument*) arg)->setMin(&nr);
				}
				((NumberArgument*) arg)->setIncludeEqualsMin(minIncludeEqualsButton->isChecked());
				if(maxButton->isChecked()) {
					Number nr(maxEdit->text().ascii());
					((NumberArgument*) arg)->setMax(&nr);
				}
				((NumberArgument*) arg)->setIncludeEqualsMax(maxIncludeEqualsButton->isChecked());
				break;
			}
			case 2: {
				arg = new IntegerArgument();
				if(minButton->isChecked()) {
					Number nr(minEdit->text().ascii());
					((IntegerArgument*) arg)->setMin(&nr);
				}
				if(maxButton->isChecked()) {
					Number nr(maxEdit->text().ascii());
					((IntegerArgument*) arg)->setMax(&nr);
				}
				break;
			}
			case 3: {arg = new SymbolicArgument(); break;}
			case 4: {arg = new TextArgument(); break;}
			case 5: {arg = new DateArgument(); break;}
			case 6: {arg = new VectorArgument(); break;}
			case 7: {arg = new MatrixArgument(); break;}
			case 8: {arg = new BooleanArgument(); break;}
			case 9: {arg = new ExpressionItemArgument(); break;}
			case 10: {arg = new FunctionArgument(); break;}
			case 11: {arg = new UnitArgument(); break;}
			case 12: {arg = new VariableArgument(); break;}
			case 13: {arg = new FileArgument(); break;}
			case 14: {arg = new AngleArgument(); break;}
			default: {arg = new Argument(); break;}
		}
		arg->setName(nameEdit->text().ascii());
		arg->setZeroForbidden(zeroButton->isChecked());
		arg->setTests(testButton->isChecked());
		if(conditionButton->isChecked()) {
			arg->setCustomCondition(conditionEdit->text().ascii());
		} else {
			arg->setCustomCondition("");
		}
		if(arg->type() != ARGUMENT_TYPE_FREE && arg->type() != ARGUMENT_TYPE_MATRIX) arg->setMatrixAllowed(matrixButton->isChecked());
		return arg;	
	}
	return NULL;

}

#include "qalculateeditfunctiondialog.moc"

