#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-
#
# sebastien.martini@gmail.com
#
# 07/16/2006 - MIT License
#
# Beware that your processing are not conflicting with the paths
# that you are watching i.e. risking to produce infinite loops.
#

import os
import sys
from os.path import join
from optparse import OptionParser

# please read src/examples/README
try:
    # import local build
    import autopath
    from src.pyinotify.pyinotify import ProcessEvent, ThreadedNotifier, \
         WatchManager, EventsCodes
except ImportError:
    # import global (installed) pyinotify
    from pyinotify import ProcessEvent, ThreadedNotifier, WatchManager, \
         EventsCodes


__metaclass__ = type  # Use new-style classes by default

DEBUG = True


class _Queue(list):
    """
    Simple Queue container.

    """
    def __init__(self):
        super(_Queue, self).__init__()
        self._len = 0

    def pop(self):
        """
        Get the first element and remove it from the queue.

        @return: element.
        @rtype: element instance, None if the queue is empty
        """
        try:
            ret = list.pop(self)
            self._len -= 1
            return ret
        except IndexError:
            return None

    def length(self):
        return self._len

    def push(self, new_e):
        """
        Insert back, one element.

        @param new_e: element.
        @type new_e: instance of class
        """
        self.insert(0, new_e)
        self._len += 1


class PyinotifyFS(ProcessEvent):
    """
    pyinotify fs built by writing vtagfs' mount point.
    """
    COUNT = 0  # event counter

    def __init__(self, mountpt, maxlastevent=50):
        # mount point
        self._mp = mountpt
        assert os.path.ismount(self._mp)
        # last event queue
        self._last_eventsq = _Queue()
        # fixme: /!\ this value can't this be too high !
        self._maxsize = maxlastevent
        self._lasteventtag = 'LAST_%d_EVENTS' % self._maxsize
        # declare tags
        os.mkdir(join(self._mp, self._lasteventtag))
        for ec in dir(EventsCodes):
            if ec.startswith('IN_') and ec != 'IN_ISDIR':
                os.mkdir(join(self._mp, ec))

    def process_default(self, event):
        # print output message through base method
	if DEBUG:
            super(PyinotifyFS, self).process_default(event)

        srcname = '%s_%s' % (event.event_name, PyinotifyFS.COUNT)

        if hasattr(event, 'path'):
            target = event.path
            if event.name:
                target = join(event.path, event.name)

            os.symlink(target, join(self._mp, event.event_name, srcname))
            os.symlink(target, join(self._mp, self._lasteventtag, srcname))
        else:
            # for IN_UNMOUNT and IN_Q_OVERFLOW
            os.mknod(join(self._mp, event.event_name, srcname))
            os.mknod(join(self._mp, self._lasteventtag, srcname))

        event.fsname = srcname
        self._last_eventsq.push(event)
        if self._last_eventsq.length() > self._maxsize:
            e_ = self._last_eventsq.pop()
            # fixme: keep it forever ?
            os.unlink(join(self._mp, e_.event_name, e_.fsname))
            os.unlink(join(self._mp, self._lasteventtag, e_.fsname))

        PyinotifyFS.COUNT += 1


if __name__ == '__main__':
    usage = "usage: %prog [options] mountpoint path1 [path2] [pathn]"

    parser = OptionParser(usage=usage)
    parser.add_option("-v", "--verbose", action="store_true",
                      dest="verbose", help="Verbose mode")
    parser.add_option("-r", "--recursive", action="store_true",
                      dest="recursive",
                      help="Add watchs recursively on paths")
    parser.add_option("-a", "--auto_add", action="store_true",
                      dest="auto_add",
                      help="Automatically add watchs on new directories")
    parser.add_option("-s", "--max_size", type="int", nargs=1, action="store",
                      dest="max_size",
                      help="keep max_size events in fs.")

    (options, args) = parser.parse_args()

    global VERBOSE
    VERBOSE = options.verbose

    if len(args) < 2:
        parser.error()

    if options.max_size:
        maxlastevent = options.max_size
    else:
        maxlastevent = 200

    mountpt = args[0]
    path = args[1:]

    wm = WatchManager()
    notifier = ThreadedNotifier(wm, PyinotifyFS(mountpt, maxlastevent))
    notifier.start()

    # only watch these events
    mask = EventsCodes.IN_CREATE | EventsCodes.IN_DELETE | \
           EventsCodes.IN_DELETE_SELF | EventsCodes.IN_MODIFY | \
           EventsCodes.IN_MOVE_SELF | EventsCodes.IN_MOVED_FROM | \
           EventsCodes.IN_MOVED_TO
    # EventsCodes.ALL_EVENTS


    wm.add_watch(path, mask, rec=options.recursive, auto_add=options.auto_add)

    print 'start monitoring %s' % path
    while True:
        try:
            import time
            time.sleep(5)
        except KeyboardInterrupt:
            print 'stop monitoring...'
            # stop thread
            notifier.stop()
            # cleanup fs
            assert os.path.ismount(mountpt)
            for root, dirs, files in os.walk(mountpt, topdown=False):
                for name in files:
                    os.unlink(join(root, name))
                for name in dirs:
                    p_ = join(root, name)
                    if os.path.islink(p_):
                        os.unlink(p_)
                    elif name.startswith('IN_') or \
                         name.startswith('LAST_'):
                        os.rmdir(p_)
            break
        except Exception, err:
            print err

