/*
    ApplicationTypeAttributes.m

    Implementation of the ApplicationTypeAttributes class for the
    ProjectManager application.

    Copyright (C) 2005, 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ApplicationTypeAttributes.h"

#import <Foundation/NSString.h>

#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTextField.h>
#import <AppKit/NSTableView.h>
#import <AppKit/NSImageView.h>

#import "../../ProjectDocument.h"

@implementation ApplicationTypeAttributes

- (void) dealloc
{
  TEST_RELEASE(view);
  TEST_RELEASE(typeInfo);

  [super dealloc];
}

- initWithTypesArray: (NSMutableArray *) types
            document: (ProjectDocument *) doc
{
  if ([self init])
    {
      ASSIGN(typeInfo, types);
      document = doc;

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) awakeFromNib
{
  [view retain];
  [view removeFromSuperview];
  DESTROY(bogusWindow);
}

- (void) add: (id)sender
{
  [typeInfo addObject: [NSDictionary dictionaryWithObject: _(@"newtype")
                                                   forKey: @"NSName"]];

  [document updateChangeCount: NSChangeDone];
  [table reloadData];
}


- (void) changeDescription: (id)sender
{
  NSMutableDictionary * dict;

  dict = [[[typeInfo objectAtIndex: [table selectedRow]] mutableCopy]
    autorelease];
  [dict setObject: [description stringValue]
           forKey: @"NSHumanReadableName"];
  [typeInfo replaceObjectAtIndex: [table selectedRow]
                      withObject: dict];

  [document updateChangeCount: NSChangeDone];
}


- (void) changeDocumentClass: (id)sender
{
  NSMutableDictionary * dict;

  dict = [[[typeInfo objectAtIndex: [table selectedRow]] mutableCopy]
    autorelease];
  [dict setObject: [documentClass stringValue] forKey: @"NSDocumentClass"];
  [typeInfo replaceObjectAtIndex: [table selectedRow]
                      withObject: dict];

  [document updateChangeCount: NSChangeDone];
}


- (void) changeIconFilename: (id)sender
{
  NSMutableDictionary * dict;

  dict = [[[typeInfo objectAtIndex: [table selectedRow]] mutableCopy]
    autorelease];
  [dict setObject: [iconFilename stringValue] forKey: @"NSIcon"];
  [typeInfo replaceObjectAtIndex: [table selectedRow]
                      withObject: dict];

  [document updateChangeCount: NSChangeDone];
}


- (void) changeExtensions: (id)sender
{
  NSString * string = [extensions stringValue];
  NSMutableArray * extList;
  NSMutableDictionary * dict;

  string = [string stringByReplacingString: @"," withString: @" "];
  extList = [[[string componentsSeparatedByString: @" "] mutableCopy]
    autorelease];
  [extList removeObject: @""];

  dict = [[[typeInfo objectAtIndex: [table selectedRow]] mutableCopy]
    autorelease];
  [dict setObject: extList forKey: @"NSUnixExtensions"];
  [typeInfo replaceObjectAtIndex: [table selectedRow]
                      withObject: dict];

  [document updateChangeCount: NSChangeDone];
}


- (void) remove: (id)sender
{
  int row = [table selectedRow];

  if (row >= 0)
    {
      [typeInfo removeObjectAtIndex: row];
      [table reloadData];
      [self showType: nil];
      [document updateChangeCount: NSChangeDone];
    }
}

- (void) showType: sender
{
  int row = [table selectedRow];

  if (row >= 0)
    {
      NSDictionary * info = [typeInfo objectAtIndex: row];

      [description setStringValue: [info objectForKey:
        @"NSHumanReadableDescription"]];
      [description setEditable: YES];

      [iconFilename setStringValue: [info objectForKey: @"NSIcon"]];
      [iconFilename setEditable: YES];

      {
        NSArray * exts;
        int i, n;
        NSString * extensionsString = @"";

        exts = [info objectForKey: @"NSUnixExtensions"];
        for (i = 0, n = [exts count]; i < n; i++)
          {
            NSString * extension = [exts objectAtIndex: i];

            extensionsString = [extensionsString
              stringByAppendingString: extension];

            if (i < n - 1)
              {
                extensionsString = [extensionsString
                  stringByAppendingString: @", "];
              }
          }

        [extensions setStringValue: extensionsString];
        [extensions setEditable: YES];

        [documentClass setStringValue: [info objectForKey: @"NSDocumentClass"]];
        [documentClass setEditable: YES];
      }
    }
  else
    {
      [description setStringValue: nil];
      [description setEditable: NO];

      [iconFilename setStringValue: nil];
      [iconFilename setEditable: NO];

      [extensions setStringValue: nil];
      [extensions setEditable: NO];

      [documentClass setStringValue: nil];
      [documentClass setEditable: NO];

      [icon setImage: nil];
    }
}

- (NSView *) view
{
  if (view == nil)
    {
      [NSBundle loadNibNamed: @"ApplicationTypeAttributes" owner: self];
    }

  return view;
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [typeInfo count];
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  return [[typeInfo objectAtIndex: rowIndex] objectForKey: @"NSName"];
}

- (void) tableView: (NSTableView *)aTableView 
    setObjectValue: (id)anObject 
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  NSMutableDictionary * dict;

  dict = [[[typeInfo objectAtIndex: rowIndex] mutableCopy]
    autorelease];
  [dict setObject: anObject forKey: @"NSName"];
  [typeInfo replaceObjectAtIndex: [table selectedRow]
                      withObject: dict];

  [document updateChangeCount: NSChangeDone];
}

@end
