/*
 * prismstumbler wireless lan analyzing tool
 *
 * Copyright (C) 2000 and created by Jan Fernquist  <Jan.B.Fernquist@telia.com>
 * Copyright (C) 2003 Florian Boor <florian.boor@kernelconcepts.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * database tools
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include "ps_database.h"
#include "psgui.h"
#include "prismstumbler.h"

#define SQLITE_OK           0   /* Successful result */
#define DB_DIR				".pstumbler"
#define DATABASENAME		"prismstumblerdb"
#define TABLE_NETWORKS		"networks"

#define SQL_CREATE_TABLE	"create table " TABLE_NETWORKS "(bssid char(32), ssid char(40), first timestamp, \
							last timestamp, type char(20), ip_range char(32), wep int, dhcp int, ap char(16), \
							pdata long, psum long, pint long, wep_key char(48), longitude float, latitude float, \
							channel int, signal int, ishidden int, pcount int, pvec char(20), speed int, primary key(bssid))"
#define SQL_CHECK_TABLE		"select psum from " TABLE_NETWORKS ";"
#define SQL_DROP_TABLE		"drop table " TABLE_NETWORKS ";"
#define SQL_GETALL			"select ssid,channel,bssid,signal from " TABLE_NETWORKS ";"
#define SQL_SELECTBY_BSSID	"select * from " TABLE_NETWORKS " where bssid = '%q';"
#define SQL_SELECTPOSBY_BSSID	"select ssid,bssid,longitude,latitude from " TABLE_NETWORKS " where bssid = '%q';"
#define SQL_ADD_NETY		"insert into " TABLE_NETWORKS " values('%q','%q',%i,%i,'%q','%q',%i,%i,'%q',%i,%i,%i,'%q',%f,%f);"
#define SQL_ADD_NET			"insert into " TABLE_NETWORKS " (bssid,ssid,first,last,type,ip_range,wep,dhcp,ap,pdata,psum, \
							pint,wep_key,longitude,latitude,channel,signal,ishidden,pcount,pvec,speed) \
							values('%q','%q',%i,%i,'%q','%q',%i,%i,'%q',%ld,%i,%i,'%q',%f,%f,%i,%i,%i,%i,'%q',%i);"
#define SQL_SETUP_DB		"PRAGMA empty_result_callbacks = ON;"
#define SQL_UPDATE_NET		"update " TABLE_NETWORKS " set ssid='%q',last=%i,wep=%i,dhcp=%i,ap='%q',pdata=%ld,psum=%ld, \
							pint=%ld,wep_key='%q',longitude=%f,latitude=%f,channel=%i, signal=%i,ishidden=%i, \
							ip_range='%q', pcount=%i, pvec='%q', speed=%i where bssid = '%q';"
#define SQL_DELETEBY_BSSID	"delete from " TABLE_NETWORKS " where bssid = '%q' and channel = %i;"


static sqlite* psdb = NULL;
char **myresult;
pthread_mutex_t db_mutex = PTHREAD_MUTEX_INITIALIZER;

// forward declarations
int getall_networks(sqlite_callback fCallback);
psnetinfo_t get_network(char* my_bssid);
int save_network(psnetinfo_t my_net);
int check_table();
int init_db();
int create_table();
void close_db();



inline void lock_db()
{
	pthread_mutex_lock(&db_mutex);
}


inline void unlock_db()
{
	pthread_mutex_unlock(&db_mutex);
}


int deletenetwork(char* my_bssid,int chan)
{
	char* errmsg = NULL;
	int retval;
	int nrow = 0;
	int ncolumn = 0;
	lock_db();
	retval = sqlite_get_table_printf(psdb,SQL_DELETEBY_BSSID,&myresult,&nrow,&ncolumn,&errmsg,my_bssid,chan);
	if (myresult) sqlite_free_table(myresult);
	unlock_db();
	
	if (retval){
		fprintf(stderr,"Error delete: %s\n",errmsg);
		free(errmsg);
		return -1;
	}
	return 0;
}


int getnetworkpos(char* my_bssid,sqlite_callback fCallback)
{
	char* errmsg = NULL;
	int retval;

	retval = sqlite_exec_printf(psdb,SQL_SELECTPOSBY_BSSID,fCallback,NULL,&errmsg,my_bssid);
	
	if (retval){
		fprintf(stderr,"Error get networkpos: %s\n",errmsg);
		free(errmsg);
		return -1;
	}
	return 0;
}


int getall_networks(sqlite_callback fCallback)
{
	char* errmsg = NULL;
	int retval;

	retval = sqlite_exec(psdb,SQL_GETALL,fCallback,NULL,&errmsg);
	if (retval){
		fprintf(stderr,"Error get networks: %s\n",errmsg);
		free(errmsg);
		return -1;
	}
	return 0;
}



psnetinfo_t get_network(char* my_bssid)
{
	char* errmsg = NULL;
	int nrow = 0;
	int ncolumn = 0;
	static psnetinfo_t my_net;

	memset(&my_net,0,sizeof(psnetinfo_t));
	if (psdb)
	{
		#ifdef DEBUG
		printf("fetching network...\n");
		#endif
		lock_db();
		sqlite_get_table_printf(psdb,SQL_SELECTBY_BSSID,&myresult,&nrow,&ncolumn,&errmsg,my_bssid);
		unlock_db();

		if (nrow > 0) // we found this net in the database
		{
			// get data
			//HACK
			ncolumn-=1;
			strcpy(my_net.bssid,myresult[ncolumn+1]);
			strcpy(my_net.ssid,myresult[ncolumn+2]);
			my_net.first = atol(myresult[ncolumn+3]);
			my_net.last = atol(myresult[ncolumn+4]);

			strcpy(my_net.type,myresult[ncolumn+5]);
			strcpy(my_net.ip_range,myresult[ncolumn+6]);

			my_net.wep = atol(myresult[ncolumn+7]);
			my_net.dhcp = atol(myresult[ncolumn+8]);
			strcpy(my_net.ap,myresult[ncolumn+9]);
			my_net.pdata = atol(myresult[ncolumn+10]);
			my_net.psum = atol(myresult[ncolumn+11]);
			my_net.pint = atol(myresult[ncolumn+12]);
			strcpy(my_net.wep_key,myresult[ncolumn+13]);
			my_net.longitude = atof(myresult[ncolumn+14]);
			my_net.latitude = atof(myresult[ncolumn+15]);
			my_net.channel = atol(myresult[ncolumn+16]);
			my_net.maxsiglevel = atol(myresult[ncolumn+17]);
			my_net.ishidden = atol(myresult[ncolumn+18]);

			my_net.pcount = atol(myresult[ncolumn+19]);
			strncpy(my_net.pvec,myresult[ncolumn+20],my_net.pcount);
			my_net.speed = atol(myresult[ncolumn+21]);
			
			#ifdef DEBUG
			printf("found: %i c: %i\n",nrow,ncolumn);
			printf("read: %s\n",my_net.bssid);
			#endif
			sqlite_free_table(myresult);
		}

		if (errmsg){
			fprintf(stderr,"Error fetch network: %s\n",errmsg);
			free(errmsg);
		}
	}
	return my_net;
}


int save_network(psnetinfo_t my_net)
{
	char* errmsg = NULL;
	int nrow = 0;
	int ncolumn = 0;
	int ret;

	if (psdb)
	{
		#ifdef DEBUG
		printf("saving network...");
		#endif
		lock_db();
		sqlite_get_table_printf(psdb,SQL_SELECTBY_BSSID,&myresult,&nrow,&ncolumn,&errmsg,my_net.bssid);
		if (nrow > 0) // we found this net in the database
		{
			#ifdef DEBUG
			printf("updating %c\n",my_net.pvec[0]);
			#endif
			sqlite_free_table(myresult);
			// update network
			ret = sqlite_get_table_printf(psdb,SQL_UPDATE_NET,&myresult,&nrow,&ncolumn,&errmsg,
					my_net.ssid,my_net.last,my_net.wep,my_net.dhcp,my_net.ap,my_net.pdata,my_net.psum,
					my_net.pint,my_net.wep_key,my_net.longitude,my_net.latitude,my_net.channel,my_net.maxsiglevel,my_net.ishidden,
					my_net.ip_range,my_net.pcount,my_net.pvec,my_net.speed,my_net.bssid);
			if (ret){
				fprintf(stderr,"Error db update: %s\n",errmsg);
				free(errmsg);
				unlock_db();
				return -1;
			}
		}
		else // new network
		{
			#ifdef DEBUG
			printf("adding\n");
			#endif
			// first = last - caution on c&p
			ret = sqlite_get_table_printf(psdb,SQL_ADD_NET,&myresult,&nrow,&ncolumn,&errmsg,my_net.bssid,
					my_net.ssid,my_net.last, my_net.last, my_net.type, my_net.ip_range, my_net.wep, 
					my_net.dhcp, my_net.ap, my_net.pdata, my_net.psum, my_net.pint, my_net.wep_key, 
					my_net.longitude, my_net.latitude, my_net.channel,my_net.maxsiglevel,my_net.ishidden,
					my_net.pcount,my_net.pvec,my_net.speed);
			if (ret){
				fprintf(stderr,"Error db adding network: %s\n",errmsg);
				free(errmsg);
				unlock_db();
				return -1;
			}
			if (myresult) sqlite_free_table(myresult);
		}
		unlock_db();
	}
	return 0;
}


int check_table()
{
	char* errmsg = NULL;
	int nrow, ncolumn;

	if (!psdb) return -1;

	lock_db();
	sqlite_get_table(psdb,SQL_CHECK_TABLE,&myresult,&nrow,&ncolumn,&errmsg);
	unlock_db();
	if (myresult) 
	{
		if (atol(myresult[1]) != DBVERSION) 
		{		
			sqlite_free_table(myresult);
			return -2;
		}
		else
			sqlite_free_table(myresult);
	}
	if (errmsg){
		fprintf(stderr,"Error db check: %s\n",errmsg);
		free(errmsg);
		return -1;
	}
    return 0;
}


int init_db()
{
	char* errmsg = NULL;
	int nrow, ncolumn, res;
	char* homedir;
	char* envtmp;
	DIR* dp;
	char **myresult;
	

	envtmp = getenv("HOME");
	if (!envtmp) return -1;

	homedir=malloc(strlen(envtmp)+1);
	homedir=strcpy(homedir,envtmp);
	
	homedir = realloc(homedir,sizeof(char)*(strlen(homedir)+strlen(DB_DIR)+2));
	homedir = strcat(homedir,"/");
	homedir = strcat(homedir,DB_DIR);
	
	if ((dp = opendir(homedir)) == NULL)
	{
		if (mkdir(homedir,S_IRUSR | S_IWUSR | S_IXUSR)) return -1;
	}
	else closedir(dp);
	
	homedir = realloc(homedir,sizeof(char)*(strlen(homedir)+strlen(DATABASENAME)+2));
	homedir = strcat(homedir,"/");
	homedir = strcat(homedir,DATABASENAME);
		
	lock_db();
	psdb = sqlite_open(homedir, 666, &errmsg);
	unlock_db();
	free(homedir);

	if (errmsg){
	 	fprintf(stderr,"Error init db: %s\n",errmsg);
		free(errmsg);
		psdb = NULL;
		return -1;
	}
	lock_db();
	sqlite_get_table(psdb,SQL_SETUP_DB,&myresult,&nrow,&ncolumn,&errmsg);
	if (myresult) sqlite_free_table(myresult);
	unlock_db();
	if (errmsg){
		if (!QUIET_MODE) fprintf(stderr,"Error db setup: %s\n",errmsg);
		free(errmsg);
		sqlite_close(psdb);
		psdb = NULL;
		return -1;
	}
	
	/* check opened table for version etc. */
	res = check_table();

	switch (res)
	{
	case -2: /* wrong version - create new */
		lock_db();
		sqlite_get_table(psdb,SQL_DROP_TABLE,&myresult,&nrow,&ncolumn,&errmsg);
		if (myresult) sqlite_free_table(myresult);
		unlock_db();
		if (errmsg){
			fprintf(stderr,"Error db setup: %s\n",errmsg);
			free(errmsg);
		}
		create_table();
	break;
	case -1: /* critical error, try to create table */
		return create_table();
	break;	
	default:
		return 0;
	break;	
	}
	return 0;
}


int create_table()
{
	char* errmsg = NULL;
	int nrow, ncolumn;
	int retval;
	psnetinfo_t* new_net;

	if (!psdb) return -1;

	lock_db();
	retval = sqlite_get_table(psdb,SQL_CREATE_TABLE,&myresult,&nrow,&ncolumn,&errmsg);
	if (myresult) sqlite_free_table(myresult);
	unlock_db();
	
	if (retval){
		fprintf(stderr,"Error creating db: %s\n",errmsg);
		free(errmsg);
		return -1;
	}
	
	new_net = malloc(sizeof(psnetinfo_t));
	memset(new_net,0,sizeof(psnetinfo_t));
	
	sprintf(new_net->ssid,"%s","this is my net");
	sprintf(new_net->bssid,"%s","G0:00:00:00:00:00");
	sprintf(new_net->pvec,"%s","\0");
	new_net->pcount = 0;
	new_net->psum = DBVERSION;
	save_network(*new_net);

	free(new_net);
	return 0;
}


void close_db()
{
	lock_db();
	sqlite_close(psdb);
	unlock_db();
}
