/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

namespace Dock {

    [DBus (name = "net.launchpad.DockManager")]
    interface DockManagerService : Object {
        public abstract string[] get_items_by_name (string app_name) throws IOError;
        public signal void item_added (string object_path);
        public signal void item_removed (string object_path);
    }
    [DBus (name = "net.launchpad.DockItem")]
    interface DockItemService : GLib.DBusProxy {
        public abstract void update_dock_item (HashTable<string, Variant> hints) throws IOError;
        public abstract int32 add_menu_item (HashTable<string, Variant> hints) throws IOError;
        public signal void menu_item_activated (int32 id);
    }

    public class Manager : Object {
        DockManagerService? service = null;

        public signal void item_added (string object_path);
        public signal void item_removed (string object_path);

        public Manager () {
            try {
                service = Bus.get_proxy_sync (BusType.SESSION,
                                              "net.launchpad.DockManager",
                                              "/net/launchpad/DockManager");
                service.item_added.connect ((object_path) => {
                    item_added (object_path);
                });
                service.item_removed.connect ((object_path) => {
                    item_removed (object_path);
                });
            } catch (GLib.Error error) {
                GLib.debug ("No DockManager running: %s", error.message);
            }
        }

        public string[] get_items_by_name (string app_name) {
            try {
                if (service == null)
                    throw new GLib.IOError.FAILED ("Service unavailable");
                string[] items = service.get_items_by_name (app_name);
                return items;
            } catch (GLib.Error error) {
                return {};
            }
        }
    }

    public class Item : Object {
        Manager manager = new Manager ();
        uint last_count = 0;
        DockItemService? service = null;

        public signal void menu_item_activated (int32 id);

        public Item.for_name (string app_name) {
            manager.item_added.connect ((oobject_path) => {
                if (service != null)
                    return;
                var dockitems = manager.get_items_by_name (app_name);
                foreach (string object_path in dockitems) {
                    update_service (object_path);
                    set_badge (last_count);
               }
            });
            manager.item_removed.connect ((object_path) => {
                if (object_path.has_suffix (app_name))
                    service = null;
            });

            var dockitems = manager.get_items_by_name (app_name);
            foreach (string object_path in dockitems) {
                update_service (object_path);
                break;
            }
        }

        public void set_badge (uint count) {
            if (service == null)
                return;

            var hints = new HashTable<string, Variant> (str_hash, str_equal);
            if (count == 0)
                hints.insert ("badge", "");
            else
                hints.insert ("badge", count.to_string ());
            update_dock_item (hints);
            last_count = count;
        }

        void update_service (string object_path) {
            try {
                service = Bus.get_proxy_sync (BusType.SESSION,
                                              "net.launchpad.DockManager",
                                              object_path);
                service.menu_item_activated.connect ((id) => {
                    menu_item_activated (id);
                });
            } catch (GLib.Error error) {
                GLib.critical ("DockItem not running: %s", error.message);
            }
        }

        public Item (string object_path) {
            update_service (object_path);
        }

        public void update_dock_item (HashTable<string, Variant> hints) {
            try {
                if (service == null)
                    throw new GLib.IOError.FAILED ("Service unavailable");
                service.update_dock_item (hints);
            } catch (GLib.Error error) {
                GLib.critical ("Failed to update DockItem: %s", error.message);
            }
        }

        public int32 add_menu_item (string icon_name, string label) {
            var hints = new HashTable<string, Variant> (str_hash, str_equal);
            hints.insert ("icon-name", icon_name);
            hints.insert ("label", label);
            try {
                if (service == null)
                    throw new GLib.IOError.FAILED ("Service unavailable");
                return service.add_menu_item (hints);
            } catch (GLib.Error error) {
                GLib.critical ("Failed to add menu item for DockItem: %s", error.message);
            }
            return 0;
        }
    }
}

