/* Copyright (c) 2001-2007, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.database;

import com.pixelmed.dicom.*;

import java.sql.*;
import java.util.Iterator;
import java.util.Map;

/**
 * <p>The {@link com.pixelmed.database.StudySeriesInstanceSelectiveMatchModel StudySeriesInstanceSelectiveMatchModel} class
 * supports a minimal DICOM Study/Series/Instance model.</p>
 *
 * <p>Matching of each information entity is performed by all appropriate attributes at
 * that level, not just the instance UIDs alone that are used in {@link com.pixelmed.database.StudySeriesInstanceModel StudySeriesInstanceModel}.</p>
 *
 * <p>Attributes of other DICOM entities than Study, Series and Instance are included at the appropriate lower level entity.</p>
 *
 * @see com.pixelmed.database.StudySeriesInstanceModel
 *
 * @author	dclunie
 */
public class StudySeriesInstanceSelectiveMatchModel extends StudySeriesInstanceModel {

	/***/
	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/database/StudySeriesInstanceSelectiveMatchModel.java,v 1.5 2007/07/04 18:13:58 dclunie Exp $";

	/**
	 * <p>Construct a model with the attributes from the default dictionary.</p>
	 *
	 * <p>The dictionary {@link com.pixelmed.database.DicomDictionaryForStudySeriesInstanceModel DicomDictionaryForStudySeriesInstanceModel} is used.</p>
	 * 
	 * @param	databaseFileName
	 * @exception	DicomException
	 */
	public StudySeriesInstanceSelectiveMatchModel(String databaseFileName) throws DicomException {
		super(databaseFileName);
	}

	/**
	 * <p>Construct a model with the attributes from the default dictionary allowing external SQL access.</p>
	 *
	 * <p>The dictionary {@link com.pixelmed.database.DicomDictionaryForStudySeriesInstanceModel DicomDictionaryForStudySeriesInstanceModel} is used.</p>
	 * 
	 * @param	databaseFileName
	 * @param	databaseServerName
	 * @exception	DicomException
	 */
	public StudySeriesInstanceSelectiveMatchModel(String databaseFileName,String databaseServerName) throws DicomException {
		super(databaseFileName,databaseServerName);
	}

	/**
	 * <p>Construct a model with the attributes from the specified dictionary.</p>
	 *
	 * @param	databaseFileName
	 * @param	dictionary
	 * @exception	DicomException
	 */
	public StudySeriesInstanceSelectiveMatchModel(String databaseFileName,DicomDictionary dictionary) throws DicomException {
		super(databaseFileName,dictionary);
	}

	/**
	 * <p>Construct a model with the attributes from the specified dictionary allowing external SQL access.</p>
	 *
	 * @param	databaseFileName
	 * @param	databaseServerName
	 * @param	dictionary
	 * @exception	DicomException
	 */
	public StudySeriesInstanceSelectiveMatchModel(String databaseFileName,String databaseServerName,DicomDictionary dictionary) throws DicomException {
		super(databaseFileName,databaseServerName,dictionary);
	}

	/**
	 * @param	b
	 * @param	list
	 * @param	ie
	 * @exception	DicomException
	 */
	protected void extendStatementStringWithMatchingAttributesForSelectedInformationEntity(StringBuffer b,AttributeList list,InformationEntity ie) throws DicomException {

		// two possibilities ...
		// 1. iterate through whole list of attributes and insist on match for all present for that IE
		// 2. be more selective ... consider match only on "unique key(s)" for a particular level and ignore others
		//
		// adopt the former approach ...

		// also need to escape wildcards and so on, but ignore for now ...

		if      (ie == InformationEntity.STUDY) {
			// no AND since would be no parent reference preceding
			b.append("STUDY.StudyInstanceUID");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.StudyInstanceUID)));
			b.append(" AND ");
			b.append("STUDY.StudyID");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.StudyID)));
			b.append(" AND ");
			b.append("STUDY.StudyDate");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.StudyDate)));
			b.append(" AND ");
			b.append("STUDY.StudyTime");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.StudyTime)));
			b.append(" AND ");
			b.append("STUDY.StudyDescription");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.StudyDescription)));
			b.append(" AND ");
			b.append("STUDY.AccessionNumber");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.AccessionNumber)));
			b.append(" AND ");
			b.append("STUDY.PatientName");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.PatientName)));
			b.append(" AND ");
			b.append("STUDY.PatientID");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.PatientID)));
			b.append(" AND ");
			b.append("STUDY.PatientBirthDate");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.PatientBirthDate)));
			b.append(" AND ");
			b.append("STUDY.PatientSex");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.PatientSex)));
		}
		else if (ie == InformationEntity.SERIES) {
			b.append(" AND ");
			b.append("SERIES.SeriesInstanceUID");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.SeriesInstanceUID)));
			b.append(" AND ");
			b.append("SERIES.SeriesDate");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.SeriesDate)));
			b.append(" AND ");
			b.append("SERIES.SeriesTime");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.SeriesTime)));
			b.append(" AND ");
			b.append("SERIES.SeriesNumber");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.SeriesNumber)));
			b.append(" AND ");
			b.append("SERIES.SeriesDescription");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.SeriesDescription)));
			b.append(" AND ");
			b.append("SERIES.Modality");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.Modality)));
			b.append(" AND ");
			b.append("SERIES.Manufacturer");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.Manufacturer)));
			b.append(" AND ");
			b.append("SERIES.InstitutionName");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.InstitutionName)));
		}
		else if (ie == InformationEntity.INSTANCE) {
			b.append(" AND ");
			b.append("INSTANCE.SOPInstanceUID");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.SOPInstanceUID)));
			b.append(" AND ");
			b.append("INSTANCE.ContentDate");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.ContentDate)));
			b.append(" AND ");
			b.append("INSTANCE.ContentTime");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.ContentTime)));
			b.append(" AND ");
			b.append("INSTANCE.AcquisitionDate");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.AcquisitionDate)));
			b.append(" AND ");
			b.append("INSTANCE.AcquisitionTime");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.AcquisitionTime)));
			b.append(" AND ");
			b.append("INSTANCE.AcquisitionDateTime");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.AcquisitionDateTime)));
			b.append(" AND ");
			b.append("INSTANCE.InstanceNumber");
			appendExactOrIsNullMatch(b,getQuotedEscapedSingleStringValueOrNull(list.get(TagFromName.InstanceNumber)));
		}
	}
}

