%{
/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "src/types.h"
#include "src/util.h"
#include "src/list.h"
#include "src/expr.h"
#include "src/parser.tab.h"
#include "src/scanner.h"
#include "src/parser.h"

void
yyerror(YYLTYPE *locp, yyscan_t scanner, const char *filename, char *str)
{
	fprintf(stderr, "%s:%d:%d: error: %s\n", filename, locp->first_line,
		locp->first_column, str);
}

/* Comparison functions */

static int
port_cmp_id(void *p, void *id)
{
	return strcmp(((struct port *)p)->id, (char *)id);
}

static int
conn_elem_cmp(void *e1, void *e2)
{
	struct conn_elem *elem1;
	struct conn_elem *elem2;
	int i;

	elem1 = (struct conn_elem *)e1;
	elem2 = (struct conn_elem *)e2;

	if (elem1->type > elem2->type)
		return 1;
	if (elem1->type < elem2->type)
		return -1;

	if (elem1->type == conn_elem_type_value)
		return 0;

	if ((elem1->c.id.component == NULL) && (elem2->c.id.component != NULL))
		return -1;
	if ((elem1->c.id.component != NULL) && (elem2->c.id.component == NULL))
		return 1;

	if (elem1->c.id.component != NULL)
	  {
		i = strcmp(elem1->c.id.component, elem2->c.id.component);
		if (i != 0)
			return i;
	  }

	return strcmp(elem1->c.id.port, elem2->c.id.port);
}

static int
connection_cmp(void *c1, void *c2)
{
	struct connection *conn1;
	struct connection *conn2;
	int i;

	conn1 = (struct connection *)c1;
	conn2 = (struct connection *)c2;

	i = conn_elem_cmp(&conn1->output, &conn2->output);

	return (i != 0) ? i : conn_elem_cmp(&conn1->input, &conn2->input);
}

static int
component_cmp_id(void *c, void *id)
{
	return strcmp(((struct component *)c)->id, (char *)id);
}

static int
stmt_cmp_output(void *s1, void *s2)
{
	struct stmt *stmt1;
	struct stmt *stmt2;
	int i;

	stmt1 = (struct stmt *)s1;
	stmt2 = (struct stmt *)s2;

	i = strcmp(stmt1->out.id.id, stmt2->out.id.id);
	if (i != 0)
		return i;

	return stmt1->out.id.out ? (stmt2->out.id.out ? 0 : 1)
				 : (stmt2->out.id.out ? -1 : 0);
}

static int
const_cmp_id(void *c, void *id)
{
	return strcmp(((struct const_v *)c)->id, (char *)id);
}

static int
ext_func_cmp_id(void *f, void *id)
{
	return strcmp(((struct ext_func *)f)->id, (char *)id);
}

static int
block_cmp_id(void *b, void *id)
{
	return strcmp(((struct block *)b)->id, (char *)id);
}

static int
system_cmp_id(void *s, void *id)
{
	return strcmp(((struct system *)s)->id, (char *)id);
}

static int
component_cmp_ports_id(void *c, void *p)
{
	struct component *comp;
	list_t ports;

	comp = (struct component *)c;
	ports = (list_t)p;

	if (list_find(ports, port_cmp_id, comp->id) != NULL)
		return 0;

	return 1;
}

/* Utility functions */

struct stmt_defines_port_output_context
  {
	const char	*filename;
	YYLTYPE		*locp;
	struct port	*port;
	char		 defined;
  };

static void
stmt_defines_port_output(void *data, void *context)
{
	struct stmt *stmt;
	struct stmt_defines_port_output_context *ctx;

	stmt = (struct stmt *)data;
	ctx = (struct stmt_defines_port_output_context *)context;

	if (ctx->defined)
		return;

	if (!strcmp(stmt->out.id.id, ctx->port->id))
	  {
		if (stmt->out.id.out && (ctx->port->type != port_type_w)
		    && (ctx->port->type != port_type_k))
		  {
			fprintf(stderr, "%s:%d:%d: error: port `%s' is not a "
				"physical port\n", ctx->filename,
				ctx->locp->first_line, ctx->locp->first_column,
				stmt->out.id.id);
			exit(EXIT_FAILURE);
		  }
		else if (!stmt->out.id.out
			 && (ctx->port->type != port_type_input)
		         && (ctx->port->type != port_type_output))
		  {
			fprintf(stderr, "%s:%d:%d: error: port `%s' is a "
				"physical port\n", ctx->filename,
				ctx->locp->first_line, ctx->locp->first_column,
				stmt->out.id.id);
			exit(EXIT_FAILURE);
		  }

		ctx->defined = 1;
	  }
}

struct port_output_defined_context
  {
	const char	*filename;
	YYLTYPE		*locp;
	list_t		 stmts;
  };

static void
port_output_defined(void *data, void *context)
{
	struct port *port;
	struct port_output_defined_context *ctx;
	struct stmt_defines_port_output_context s_ctx;

	port = (struct port *)data;
	ctx = (struct port_output_defined_context *)context;

	if (port->type == port_type_input)
		return;

	s_ctx.filename = ctx->filename;
	s_ctx.locp = ctx->locp;
	s_ctx.port = port;
	s_ctx.defined = 0;
	list_for_each(ctx->stmts, stmt_defines_port_output, &s_ctx);

	if (!s_ctx.defined)
	  {
		fprintf(stderr, "%s:%d:%d: error: the output of port `%s' was "
			"not defined\n", ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, port->id);
		exit(EXIT_FAILURE);
	  }
}

struct stmt_bind_context
  {
	const char	*filename;
	YYLTYPE		*locp;
	list_t		 ports;
  };

static void
stmt_bind(void *data, void *context)
{
	struct stmt *stmt;
	struct stmt_bind_context *ctx;
	struct port *port;
	struct expr_signal_id *id;
	struct ext_func *f;
	char *s;

	stmt = (struct stmt *)data;
	ctx = (struct stmt_bind_context *)context;

	port = list_find(ctx->ports, port_cmp_id, stmt->out.id.id);
	if (port == NULL)
	  {
		fprintf(stderr, "%s:%d:%d: error: unknown port `%s'\n",
			ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, stmt->out.id.id);
		exit(EXIT_FAILURE);
	  }

	if (stmt->out.id.out && (port->type != port_type_w)
	    && (port->type != port_type_k))
	  {
		fprintf(stderr, "%s:%d:%d: error: port `%s' is not a physical "
			"port\n", ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, stmt->out.id.id);
		exit(EXIT_FAILURE);
	  }
	else if (!stmt->out.id.out && (port->type != port_type_input)
		 && (port->type != port_type_output))
	  {
		fprintf(stderr, "%s:%d:%d: error: port `%s' is a physical "
			"port\n", ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, stmt->out.id.id);
		exit(EXIT_FAILURE);
	  }

	stmt->out.port = port;

	id = expr_bind_to_ports(stmt->expr, ctx->ports);
	if (id != NULL)
	  {
		if (id->in)
			fprintf(stderr, "%s:%d:%d: error: port `%s' is not a "
				"physical port\n", ctx->filename,
				ctx->locp->first_line, ctx->locp->first_column,
				id->id);
		else
			fprintf(stderr, "%s:%d:%d: error: port `%s' is a "
				"physical port\n", ctx->filename,
				ctx->locp->first_line, ctx->locp->first_column,
				id->id);
		exit(EXIT_FAILURE);
	  }

	expr_bind_consts(stmt->expr, parser_consts);

	id = expr_find_unbinded(stmt->expr);
	if (id != NULL)
	  {
		fprintf(stderr, "%s:%d:%d: error: unknown port `%s'\n",
			ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, id->id);
		exit(EXIT_FAILURE);
	  }

	s = expr_bind_ext_funcs(stmt->expr, parser_ext_funcs);
	if (s != NULL)
	  {
		fprintf(stderr, "%s:%d:%d: error: unknown function `%s'\n",
			ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, s);
		exit(EXIT_FAILURE);
	  }

	f = expr_check_ext_func_args(stmt->expr);
	if (f != NULL)
	  {
		fprintf(stderr, "%s:%d:%d: error: function `%s' takes %lu "
			"arguments\n", ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, f->id, f->n_args);
		exit(EXIT_FAILURE);
	  }
}

struct component_bind_type_context
  {
	const char	*filename;
	YYLTYPE		*locp;
  };

static void
component_bind_type(void *data, void *context)
{
	struct component *comp;
	struct component_bind_type_context *ctx;
	void *p;

	comp = (struct component *)data;
	ctx = (struct component_bind_type_context *)context;

	p = list_find(parser_blocks, block_cmp_id, comp->type.id);
	if (p != NULL)
	  {
		free(comp->type.id);
		comp->type.t.is_macro = 0;
		comp->type.t.p = p;
		return;
	  }

	p = list_find(parser_macros, system_cmp_id, comp->type.id);
	if (p != NULL)
	  {
		free(comp->type.id);
		comp->type.t.is_macro = 1;
		comp->type.t.p = p;
		return;
	  }

	fprintf(stderr, "%s:%d:%d: error: unknown component type `%s'\n",
		ctx->filename, ctx->locp->first_line, ctx->locp->first_column,
		comp->type.id);
	exit(EXIT_FAILURE);
}

static void
conn_elem_bind(struct conn_elem *elem, list_t components, list_t ports,
	       const char *filename, YYLTYPE *locp)
{
	struct component *comp;
	struct port *port;
	struct const_v *const_v;

	if (elem->type == conn_elem_type_value)
		return;

	if (elem->c.id.component != NULL)
	  {
		comp = list_find(components, component_cmp_id,
				 elem->c.id.component);
		if (comp == NULL)
		  {
			fprintf(stderr, "%s:%d:%d: error: unknown component "
				"`%s'\n", filename, locp->first_line,
				locp->first_column, elem->c.id.component);
			exit(EXIT_FAILURE);
		  }

		port = list_find(comp->type.t.is_macro
				 ? ((struct system *)comp->type.t.p)->ports
				 : ((struct block *)comp->type.t.p)->ports,
				 port_cmp_id, elem->c.id.port);
		if (port == NULL)
		  {
			fprintf(stderr, "%s:%d:%d: error: component `%s' does "
				"not have a port named `%s'\n", filename,
				locp->first_line, locp->first_column,
				elem->c.id.component, elem->c.id.port);
			exit(EXIT_FAILURE);
		  }

		free(elem->c.id.component);
		free(elem->c.id.port);
		elem->c.p.component = comp;
		elem->c.p.port = port;
	  }
	else
	  {
		port = list_find(ports, port_cmp_id, elem->c.id.port);
		if (port == NULL)
		  {
			const_v = list_find(parser_consts, const_cmp_id,
					    elem->c.id.port);
			if (const_v == NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: unknown port "
					"`%s'\n", filename, locp->first_line,
					locp->first_column, elem->c.id.port);
				exit(EXIT_FAILURE);
			  }

			free(elem->c.id.port);
			elem->c.value = const_v->value;
			elem->type = conn_elem_type_value;
			return;
		  }

		free(elem->c.id.port);
		elem->c.p.component = NULL;
		elem->c.p.port = port;
	  }

	if ((elem->type != conn_elem_type_port)
	    && (elem->c.p.port->type != port_type_w)
	    && (elem->c.p.port->type != port_type_k))
	  {
		fprintf(stderr, "%s:%d:%d: error: port `%s%s%s' is not a "
			"physical port\n", filename, locp->first_line,
			locp->first_column,
			(elem->c.p.component != NULL) ? elem->c.p.component->id
			: "", (elem->c.p.component != NULL) ? "." : "",
			elem->c.p.port->id);
		exit(EXIT_FAILURE);
	  }
}

struct connection_bind_context
  {
	const char	*filename;
	YYLTYPE		*locp;
	list_t		 ports;
	list_t		 components;
  };

#define CONN_ELEM_IS_W(elem) \
	(((elem).type == conn_elem_type_port) \
	 && ((elem).c.p.port->type == port_type_w))

#define CONN_ELEM_IS_K(elem) \
	(((elem).type == conn_elem_type_port) \
	 && ((elem).c.p.port->type == port_type_k))

#define CONN_ELEM_IS_PHYS(elem)	(CONN_ELEM_IS_W(elem) || CONN_ELEM_IS_K(elem))

static void
connection_bind(void *data, void *context)
{
	struct connection *conn;
	struct connection_bind_context *ctx;

	conn = (struct connection *)data;
	ctx = (struct connection_bind_context *)context;

	conn_elem_bind(&conn->output, ctx->components, ctx->ports,
		       ctx->filename, ctx->locp);

	if (((conn->output.c.p.component == NULL)
	     && (conn->output.c.p.port->type == port_type_input))
	    || ((conn->output.c.p.component != NULL)
	        && (conn->output.c.p.port->type == port_type_output)))
	  {
		fprintf(stderr, "%s:%d:%d: error: invalid connection\n",
			ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column);
		exit(EXIT_FAILURE);
	  }

	if (conn->input.type == conn_elem_type_value)
	  {
		if (CONN_ELEM_IS_PHYS(conn->output))
		  {
			fprintf(stderr, "%s:%d:%d: error: cannot assign a "
				"value to a physical port\n", ctx->filename,
				ctx->locp->first_line, ctx->locp->first_column);
			exit(EXIT_FAILURE);
		  }
		return;
	  }

	conn_elem_bind(&conn->input, ctx->components, ctx->ports, ctx->filename,
		       ctx->locp);

	if (conn->input.type == conn_elem_type_value)
	  {
		if (CONN_ELEM_IS_PHYS(conn->output))
		  {
			fprintf(stderr, "%s:%d:%d: error: cannot assign a "
				"value to a physical port\n", ctx->filename,
				ctx->locp->first_line, ctx->locp->first_column);
			exit(EXIT_FAILURE);
		  }
		return;
	  }

	if (((conn->input.c.p.component != NULL)
	     && (conn->input.c.p.port->type == port_type_input))
	    || ((conn->input.c.p.component == NULL)
	        && (conn->input.c.p.port->type == port_type_output))
	    || (CONN_ELEM_IS_W(conn->input) && !CONN_ELEM_IS_W(conn->output))
	    || (!CONN_ELEM_IS_W(conn->input) && CONN_ELEM_IS_W(conn->output))
	    || (CONN_ELEM_IS_K(conn->input) && !CONN_ELEM_IS_K(conn->output))
	    || (!CONN_ELEM_IS_K(conn->input) && CONN_ELEM_IS_K(conn->output)))
	  {
		fprintf(stderr, "%s:%d:%d: error: invalid connection\n",
			ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column);
		exit(EXIT_FAILURE);
	  }

	if (conn->input.c.p.port == conn->output.c.p.port)
	  {
		fprintf(stderr, "%s:%d:%d: error: self connection\n",
			ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column);
		exit(EXIT_FAILURE);
	  }
}

struct connection_defines_port_context
  {
	const char		*filename;
	YYLTYPE			*locp;
	struct component	*comp;
	struct port		*port;
	char			 defined;
  };

static void
connection_defines_port(void *data, void *context)
{
	struct connection *conn;
	struct connection_defines_port_context *ctx;
	char defined;

	conn = (struct connection *)data;
	ctx = (struct connection_defines_port_context *)context;

	defined = ((conn->output.c.p.port == ctx->port)
		   && (conn->output.c.p.component == ctx->comp));
	if (((ctx->port->type == port_type_w)
	     || (ctx->port->type == port_type_k))
	    && (conn->input.type == conn_elem_type_port))
		defined = defined
			  || ((conn->input.c.p.port == ctx->port)
			      && (conn->input.c.p.component == ctx->comp));

	if (!defined)
		return;

	if (ctx->defined)
	  {
		fprintf(stderr, "%s:%d:%d: error: port `%s%s%s' is connected "
			"more than once\n", ctx->filename,
			ctx->locp->first_line, ctx->locp->first_column,
			(ctx->comp != NULL) ? ctx->comp->id : "",
			(ctx->comp != NULL) ? "." : "", ctx->port->id);
		exit(EXIT_FAILURE);
	  }

	ctx->defined = 1;
}

struct output_port_defined_context
  {
	const char	*filename;
	YYLTYPE		*locp;
	list_t		 connections;
  };

static void
output_port_defined(void *data, void *context)
{
	struct port *port;
	struct output_port_defined_context *ctx;
	struct connection_defines_port_context c_ctx;

	port = (struct port *)data;
	ctx = (struct output_port_defined_context *)context;

	if (port->type == port_type_input)
		return;

	c_ctx.filename = ctx->filename;
	c_ctx.locp = ctx->locp;
	c_ctx.comp = NULL;
	c_ctx.port = port;
	c_ctx.defined = 0;
	list_for_each(ctx->connections, connection_defines_port, &c_ctx);

	if (!c_ctx.defined)
	  {
		fprintf(stderr, "%s:%d:%d: error: the output of port `%s' was "
			"not defined\n", ctx->filename, ctx->locp->first_line,
			ctx->locp->first_column, port->id);
		exit(EXIT_FAILURE);
	  }
}

struct component_single_conn_context
  {
	const char		*filename;
	YYLTYPE			*locp;
	struct component	*comp;
	list_t			 connections;
  };

static void
port_single_conn(void *data, void *context)
{
	struct port *port;
	struct component_single_conn_context *ctx;
	struct connection_defines_port_context c_ctx;

	port = (struct port *)data;
	ctx = (struct component_single_conn_context *)context;

	if (port->type == port_type_output)
		return;

	c_ctx.filename = ctx->filename;
	c_ctx.locp = ctx->locp;
	c_ctx.comp = ctx->comp;
	c_ctx.port = port;
	c_ctx.defined = 0;
	list_for_each(ctx->connections, connection_defines_port, &c_ctx);	
}

static void
component_single_conn(void *data, void *context)
{
	struct component *comp;
	struct component_single_conn_context *ctx;

	comp = (struct component *)data;
	ctx = (struct component_single_conn_context *)context;

	ctx->comp = comp;

	list_for_each(comp->type.t.is_macro
		      ? ((struct system *)comp->type.t.p)->ports
		      : ((struct block *)comp->type.t.p)->ports,
		      port_single_conn, ctx);
}

static void
system_bind(struct system *system, list_t ports, list_t components,
	    list_t connections, const char *filename, YYLTYPE *locp)
{
	struct component *comp;
	struct component_bind_type_context c_ctx;
	struct connection_bind_context cb_ctx;
	struct output_port_defined_context o_ctx;
	struct component_single_conn_context s_ctx;

	system->ports = ports;
	system->components = components;
	system->connections = connections;

	/* ports and components to have different ids */
	comp = list_find(components, component_cmp_ports_id, ports);
	if (comp != NULL)
	  {
		fprintf(stderr, "%s:%d:%d: error: id `%s' was already used for "
			"a port\n", filename, locp->first_line,
			locp->first_column, comp->id);
		exit(EXIT_FAILURE);
	  }

	/* bind component types */
	c_ctx.filename = filename;
	c_ctx.locp = locp;
	list_for_each(components, component_bind_type, &c_ctx);

	/* bind connections */
	cb_ctx.filename = filename;
	cb_ctx.locp = locp;
	cb_ctx.ports = ports;
	cb_ctx.components = components;
	list_for_each(connections, connection_bind, &cb_ctx);

	/* all external outputs/external physical ports defined (once) */
	o_ctx.filename = filename;
	o_ctx.locp = locp;
	o_ctx.connections = connections;
	list_for_each(ports, output_port_defined, &o_ctx);

	/* 1 connection per internal input/physical port */
	s_ctx.filename = filename;
	s_ctx.locp = locp;
	s_ctx.comp = NULL;
	s_ctx.connections = connections;
	list_for_each(components, component_single_conn, &s_ctx);
}

%}

%union {
	unsigned long		 l;
	double			 val;
	char			*str;
	list_t			 list;
	struct const_v		 const_v;
	struct ext_func		 ext_func;
	struct block		 block;
	struct system		 system;
	struct stmt		 stmt;
	expr_t			 expr;
	struct component	 component;
	struct connection	 connection;
	struct conn_elem	 conn_elem;
	struct port		 port;
	enum port_type		 port_type;
	enum port_sync		 port_sync;
}

%token		IMPORT CONST EXT_FUNCTION BLOCK MACRO SYSTEM SYNC ASYNC INPUT
		OUTPUT W_PORT K_PORT IN OUT SAMPLE_RATE SEMICOLON LBRACE RBRACE
		DOT COMA EQUALS LPAR RPAR LSBRACKET RSBRACKET
%token <str>	ID STRING
%token <l>	INTEGER
%token <val>	FLOAT

%left		PLUS MINUS
%left		ASTERISK SLASH
%left		NEG POS

%type <const_v>		const
%type <ext_func>	ext_func
%type <block>		block
%type <block>		block_code
%type <list>		stmts
%type <stmt>		stmt
%type <expr>		expr
%type <list>		expr_list
%type <system>		macro
%type <system>		macro_code
%type <system>		system
%type <system>		system_code
%type <list>		components
%type <component>	component
%type <list>		connections
%type <connection>	connection
%type <conn_elem>	conn_elem
%type <val>		sign_value
%type <list>		ports
%type <port>		port
%type <port_type>	port_type
%type <list>		sys_ports
%type <port>		sys_port
%type <port_sync>	sys_port_sync
%type <val>		value

%defines "src/parser.tab.h"
%output "src/parser.tab.c"

%define api.pure
%locations
%parse-param {yyscan_t yyscanner}
%parse-param {const char *filename}
%lex-param {yyscan_t yyscanner}

%%

file:		imports defs;

imports:	/* empty */
		| imports import;

import:		IMPORT ID SEMICOLON	{ parser_import($2); }
		;

defs:		/* empty */
		| defs const
		  {
			if (list_find(parser_consts, const_cmp_id, $2.id)
			    != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a constant "
					"with id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			list_append_copy(parser_consts, &$2, sizeof($2));
		  }
		| defs ext_func
		  {
			if (list_find(parser_ext_funcs, ext_func_cmp_id, $2.id)
			    != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a function "
					"with id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			list_append_copy(parser_ext_funcs, &$2, sizeof($2));
		  }
		| defs block
		  {
			if (list_find(parser_blocks, block_cmp_id, $2.id)
			    != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a block with "
					"id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			list_append_copy(parser_blocks, &$2, sizeof($2));
		  }
		| defs macro
		  {
			if (list_find(parser_macros, system_cmp_id, $2.id)
			    != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a macro with "
					"id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			list_append_copy(parser_macros, &$2, sizeof($2));
		  }
		| defs system
		  {
			if (list_find(parser_systems, system_cmp_id, $2.id)
			    != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a system "
					"with id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			list_append_copy(parser_systems, &$2, sizeof($2));
		  }
		;

const:		CONST ID EQUALS sign_value SEMICOLON
		  {
			$$.id = $2;
			$$.value = $4;
		  }
		;

ext_func:	EXT_FUNCTION ID EQUALS ID COMA ID COMA INTEGER COMA STRING COMA
		STRING SEMICOLON
		  {
			if ($8 == 0)
			  {
				fprintf(stderr, "%s:%d:%d: error: function "
					"`%s' must take at least one "
					"argument\n", filename, @2.first_line,
					@2.first_column, $2);
				exit(EXIT_FAILURE);
			  }

			$$.id = $2;
			$$.f_id = $4;
			$$.d_id = $6;
			$$.n_args = $8;
			$$.include = $10;
			$$.lib = $12;
		  }
		;

block:		BLOCK ID LBRACE block_code RBRACE
		  {
			$$ = $4;
			$$.id = $2;
		  }
		;

block_code:	ports stmts
		  {
			struct port_output_defined_context p_ctx;
			struct stmt_bind_context s_ctx;

			/* all output ports defined */
			p_ctx.filename = filename;
			p_ctx.locp = &(@1);
			p_ctx.stmts = $2;
			list_for_each($1, port_output_defined, &p_ctx);

			/* bind statements */
			s_ctx.filename = filename;
			s_ctx.locp = &(@1);
			s_ctx.ports = $1;
			list_for_each($2, stmt_bind, &s_ctx);
			
			$$.id = NULL;
			$$.ports = $1;
			$$.stmts = $2;
		  }
		;

stmts:		stmt
		  {
			$$ = list_new();
			list_append_copy($$, &($1), sizeof($1));
		  }
		| stmts stmt
		  {
			if (list_find($1, stmt_cmp_output, &$2) != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: double "
					"statement for `%s%s'\n",
					filename, @1.first_line,
					@1.first_column, $2.out.id.id,
					$2.out.id.out ? ".out" : "");
				exit(EXIT_FAILURE);
			  }
			$$ = $1;
			list_append_copy($$, &($2), sizeof($2));
		  }
		;

stmt:		ID EQUALS expr SEMICOLON
		  {
			$$.out.id.id = $1;
			$$.out.id.out = 0;
			$$.expr = $3;
		  }
		| ID DOT OUT EQUALS expr SEMICOLON
		  {
			$$.out.id.id = $1;
			$$.out.id.out = 1;
			$$.expr = $5;
		  }
		;

expr:		value
		  {
			$$ = expr_new_value($1);
		  }
		| SAMPLE_RATE
		  {
			$$ = expr_new_sample_rate();
		  }
		| ID
		  {
			$$ = expr_new_signal($1, 0, NULL, NULL);
		  }
		| ID DOT IN
		  {
			$$ = expr_new_signal($1, 1, NULL, NULL);
		  }
		| ID LSBRACKET expr RSBRACKET
		  {
			$$ = expr_new_signal($1, 0, $3, NULL);
		  }
		| ID DOT IN LSBRACKET expr RSBRACKET
		  {
			$$ = expr_new_signal($1, 1, $5, NULL);
		  }
		| ID LSBRACKET expr COMA expr RSBRACKET
		  {
			$$ = expr_new_signal($1, 0, $3, $5);
		  }
		| ID DOT IN LSBRACKET expr COMA expr RSBRACKET
		  {
			$$ = expr_new_signal($1, 1, $5, $7);
		  }
		| ID LPAR expr_list RPAR
		  {
			$$ = expr_new_call($1, $3);
		  }
		| expr PLUS expr
		  {
			expr_push_add($1, $3);
			$$ = $1;
		  }
		| expr MINUS expr
		  {
			expr_push_sub($1, $3);
			$$ = $1;
		  }
		| expr ASTERISK expr
		  {
			expr_push_mul($1, $3);
			$$ = $1;
		  }
		| expr SLASH expr
		  {
			expr_push_div($1, $3);
			$$ = $1;
		  }
		| PLUS expr %prec POS
		  {
			if (expr_top_is_sign($2))
			  {
				fprintf(stderr, "%s:%d:%d: error: multiple "
					"sign operators in expression\n",
					filename, @1.first_line,
					@1.first_column);
				exit(EXIT_FAILURE);
			  }
			expr_push_plus($2);
			$$ = $2;
		  }
		| MINUS expr %prec NEG
		  {
			if (expr_top_is_sign($2))
			  {
				fprintf(stderr, "%s:%d:%d: error: multiple "
					"sign operators in expression\n",
					filename, @1.first_line,
					@1.first_column);
				exit(EXIT_FAILURE);
			  }
			expr_push_minus($2);
			$$ = $2;
		  }
		| LPAR expr RPAR
		  {
			$$ = $2;
		  }
		;

expr_list:	expr
		  {
			$$ = list_new();
			list_append($$, $1);
		  }
		| expr_list COMA expr
		  {
			$$ = $1;
			list_append($$, $3);
		  }
		;

macro:		MACRO ID LBRACE macro_code RBRACE
		  {
			$$ = $4;
			$$.id = $2;
		  }
		;

macro_code:	ports components connections
		  {
			$$.id = NULL;
			system_bind(&$$, $1, $2, $3, filename, &(@1));
		  }
		;

system:		SYSTEM ID LBRACE system_code RBRACE
		  {
			$$ = $4;
			$$.id = $2;
		  }
		;

system_code:	sys_ports components connections
		  {
			$$.id = NULL;
			system_bind(&$$, $1, $2, $3, filename, &(@1));
		  }
		;

components:	/* empty */
		  {
			$$ = list_new();
		  }
		| components component
		  {
			if (list_find($1, component_cmp_id, $2.id) != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a component "
					"with id `%s' was already specified\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			$$ = $1;
			list_append_copy($$, &($2), sizeof($2));
		  }
		;

component:	ID ID SEMICOLON
		  {
			$$.type.id = $1;
			$$.id = $2;
		  }
		;

connections:	connection
		  {
			$$ = list_new();
			list_append_copy($$, &($1), sizeof($1));
		  }
		| connections connection
		  {
			if (list_find($1, connection_cmp, &$2) != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: connection "
					"already specified\n", filename,
					@2.first_line, @2.first_column);
				exit(EXIT_FAILURE);
			  }
			$$ = $1;
			list_append_copy($$, &($2), sizeof($2));
		  }
		;

connection:	conn_elem EQUALS conn_elem SEMICOLON
		  {
			if ((($1.c.id.component == NULL)
			     && ($1.type == conn_elem_type_in))
			    || (($1.c.id.component != NULL)
				&& ($1.type == conn_elem_type_out))
			    || (($3.c.id.component == NULL)
			        && ($3.type == conn_elem_type_out))
			    || (($3.c.id.component != NULL)
				&& ($3.type == conn_elem_type_in))
			    || ($1.type == conn_elem_type_value)
			    || (($3.type == conn_elem_type_value)
			        && ((($1.c.id.component == NULL)
			             && ($1.type == conn_elem_type_in))
			            || (($1.c.id.component != NULL)
			                && ($1.type == conn_elem_type_out)))))
			  {
				fprintf(stderr, "%s:%d:%d: error: invalid "
					"connection\n", filename, @1.first_line,
					@1.first_column);
				exit(EXIT_FAILURE);
			  }
			$$.output = $1;
			$$.input = $3;
		  }
		;

conn_elem:	ID
		  {
			$$.type = conn_elem_type_port;
			$$.c.id.component = NULL;
			$$.c.id.port = $1;
		  }
		| ID DOT IN
		  {
			$$.type = conn_elem_type_in;
			$$.c.id.component = NULL;
			$$.c.id.port = $1;
		  }
		| ID DOT OUT
		  {
			$$.type = conn_elem_type_out;
			$$.c.id.component = NULL;
			$$.c.id.port = $1;
		  }
		| ID DOT ID
		  {
			$$.type = conn_elem_type_port;
			$$.c.id.component = $1;
			$$.c.id.port = $3;
		  }
		| ID DOT ID DOT IN
		  {
			$$.type = conn_elem_type_in;
			$$.c.id.component = $1;
			$$.c.id.port = $3;
		  }
		| ID DOT ID DOT OUT
		  {
			$$.type = conn_elem_type_out;
			$$.c.id.component = $1;
			$$.c.id.port = $3;
		  }
		| sign_value
		  {
			$$.type = conn_elem_type_value;
			$$.c.value = $1;
		  }
		;

ports:		port
		  {
			$$ = list_new();
			list_append_copy($$, &($1), sizeof($1));
		  }
		| ports port
		  {
			if (list_find($1, port_cmp_id, $2.id) != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a port with "
					"id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			$$ = $1;
			list_append_copy($$, &($2), sizeof($2));
		  }
		;

port:		port_type ID SEMICOLON
		  {
			$$.sync = port_sync_unknown;
			$$.type = $1;
			$$.id = $2;
		  }
		;

sys_ports:	sys_port
		  {
			$$ = list_new();
			list_append_copy($$, &($1), sizeof($1));
		  }
		| sys_ports sys_port
		  {
			if (list_find($1, port_cmp_id, $2.id) != NULL)
			  {
				fprintf(stderr, "%s:%d:%d: error: a port with "
					"id `%s' was already defined\n",
					filename, @2.first_line,
					@2.first_column, $2.id);
				exit(EXIT_FAILURE);
			  }
			$$ = $1;
			list_append_copy($$, &($2), sizeof($2));
		  }
		;

sys_port:	sys_port_sync port_type ID SEMICOLON
		  {
			if (($2 == port_type_w) || ($2 == port_type_k))
			  {
				fprintf(stderr, "%s:%d:%d: error: systems "
					"cannot expose physical ports\n",
					filename, @2.first_line,
					@2.first_column);
				exit(EXIT_FAILURE);
			  }
			$$.sync = $1;
			$$.type = $2;
			$$.id = $3;
		  }
		;

sys_port_sync:	SYNC		{ $$ = port_sync_sync;  }
		| ASYNC		{ $$ = port_sync_async; }
		;

port_type:	INPUT		{ $$ = port_type_input;  }
		| OUTPUT	{ $$ = port_type_output; }
		| W_PORT	{ $$ = port_type_w;      }
		| K_PORT	{ $$ = port_type_k;      }
		;

sign_value:	value		{ $$ = $1;  }
		| PLUS value	{ $$ = $2;  }
		| MINUS value	{ $$ = -$2; }
		;

value:		INTEGER		{ $$ = (double)$1; }
		| FLOAT		{ $$ = $1; }
		;
