/* $Id: darwin.c 11744 2009-06-03 12:39:15Z bill $ */
/*
Copyright (c) 2002 Peter O'Gorman <ogorman@users.sourceforge.net>

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

/* Modified for PARI/GP by the PARI group */
#include "pari.h"
#ifndef HAS_DLOPEN

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <limits.h>
#include <mach-o/dyld.h>
#include "dlfcn.h"

#define ERR_STR_LEN 256

static void *dlsymIntern(void *handle, const char *symbol);
static const char *error(int setget, const char *str, ...);

/* Set and get the error string for use by dlerror */
static const char *error(int setget, const char *str, ...)
{
  static char errstr[ERR_STR_LEN];
  static int err_filled = 0;
  const char *retval;
  va_list arg;
  if (setget == 0)
  {
    va_start(arg, str);
    strncpy(errstr, "dlsimple: ", ERR_STR_LEN);
    vsnprintf(errstr + 10, ERR_STR_LEN - 10, str, arg);
    va_end(arg);
    err_filled = 1;
    retval = NULL;
  }
  else
  {
    retval = err_filled? errstr: NULL;
    err_filled = 0;
  }
  return retval;
}

/* dlopen */
void *dlopen(const char *path, int mode)
{
  void *module = 0;
  NSObjectFileImage ofi = 0;
  NSObjectFileImageReturnCode ofirc;
  static int (*make_private_module_public) (NSModule module) = 0;
  unsigned int flags =  NSLINKMODULE_OPTION_RETURN_ON_ERROR | NSLINKMODULE_OPTION_PRIVATE;

  /* If we got no path, the app wants the global namespace, use -1 as the marker
     in this case */
  if (!path)
    return (void *)-1;

  /* Create the object file image, works for things linked with the -bundle arg to ld */
  ofirc = NSCreateObjectFileImageFromFile(path, &ofi);
  switch (ofirc)
  {
    case NSObjectFileImageSuccess:
      /* It was okay, so use NSLinkModule to link in the image */
      if (!(mode & RTLD_LAZY)) flags += NSLINKMODULE_OPTION_BINDNOW;
      module = NSLinkModule(ofi, path,flags);
      /* Don't forget to destroy the object file image, unless you like leaks */
      NSDestroyObjectFileImage(ofi);
      /* If the mode was global, then change the module, this avoids
         multiply defined symbol errors to first load private then make
         global. Silly, isn't it. */
      if ((mode & RTLD_GLOBAL))
      {
        if (!make_private_module_public)
        {
          _dyld_func_lookup("__dyld_NSMakePrivateModulePublic",
        (void**)&make_private_module_public);
        }
        make_private_module_public((NSModule)module);
      }
      break;
    case NSObjectFileImageInappropriateFile:
      /* It may have been a dynamic library rather than a bundle, try to load it */
      module = (void *)NSAddImage(path, NSADDIMAGE_OPTION_RETURN_ON_ERROR);
      break;
    case NSObjectFileImageFailure:
      error(0,"Object file setup failure :  \"%s\"", path);
      return 0;
    case NSObjectFileImageArch:
      error(0,"No object for this architecture :  \"%s\"", path);
      return 0;
    case NSObjectFileImageFormat:
      error(0,"Bad object file format :  \"%s\"", path);
      return 0;
    case NSObjectFileImageAccess:
      error(0,"Can't read object file :  \"%s\"", path);
      return 0;
  }
  if (!module)
    error(0, "Can not open \"%s\"", path);
  return module;
}

static int
is_mach_header(void *handle)
{ /* Check for both possible magic numbers depending on x86/ppc byte order */
  return ((((struct mach_header *)handle)->magic == MH_MAGIC) ||
          (((struct mach_header *)handle)->magic == MH_CIGAM));
}

/* used by dlsym to find the symbol */
void *dlsymIntern(void *handle, const char *symbol)
{
  NSSymbol nssym = NULL;
  if (handle == (void *)-1)
  { /* Global context */
    if (NSIsSymbolNameDefined(symbol))
      nssym = NSLookupAndBindSymbol(symbol);
  }
  else
  {
    if (is_mach_header(handle))
    { /* library */
      if (NSIsSymbolNameDefinedInImage((struct mach_header *)handle, symbol))
        nssym = NSLookupSymbolInImage((struct mach_header *)handle, symbol,
                        NSLOOKUPSYMBOLINIMAGE_OPTION_BIND
                        | NSLOOKUPSYMBOLINIMAGE_OPTION_RETURN_ON_ERROR);
    }
    else /* bundle */
      nssym = NSLookupSymbolInModule((NSModule)handle, symbol);
  }
  if (!nssym)
  {
    error(0, "Symbol \"%s\" Not found", symbol);
    return NULL;
  }
  return NSAddressOfSymbol(nssym);
}

const char *dlerror(void)
{
  return error(1, (char *)NULL);
}

int dlclose(void *handle)
{
  if (is_mach_header(handle))
  {
    error(0, "Can't remove dynamic libraries on darwin");
    return 0;
  }
  if (!NSUnLinkModule((NSModule)handle, 0))
  {
    error(0, "unable to unlink module %s", NSNameOfModule((NSModule)handle));
    return 1;
  }
  return 0;
}

/* dlsym, prepend the underscore and call dlsymIntern */
void *dlsym(void *handle, const char *symbol)
{
  static char undersym[257];  /* Saves calls to malloc(3) */
  int sym_len = strlen(symbol);
  void *value = NULL;
  char *malloc_sym = NULL;

  if (sym_len < 256)
  {
    snprintf(undersym, 256, "_%s", symbol);
    value = dlsymIntern(handle, undersym);
  }
  else
  {
    malloc_sym = (char*)malloc(sym_len + 2);
    if (malloc_sym)
    {
      sprintf(malloc_sym, "_%s", symbol);
      value = dlsymIntern(handle, malloc_sym);
      pari_free(malloc_sym);
    }
    else
      error(0, "Unable to allocate memory");
  }
  return value;
}

#endif
