///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file SnappingProvider.h 
 * \brief Contains the definition of the Core::SnappingProvider class and
 *        some associated snapping classes. 
 */

#ifndef __OVITO_SNAPPING_PROVIDER_H
#define __OVITO_SNAPPING_PROVIDER_H

#include <core/Core.h>
#include <core/scene/animation/TimeInterval.h>

namespace Core {

class Viewport;			// defined in Viewport.h
class SnappingContext;	// defined below

/**
 * \brief Abstract base class for snapping provider plugins.
 * 
 * \author Alexander Stukowski
 * \sa SnappingManager
 */
class CORE_DLLEXPORT SnappingProvider : public PluginClass
{
protected:
	
	/// \brief Default constructor.
	SnappingProvider() : PluginClass() {}
	
public:

	/// \brief This methods computes hits on scene objects and records them using the
	///        given context object.
	/// \param context Contains all information about the current snapping operation and
	///                also takes the results.
	virtual void snap(SnappingContext& context) = 0;

private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(SnappingProvider)
};

/**
 * \brief Encapsulates a marker for a SnappingRecord.
 * 
 * \author Alexander Stukowski
 * \sa SnappingProvider
 */
class CORE_DLLEXPORT SnappingMarker 
{
public:
    /// \brief Virtual no-op destructor.
	virtual ~SnappingMarker() {}

	/// \brief Renders the marker for the given world point in the viewport.
	/// \param vp The viewport in which the marker should be rendered.
	/// \param worldPoint The orginal snapping point in world coordinates. 
	virtual void render(Viewport& vp, const Point3& worldPoint) = 0;
};

/**
 * \brief Contains the data of one snapping hit record.
 * 
 * Instances of this class are created by the SnappingContext::recordHit() method.
 * 
 * \author Alexander Stukowski
 */
class SnappingRecord 
{
public:
	
    /// The world coordinate of the snap point.
	Point3 worldPoint;
	
	/// The squared distance of the snap point from the cursor position in pixels.
	int squaredDistance;
	
	/// The snapping marker for this hit.
	/// This is used to visualize the hit in the viewports.
	shared_ptr<SnappingMarker> marker;

	/// \brief Returns whether this is a valid record.
	/// \return \c true if marker != \c NULL.
	bool isValid() const { return marker.get() != NULL; }

	/// \brief Renders the SnappingMarker in the given viewport.
	/// \param vp The viewport to render into.
	/// 
	/// If this record is not valid then this method does nothing. 
	void render(Viewport* vp) const;
};

/**
 * \brief This class contains information about the current snapping operation and
 *        records hits from the SnappingProvider implementations.
 * 
 * \author Alexander Stukowski
 * \sa SnappingProvider::snap()
 */
class CORE_DLLEXPORT SnappingContext : public QObject
{
	Q_OBJECT
	
public:
	
	/// \brief Initializes the context object.
	/// \param viewport The viewport in which snapping should be performed.
	/// \param time The current animation time.
	/// \param mousePos The mouse position in window coordinates.
	/// \param snappingStrength The maximum distance between the mouse cursor and the projected snapping site.
	SnappingContext(Viewport* viewport, TimeTicks time, const Point2I& mousePos, int snappingStrength) :
	  _viewport(viewport), _snapTime(time), _mousePos(mousePos), _squaredSnappingStrength(snappingStrength*snappingStrength) {}

	/// \brief Gets the viewport to use for snapping.
	/// \return The viewport in which snapping should be performed.
	Viewport* viewport() const { return _viewport; }

	/// \brief Gets the scene time to use for snapping.
	/// \return The current animation time.
	TimeTicks time() const { return _snapTime; }

	/// \brief Gets the position of the mouse in the viewport.
	/// \return The mouse position relative to the viewport window.
	const Point2I& mousePos() const { return _mousePos; }

	/// \brief Records a new hit during the snapping process.
	/// \param worldPoint The coordinates of the snapped point in world space.
	/// \param marker A marker that should be used to visualize the snap point in the viewports.
	///
	/// This method should be called from an implementation of the virtual
	/// SnappingProvider::snap() method to record a hit. The SnappingContext then
	/// finds the best hit that is closest to mouse position.
	///
	/// \sa bestHit()
	void recordHit(const Point3& worldPoint, const shared_ptr<SnappingMarker>& marker);

	/// \brief Returns whether at least one valid hit has been recorded.
	/// \return \c true if recordHit() has been called at least once and bestHit()
	///         returns a valid record; \c false otherwise.
	/// \sa bestHit()
	bool hasValidHit() const { return _bestHit.isValid(); }

	/// \brief Returns the best hit found during the snapping process.
	/// \sa hasValidHit()
	const SnappingRecord& bestHit() const { return _bestHit; }

private:

	/// The current viewport to use for snapping.
	Viewport* _viewport;

	/// The scene time to use for snapping.
	TimeTicks _snapTime;

	/// The position of the mouse cursor.
	Point2I _mousePos;

	/// Contains the maximum squared snapping distance.
	int _squaredSnappingStrength;

	/// The best hit so far.
	SnappingRecord _bestHit;
};

};

#endif // __OVITO_SNAPPING_PROVIDER_H
