///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_XFORM_MODE_H
#define __OVITO_XFORM_MODE_H

#include <core/Core.h>
#include <core/scene/SceneNode.h>
#include "ViewportInputHandler.h"
#include <core/undo/UndoManager.h>

namespace Core {

class PickRegion;		// defined in PickRegion.h

/******************************************************************************
* Base class for selection, move, rotate and scale modes.
******************************************************************************/
class CORE_DLLEXPORT XFormMode : public SimpleInputHandler
{
	Q_OBJECT
public:

	/// Returns the activation behaviour of this input handler.
	InputHandlerType handlerActivationType() { return EXCLUSIVE; }

	/// Gets the current cursor of this viewport mode.
	virtual QCursor getCursor() {
		if(showCursor && xformCursor.get()) 
			return *xformCursor;
		else
			return SimpleInputHandler::getCursor();
	}

protected:

	/// Default constructor.
	XFormMode() : currentOperation(NULL), xformOperation(NULL), cursorInitialized(false), showCursor(false), rubberBand(NULL) {}

	/// Initializes the handler on first activation.
	void onActivated();

	/// Is called when the user presses the left mouse button.
	void onMousePressed(QMouseEvent* event);

	/// Is called when the user releases the left mouse button.
	void onMouseReleased(QMouseEvent* event);

	/// Is called when the user moves the mouse while 
	/// the operation is active.
	void onMouseDrag(QMouseEvent* event);

	/// Is called when the user moves the mouse while
	/// the operation is not active.
	void onMouseFreeMove(Viewport& vp, QMouseEvent* event);

	/// Is called when the user aborts the operation by clicking the right
	/// mouse button or activating another input mode. 
	void onAbort();

	/// Is called when the user finishes the operation. 
	void onFinish();

protected:

	///////////////////////////// Overridables //////////////////////////////////

	/// Is called when the xform operation begins.
	virtual void startXForm() {}

	/// Is called during the operation.
	virtual void doXForm() {}

	/// Returns true if this is a move, rotate of scale mode.
	/// Returns false if it is a selection mode.
	virtual bool isTransformationMode() = 0;

	/// Returns the display name of this xform mode.
	/// This name will be shown in the Undo menu item.
	virtual QString displayName() = 0;

	/// Returns the cursor to display when the mouse is
	/// over an object in the viewports.
	virtual QString cursorFilename() { return QString(); }

	/// Applies the current transformation to the given nodes.
	virtual void applyXForm(const QVector<SceneNode*>& nodeSet, FloatType multiplier) {}

	/// Paints the rubber band rectangle for selecting multiple objects.
	void showSelectionRect();

	/// Removes the rubber band rectangle for selecting multiple objects.
	void hideSelectionRect();
	
protected:

	/// The initial click point
	Point2I clickPoint;

	/// The current mouse position
	Point2I currentPoint;

	/// Flag that indicates if the selection sould be cleared when the user releases the mouse button.
	bool clearSelection;

	/// Indicates if drag mode is active
	bool isDragging;

	/// Indicates if we are currently selecting objects
	bool isSelecting;

	/// The current undoable record during the operation.
	CompoundOperation* currentOperation;

	/// The current xform operation.
	CompoundOperation* xformOperation;

	/// The mouse cursor of this xform mode.
	auto_ptr<QCursor> xformCursor;
	
	/// Indicates that the cursor has been loaded.
	bool cursorInitialized;
	
	/// Indicates that the mouse cursor is over an object.
	bool showCursor;
	
	/// The rubber band used to display the selection rectangle.
	QRubberBand* rubberBand;
};

};

#endif // __OVITO_XFORM_MODE_H
