///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportPanel.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/XFormMode.h>
#include <core/scene/animation/AnimManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/data/DataSetManager.h>
#include <core/utilities/PathManager.h>
#include <core/gui/ApplicationManager.h>

namespace Core {

/******************************************************************************
* Initializes the handler on first activation.
******************************************************************************/
void XFormMode::onActivated()
{
	SimpleInputHandler::onActivated();

	if(!cursorInitialized && APPLICATION_MANAGER.guiMode()) {
		cursorInitialized = true;
		if(cursorFilename().isEmpty() == false) {
			xformCursor.reset(new QCursor(QPixmap(cursorFilename())));
		}
	}
}

/******************************************************************************
* Is called when the user presses the left mouse button.
******************************************************************************/
void XFormMode::onMousePressed(QMouseEvent* event)
{
    OVITO_ASSERT(currentOperation == NULL);
    
	clickPoint = event->pos();

	// Setup pick region.
	PointPickRegion pickRegion(clickPoint);

	// Hit test scene nodes.	
    CHECK_POINTER(SceneRenderer::activeRenderer());
	QVector<SceneNode*> pickedNodes = SceneRenderer::activeRenderer()->pickNodes(ANIM_MANAGER.time(), viewport(), pickRegion);

	clearSelection = false;
	isDragging = false;
	isSelecting = false;
	SNAPPING_MANAGER.clearLastSnapPoint();

	if(pickedNodes.empty()) {
		// Clear selection when mouse button is released.
		if(!event->modifiers().testFlag(Qt::ControlModifier))
			clearSelection = true;
	}
	else {
		// Begin recording operation.
		currentOperation = UNDO_MANAGER.beginCompoundOperation(displayName());
		
		// Check if some of the picked nodes are already selected.
		bool containsSelectedNodes = false;
		QVector<SceneNode*>::const_iterator previousSelection = pickedNodes.constBegin();
		for(; previousSelection != pickedNodes.constEnd(); ++previousSelection) {
			if((*previousSelection)->isSelected()) {
				++previousSelection;
				containsSelectedNodes = true;
				break;
			}
		}		
		if(previousSelection == pickedNodes.constEnd())
			previousSelection = pickedNodes.constBegin();
		
		if(containsSelectedNodes == false) {			
			// Select the closest node from the set of picked nodes.			
			if(event->modifiers().testFlag(Qt::ControlModifier))
				DATASET_MANAGER.currentSelection()->add(pickedNodes.front());
			else
				DATASET_MANAGER.currentSelection()->setNode(pickedNodes.front());
		}
		else {
			if(event->modifiers().testFlag(Qt::ControlModifier) && (*previousSelection)->isSelected() == false) {
				// Select the node from the set of picked nodes that comes after the previouly selected node.
				DATASET_MANAGER.currentSelection()->setNode(*previousSelection);
			}
		}
	}
}

/******************************************************************************
* Is called when the user releases the left mouse button.
******************************************************************************/
void XFormMode::onMouseReleased(QMouseEvent* event)
{
	if(isDragging && isSelecting) clearSelection = !event->modifiers().testFlag(Qt::ControlModifier);
	onFinish();
}

/******************************************************************************
* Is called when the user moves the mouse while the operation is active.
******************************************************************************/
void XFormMode::onMouseDrag(QMouseEvent* event)
{
	if(!isDragging) {
		// If delta>epsilon then start dragging.
		if(abs(clickPoint.x() - event->x()) > 5 || abs(clickPoint.y() - event->y()) > 5) {
			isDragging = true;
			currentPoint = event->pos();
			if(!isTransformationMode() || clearSelection) {
				isSelecting = true;
				showSelectionRect();
			}
			else {
				// Start xform.
				xformOperation = UNDO_MANAGER.beginCompoundOperation(displayName());
				startXForm();
			}
		}
	}
	else {
		if(isSelecting) {
			// Update rubber band rect
			currentPoint = event->pos();
			showSelectionRect();
		}
		else {
			// Do xform.
			xformOperation->clear();
			UNDO_MANAGER.endCompoundOperation();
			xformOperation = UNDO_MANAGER.beginCompoundOperation(displayName());
			currentPoint = event->pos();
			doXForm();
			VIEWPORT_MANAGER.processViewportUpdates();
		}
	}

}

/******************************************************************************
* Is called when the user moves the mouse while the operation is not active.
******************************************************************************/
void XFormMode::onMouseFreeMove(Viewport& vp, QMouseEvent* event)
{
	// Change cursor on mouse over object.

	// Setup pick region.
	PointPickRegion pickRegion(event->pos());
	
	// Hit test scene nodes.
	CHECK_POINTER(SceneRenderer::activeRenderer());
	QVector<SceneNode*> pickedNodes = SceneRenderer::activeRenderer()->pickNodes(ANIM_MANAGER.time(), &vp, pickRegion);

	if(pickedNodes.empty() && showCursor) { 
		showCursor = false;
		updateCursor();
	}
	else if(!pickedNodes.empty() && !showCursor) {
		showCursor = true;
		updateCursor();
	}

	if(isTransformationMode())
		snapPreview(vp, event);
}

/******************************************************************************
* Is called when the user aborts the operation by clicking the right
* mouse button or activating another input mode. 
******************************************************************************/
void XFormMode::onAbort()
{
	// Clear rubber band rect
	if(isDragging && isSelecting) { 
		hideSelectionRect();
	}
	if(xformOperation != NULL) {
		xformOperation->clear();
		UNDO_MANAGER.endCompoundOperation();
		xformOperation = NULL;
	}
	if(currentOperation != NULL) {
		currentOperation->clear();
		UNDO_MANAGER.endCompoundOperation();
		currentOperation = NULL;
	}
	isDragging = false;
	SimpleInputHandler::onAbort();
}

/******************************************************************************
* Is called when the user finishes the operation. 
******************************************************************************/
void XFormMode::onFinish()
{
	CHECK_POINTER(viewport());

	if(isDragging && isSelecting) {
		
		// Remove rubber band.
		hideSelectionRect();
        
		// Set up pick region.
		RectanglePickRegion pickRegion(clickPoint, currentPoint);
		
		// Hit test scene nodes.
		CHECK_POINTER(SceneRenderer::activeRenderer());
		QVector<SceneNode*> pickedNodes = SceneRenderer::activeRenderer()->pickNodes(ANIM_MANAGER.time(), viewport(), pickRegion);

		if(pickedNodes.empty() == false) {
			// Begin recording operation.
			UNDO_MANAGER.beginCompoundOperation(tr("Select"));
			// Select nodes.
			if(clearSelection)
				DATASET_MANAGER.currentSelection()->setNodes(pickedNodes);
			else
				DATASET_MANAGER.currentSelection()->addAll(pickedNodes);
			// Stop recording.
			UNDO_MANAGER.endCompoundOperation();
			clearSelection = false;
		}
		else clearSelection = true;
	}
	isDragging = false;

	// End xform.
	if(xformOperation != NULL) {
		UNDO_MANAGER.endCompoundOperation();
		xformOperation = NULL;
	}
	else if(currentOperation != NULL) {
		currentOperation->setDisplayName(tr("Select"));
	}

	if(currentOperation != NULL) {
		UNDO_MANAGER.endCompoundOperation();
		currentOperation = NULL;
	}

	if(clearSelection) {
		// Begin recording operation.
		UNDO_MANAGER.beginCompoundOperation(tr("Select"));

		// Unselect all nodes
		DATASET_MANAGER.currentSelection()->clear();

		// Stop recording.
		UNDO_MANAGER.endCompoundOperation();
	}

	SimpleInputHandler::onFinish();
}

/******************************************************************************
* Paints the rubber band rectangle for selecting multiple objects.
******************************************************************************/
void XFormMode::showSelectionRect()
{
	CHECK_POINTER(viewport());
	
	if(!rubberBand) {
		rubberBand = new QRubberBand(QRubberBand::Rectangle, viewport()->container());
		rubberBand->show();
	}

	// Compute rectangle.
	QRect rect;
	rect.setX(min(clickPoint.X, currentPoint.X));
	rect.setY(min(clickPoint.Y, currentPoint.Y));
	rect.setWidth(abs(clickPoint.X - currentPoint.X));
	rect.setHeight(abs(clickPoint.Y - currentPoint.Y));
	rect.translate(viewport()->geometry().topLeft());
	
	rubberBand->setGeometry(rect.intersected(viewport()->geometry()));
}

/******************************************************************************
* Removes the rubber band rectangle for selecting multiple objects.
******************************************************************************/
void XFormMode::hideSelectionRect()
{
	if(rubberBand) {
		delete rubberBand;
		rubberBand = NULL;
	}
}

};
