///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>
#include <core/rendering/RenderSettings.h>
#include <core/rendering/RenderSettingsEditor.h>
#include <core/rendering/PluginRenderer.h>
#include <core/viewport/Viewport.h>

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(RenderSettings, RefTarget)
DEFINE_REFERENCE_FIELD(RenderSettings, PluginRenderer, "Renderer", _renderer)
DEFINE_REFERENCE_FIELD(RenderSettings, VectorController, "BackgroundColor", _backgroundColor)
SET_PROPERTY_FIELD_LABEL(RenderSettings, _renderer, "Renderer")
SET_PROPERTY_FIELD_LABEL(RenderSettings, _backgroundColor, "Background color")

/******************************************************************************
* Constructor.
* Creates an instance of the default renderer class which can be 
* accessed via the renderer() method.
******************************************************************************/
RenderSettings::RenderSettings(bool isLoading) : RefTarget(isLoading)
{
	INIT_PROPERTY_FIELD(RenderSettings, _renderer);
	INIT_PROPERTY_FIELD(RenderSettings, _backgroundColor);
	if(!isLoading) {
		// Setup default settings.
		d.renderingRangeType = CURRENT_FRAME;
		d.imageInfo.setImageWidth(640);
		d.imageInfo.setImageHeight(480);
		d.generateAlphaChannel = false;
		
		// Setup default background color.
		_backgroundColor = CONTROLLER_MANAGER.createDefaultController<VectorController>();
		setBackgroundColor(Color(1,1,1));
		
		// Create an instance of the default renderer class.
		QList<PluginClassDescriptor*> rendererClasses = PluginRenderer::availableRendererClasses();
		if(!rendererClasses.empty()) {
			setRendererClass(rendererClasses.front());
		}					
	}
}

/******************************************************************************
* Returns the class of the current renderer or NULL if there is no current renderer. 
******************************************************************************/
PluginClassDescriptor* RenderSettings::rendererClass() const
{
	return renderer() ? renderer()->pluginClassDescriptor() : NULL;
}

/******************************************************************************
* Selects the type of renderer to use for rendering. The specified class must be derived from PluginRenderer. 
* This method will create a new instance of the given renderer class and stores the new renderer in this settings object.
* When an error occurs an exception is thrown. 
******************************************************************************/
void RenderSettings::setRendererClass(PluginClassDescriptor* rendererClass)
{
	OVITO_ASSERT(rendererClass != NULL);
	OVITO_ASSERT(rendererClass->isKindOf(PLUGINCLASSINFO(PluginRenderer)));
	
	// Create a new instance of the specified class.
	PluginRenderer::SmartPtr newRenderer = static_object_cast<PluginRenderer>(rendererClass->createInstance());
	newRenderer->_renderSettings = this;
	
	// Make the new renderer the current renderer.
	_renderer = newRenderer;
}

/******************************************************************************
* Specifies whether only the current frame or the whole animation should be rendered. 
******************************************************************************/
void RenderSettings::setRenderingRangeType(RenderingRangeType mode)
{
	if(mode == d.renderingRangeType) return;
	
	d.renderingRangeType = mode;
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Sets the width of the image to be rendered in pixels. 
******************************************************************************/
void RenderSettings::setImageWidth(int width)
{
	OVITO_ASSERT_MSG(width > 0, "RenderSettings::setImageWidth()", "The image width must be positive.");
	if(width == imageWidth()) return;
	
	d.imageInfo.setImageWidth(width);
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Sets the height of the image to be rendered in pixels. 
******************************************************************************/
void RenderSettings::setImageHeight(int height)
{
	OVITO_ASSERT_MSG(height > 0, "RenderSettings::setImageHeight()", "The image height must be positive.");
	if(height == imageHeight()) return;
	
	d.imageInfo.setImageHeight(height);
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Sets the output filename of the rendered image. 
******************************************************************************/
void RenderSettings::setImageFilename(const QString& filename)
{
	if(filename == imageFilename()) return;
	
	d.imageInfo.setFilename(filename);
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Sets whether the alpha channel will be generated. 
******************************************************************************/
void RenderSettings::setGenerateAlphaChannel(bool enable)
{
	if(enable == generateAlphaChannel()) return;
	
	d.generateAlphaChannel = enable;
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Saves the class' contents to the given stream. 
******************************************************************************/
void RenderSettings::saveToStream(ObjectSaveStream& stream)
{
	RefTarget::saveToStream(stream);

	stream.beginChunk(0x449993);
	stream.writeEnum(d.renderingRangeType);
	stream << d.imageInfo;
	stream << d.generateAlphaChannel;
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream. 
******************************************************************************/
void RenderSettings::loadFromStream(ObjectLoadStream& stream)
{
	RefTarget::loadFromStream(stream);

	stream.expectChunk(0x449993);
	stream.readEnum(d.renderingRangeType);
	stream >> d.imageInfo;
	stream >> d.generateAlphaChannel;
	stream.closeChunk();
	
	if(renderer()) renderer()->_renderSettings = this;
}

/******************************************************************************
* Creates a copy of this object. 
******************************************************************************/
RefTarget::SmartPtr RenderSettings::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	RenderSettings::SmartPtr clone = static_object_cast<RenderSettings>(RefTarget::clone(deepCopy, cloneHelper));
	
	/// Copy data values.
	clone->d = this->d;
	
	/// Copy renderer.
	OVITO_ASSERT((clone->renderer() != NULL) == (renderer() != NULL));
	if(clone->renderer()) renderer()->_renderSettings = clone.get();
		
	return clone;
}

};
