///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file RefMaker.h
 * \brief Contains the definition of the Core::RefMaker class.
 */

#ifndef __OVITO_REFMAKER_H
#define __OVITO_REFMAKER_H

#include <core/Core.h>
#include <core/plugins/PluginClass.h>
#include "RefTargetMessage.h"

namespace Core {

class RefTarget;						// defined in RefTarget.h
class CustomAttributesContainer;		// defined in CustomAttributesContainer.h
class PropertyFieldDescriptor;			// defined in PropertyFieldDescriptor.h
class SingleReferenceFieldBase;		// defined in PropertyFieldDescriptor.h
class VectorReferenceFieldBase;		// defined in PropertyFieldDescriptor.h

/**
 * \brief Exception that indicates a cylic reference that is not allowed.
 *
 * This exception object is thrown by the RefMaker class when
 * a cyclic reference would be created by setting a reference field to
 * a new target.
 *
 * \author Alexander Stukowski
 * \sa RefMaker
 */
class CyclicReferenceError : public Exception
{
public:
	/// \brief Default constructor.
	CyclicReferenceError() : Exception("Cyclic Reference Error") {}
};

/**
 * \brief Base class for all objects that hold references to other objects (reference targets).
 *
 * \author Alexander Stukowski
 * \sa RefTarget
 */
class CORE_DLLEXPORT RefMaker : public PluginClass
{
protected:
	/// \brief The default constructor.
	/// \param isLoading Specifies whether this object is being created during a normal operation (\c isLoading==false)
	///                  or because it is being loaded from a ObjectLoadStream (\c isLoading==true).
	///
	/// Subclasses should initialize wheir reference fields in the constructor using
	/// the \c INIT_PROPERTY_FIELD macro.
	RefMaker(bool isLoading = false);

	/// \brief The destructor.
	///
	/// Also deletes the CustomAttributesContainer associated with this RefMaker.
	virtual ~RefMaker();

	/////////////////////// Overridable reference field event handlers ///////////////////////////////////

	/// \brief Is called when a RefTarget referenced by this RefMaker has sent a message.
	/// \param source A direct reference target of this RefMaker specifying the source of the message.
	///               Please note that this is not necessarily the sender or generator of the notification
	///               message as it is returned by RefTargetMessage::sender().
	/// \param msg The notification message received by this RefMaker.
	/// \return \c true if the message should be recursively passed on to dependents of this object;
	///         \c false if the message should not be sent to other dependents.
	///
	/// \note When this method is overridden in sub-classes then the base implementation of this method
	///       should alywas be called from the new implementation to allow the base classes to handle
	///       message from their specific reference targets.
	///
	/// The default implementation of this method does nothing and returns \c true.
	///
	/// \sa RefTarget::notifyDependents()
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg) { return true; }

	/// \brief Is called when the value of a reference field of this RefMaker changes.
	/// \param field Specifies the reference field of this RefMaker that has been changed.
	///              This is always a single reference ReferenceField.
	/// \param oldTarget The old target that was referenced by the ReferenceField. This can be \c NULL.
	/// \param newTarget The new target that is now referenced by the ReferenceField. This can be \c NULL.
	///
	/// This method can by overridden by derived classes that want to be informed when
	/// any of their reference fields are changed.
	///
	/// \note When this method is overridden in sub-classes then the base implementation of this method
	///       should alywas be called from the new implementation to allow the base classes to handle
	///       messages for their specific reference fields.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget) {}

	/// \brief Is called when a RefTarget has been added to a VectorReferenceField of this RefMaker.
	/// \param field Specifies the reference field of this RefMaker to which a new entry has been added.
	///              This is always a VectorReferenceField.
	/// \param newTarget The new target added to the list of referenced objects.
	/// \param listIndex The index into the VectorReferenceField at which the new entry has been inserted.
	///
	/// This method can by overridden by derived classes that want to be informed when
	/// a reference has been added to one of its vector reference fields.
	///
	/// \note When this method is overridden in sub-classes then the base implementation of this method
	///       should alywas be called from the new implementation to allow the base classes to handle
	///       messages for their specific reference fields.
	///
	/// \sa VectorReferenceField::push_back()
	/// \sa onRefTargetRemoved()
	virtual void onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex) {}

	/// \brief Is called when a RefTarget has been removed from a VectorReferenceField of this RefMaker.
	/// \param field Specifies the reference field of this RefMaker from which an entry has been removed.
	///              This is always a VectorReferenceField.
	/// \param oldTarget The old target that was reference before it has been removed from the vector reference field.
	/// \param listIndex The index into the VectorReferenceField at which the old entry was stored.
	///
	/// This method can by overridden by derived classes that want to be informed when
	/// a reference has been removed from one of its vector reference fields.
	///
	/// \note When this method is overridden in sub-classes then the base implementation of this method
	///       should alywas be called from the new implementation to allow the base classes to handle
	///       messages for their specific reference fields.
	///
	/// \sa VectorReferenceField::remove()
	/// \sa onRefTargetInserted()
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex) {}

	/// \brief Is called when the value of a non-animatable property field of this RefMaker has changed.
	/// \param field Specifies the property field of this RefMaker that has changed.
	///              This is always a non-animatable PropertyField.
	///
	/// This method can by overridden by derived classes that want to be informed when
	/// any of their property fields are changed.
	///
	/// \note When this method is overridden in sub-classes then the base implementation of this method
	///       should alywas be called from the new implementation to allow the base classes to handle
	///       messages for their specific property fields.
	virtual void onPropertyFieldValueChanged(const PropertyFieldDescriptor& field) {}

	/// \brief Handles a notification message from a RefTarget referenced by this Refmaker.
	/// \param source Specifies the RefTarget referenced by this RefMaker that delivered the message.
	/// \param msg The notification message.
	/// \return If \c true then the message is passed on to all dependents of this object.
	///
	/// Normaly you don't have to call or override this methods.
	/// Override onRefTargetMessage() to process notification messages sent by referenced objects.
	///
	/// \sa onRefTargetMessage()
	virtual bool processTargetNotification(RefTarget* source, RefTargetMessage* msg);

	/// \brief Stops observing a RefTarget object.
	/// \param target All references hold by the RefMaker to the this target are cleared.
	///
	/// All reference fields containing a pointer to \a target will be reset to \c NULL.
	/// If \a target is referenced in a VectorReferenceField then the item is
	/// removed from the vector field.
	///
	/// \undoable
	void clearReferencesTo(RefTarget* target);

	/// \brief Replaces all references of this RefMaker to some RefTarget with new ones.
	/// \param oldTarget Specifies which references should be replaced.
	/// \param newTarget Specifies the new target that should replace the old one.
	/// \undoable
	void replaceReferencesTo(RefTarget* oldTarget, RefTarget* newTarget);

	/// \brief Replaces all references of this RefMaker to some RefTarget with new ones.
	/// \param oldTarget Specifies which references should be replaced.
	/// \param newTarget Specifies the new target that should replace the old one.
	/// \note This is the same method as above but using a smart pointer for the parameter \a newTarget.
	/// \undoable
	template<class T>
	void replaceReferencesTo(RefTarget* oldTarget, const intrusive_ptr<T>& newTarget) {
		replaceReferencesTo(oldTarget, newTarget.get());
	}

	/// \brief Clears a reference field.
	/// \param field Sepcifies the reference field of this RefMaker to be cleared.
	///
	/// If the reference field specified by \a field is a single reference field then it is set to the value \c NULL.
	/// If it is a VectorReferenceField thne all references are removed.
	///
	/// \undoable
	void clearReferenceField(const PropertyFieldDescriptor& field);

	/// \brief Clears all references held by this RefMarker.
	///
	/// All single reference fields are set to \c NULL and all vector reference
	/// fields are cleared.
	///
	/// \undoable
	void clearAllReferences();

	/// \brief Is called by the system when a custom attribute of this RefMaker has changed in some way.
	///
	/// Sub-classes can override this method to process REFTARGET_CHANGED messages sent by reference targets
	/// stored in the CustomAttributesContainer of this RefMaker.
	///
	/// The default RefTarget implementation generates a CUSTOM_ATTRIBUTE_CHANGED message.
	///
	/// \sa CustomAttributesContainer
	/// \sa customAttributes()
	virtual void onCustomAttributesChanged() {}

	/// \brief Saves the class' contents to an output stream.
	/// \param stream The destination stream.
	///
	/// Derived classes can overwrite this virtual method to store their specific data
	/// in the output stream. The derived class \b must always call the base implementation of the saveToStream() method
	/// before it writes its own data to the stream.
	///
	/// The RefMaker implementation of this virtual method saves all
	/// references and referenced RefTarget object of this RefMaker to the output stream.
	///
	/// \sa loadFromStream()
	virtual void saveToStream(ObjectSaveStream& stream);

	/// \brief Loads the class' contents from an input stream.
	/// \param stream The source stream.
	/// \throw Exception when a parsing error has occured.
	///
	/// Derived classes can overwrite this virtual method to read their specific data
	/// from the input stream. The derived class \b must always call the loadFromStream() method
	/// of the base class before it reads its own data from the stream.
	///
	/// The RefMaker implementation of this method restores all
	/// reference fields and loads the referenced objects that had been serialized to the data stream.
	///
	/// \sa saveToStream()
	virtual void loadFromStream(ObjectLoadStream& stream);

public:

	/////////////////////////// Runtime property field access ///////////////////////////////

	/// \brief Returns the value stored in a non-animatable property field of this RefMaker object.
	/// \param field The descriptor of a property field defined by this RefMaker derived class.
	/// \return The current value of the property field.
	/// \sa PluginClassDescriptor::firstPropertyField()
	/// \sa PluginClassDescriptor::findPropertyField()
	QVariant getPropertyFieldValue(const PropertyFieldDescriptor& field) const;

	/// \brief Sets the value stored in a non-animatable property field of this RefMaker object.
	/// \param field The descriptor of a property field defined by this RefMaker derived class.
	/// \param newValue The value to be assigned to the property. The QVariant data type must match the property data type.
	/// \sa PluginClassDescriptor::firstPropertyField()
	/// \sa PluginClassDescriptor::findPropertyField()
	void setPropertyFieldValue(const PropertyFieldDescriptor& field, const QVariant& newValue);

	/////////////////////////// Runtime reference field access //////////////////////////////

	/// \brief Looks up a reference field.
	/// \param field The descriptor of a reference field defined in this RefMaker derived class.
	/// \return The field object for this RefMaker instance and the specified field.
	/// \sa getVectorReferenceField()
	/// \sa PluginClassDescriptor::firstPropertyField()
	/// \sa PluginClassDescriptor::findPropertyField()
	const SingleReferenceFieldBase& getReferenceField(const PropertyFieldDescriptor& field) const;

	/// \brief Looks up a vector reference field.
	/// \param field The descriptor of a vector reference field defined in this RefMaker derived class.
	/// \return The field object for this RefMaker instance and the specified vector field.
	/// \sa getReferenceField()
	/// \sa PluginClassDescriptor::firstPropertyField()
	/// \sa PluginClassDescriptor::findPropertyField()
	const VectorReferenceFieldBase& getVectorReferenceField(const PropertyFieldDescriptor& field) const;

	////////////////////////////// Dependencies //////////////////////////////////

	/// \brief Checks whether this RefMaker has any (direct) references to a RefTarget.
	/// \param target Specifies the reference target.
	/// \return \c true if this RefMaker object has at least one direct reference to the given
	///         RefTarget \a target; \c false otherwise.
	///
	/// \sa isReferencedBy()
	bool hasReferenceTo(RefTarget* target) const;

	/// \brief Checks whether this object is directly or indirectly referenced by the given RefMaker.
	/// \param obj The RefMaker that might hold a reference to \c this object.
	///
	/// The RefMaker base implementation always returns \a false since this class is not a RefTarget and can therefore
	/// not be referenced. RefTarget overrides this method with a more meaningful implementation.
	virtual bool isReferencedBy(const RefMaker* obj) const { return false; }

	/// \brief Returns a list of all targets this RefMaker depends on (directly as well as indirectly).
	/// \return A list of all RefTargets that are directly or indirectly referenced by this RefMaker.
	/// \note The returned list is gathered recursively.
	QSet<RefTarget*> getAllDependencies() const;

	///////////////////////////// from PluginClass ///////////////////////////////

	/// \brief Deletes this object.
	///
	/// This implementation releases all references held by this RefMaker
	/// by calling clearAllReferences().
	/// If undo recording is not enabled it then calls PluginClass::autoDeleteObject() to delete the object physically.
	virtual void autoDeleteObject();

	//////////////////////////// Custom attributes ///////////////////////////////

	/// \brief Returns the custom attributes associated with this object.
	/// \return A pointer to the container that holds all custom attributes assigned to this RefMaker.
	/// \note This method automatically creates a CustomAttributesContainer for this RefMaker
	///       if it does not own one. If you just want to check whether this RefMaker has any
	///       custom attributes then you should use hasCustomAttributes() instead to prevent
	///       accidental creation of an CustomAttributesContainer instance.
	/// \sa hasCustomAttributes(), CustomAttributesContainer
	CustomAttributesContainer* customAttributes();

	/// \brief Returns whether this object has any custom attributes associated with it.
	/// \return \c true if this object has any custom attributes associated with it.
	///         They can then be accessed via the customAttributes() method.
	/// \sa customAttributes()
	bool hasCustomAttributes() const;


private:

	/// \brief Recursive gathering function used by getAllDependencies().
	static void walkNode(QSet<RefTarget*>& nodes, const RefMaker* node);

private:

	/// The custom attributes associated with this object (can be NULL).
	intrusive_ptr<CustomAttributesContainer> _customAttributes;

private:

	friend class RefTarget;
	friend class CustomAttributesContainer;
	friend class SingleReferenceFieldBase;
	friend class VectorReferenceFieldBase;
	friend class PropertyFieldBase;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(RefMaker);
};

};

#include "PropertyFieldDescriptor.h"

#endif // __OVITO_REFMAKER_H
