///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/NumericalControllerUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(NumericalControllerUI, ReferenceParameterUI)

/******************************************************************************
* The constructor.
******************************************************************************/
NumericalControllerUI::NumericalControllerUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField, ParameterUnit* defaultParameterUnit) : 
	ReferenceParameterUI(parentEditor, refField), _spinner(NULL), _textBox(NULL), _label(NULL), _parameterUnit(defaultParameterUnit)
{
	CHECK_OBJECT_POINTER(_parameterUnit);
	
	// Look up the ParameterUnit for this parameter.
	try {
		ParameterUnit* customUnit = referenceField().parameterUnit();
		if(customUnit != NULL) _parameterUnit = customUnit;
	}
	catch(const Exception& ex) {		
		ex.showError();
	}
	
	// Create UI widgets.
	_label = new QLabel(referenceField().displayName() + ":");

	_textBox = new QLineEdit();
	
	_spinner = new SpinnerWidget();
	connect(_spinner, SIGNAL(spinnerValueChanged()), this, SLOT(OnSpinnerValueChanged()));	
	connect(_spinner, SIGNAL(spinnerDragStart()), this, SLOT(OnSpinnerDragStart()));
	connect(_spinner, SIGNAL(spinnerDragStop()), this, SLOT(OnSpinnerDragStop()));
	connect(_spinner, SIGNAL(spinnerDragAbort()), this, SLOT(OnSpinnerDragAbort()));
	_spinner->setUnit(_parameterUnit);
	_spinner->setTextBox(_textBox);
	
	// Update the displayed value when the animation time has changed.
	connect(&ANIM_MANAGER, SIGNAL(timeChanged(TimeTicks)), this, SLOT(updateUI()));
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
NumericalControllerUI::~NumericalControllerUI()
{
	// Release GUI controls. 
	delete label();
	delete spinner();
	delete textBox(); 
}

/******************************************************************************
* This method is called when parameter object has been assigned to the reference field of the editable object 
* this parameter UI is bound to. It is also called when the editable object itself has
* been replaced in the editor. This implementation of the method automatically
* disables the UI widgets when the reference field of the edited object does not contain a
* controller. 
******************************************************************************/
void NumericalControllerUI::resetUI()
{
	ReferenceParameterUI::resetUI();	
	
	if(spinner()) 
		spinner()->setEnabled(parameterObject() != NULL && isEnabled());
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void NumericalControllerUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	ReferenceParameterUI::setEnabled(enabled);
	if(spinner()) spinner()->setEnabled(parameterObject() != NULL && isEnabled());
}

/******************************************************************************
* Is called when the spinner value has changed.
******************************************************************************/
void NumericalControllerUI::OnSpinnerValueChanged()
{
	ViewportSuspender noVPUpdate;
	if(!UNDO_MANAGER.isRecording()) {
		UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));		
		updateParameterValue();
		UNDO_MANAGER.endCompoundOperation();
	}
	else {
		UNDO_MANAGER.currentCompoundOperation()->clear();
		updateParameterValue();
	}
}

/******************************************************************************
* Is called when the user begins dragging the spinner interactively.
******************************************************************************/
void NumericalControllerUI::OnSpinnerDragStart()
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));
}

/******************************************************************************
* Is called when the user stops dragging the spinner interactively.
******************************************************************************/
void NumericalControllerUI::OnSpinnerDragStop()
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user aborts dragging the spinner interactively.
******************************************************************************/
void NumericalControllerUI::OnSpinnerDragAbort()
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.currentCompoundOperation()->clear();
	UNDO_MANAGER.endCompoundOperation();
}

};

