///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/PropertiesPanel.h>
#include <core/gui/panels/CommandPanel.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/scene/objects/CreationMode.h>

namespace Core {

/******************************************************************************
* The creation page lets the user create new objects.
******************************************************************************/
class CreationCommandPage : public CommandPanelPage
{
	Q_OBJECT

public:
	/// Initializes the creation page.
    CreationCommandPage();

	/// The types of objects that can be created.
	enum ObjectClass {
		GEOMETRY = 0,
		SHAPES,
		LIGHTS,
		CAMERAS,
		NUM_OBJ_CLASSES
	};

	/// Selects the given object class.
	void setObjectClass(ObjectClass objClass);

	/// Resets the creation panel to the initial state.
	virtual void reset();

	/// Is called when the user selects another page.
	virtual void onLeave();

protected Q_SLOTS:

	/// This is called after all changes to the selection set have been completed.
	virtual void onSelectionChangeComplete(SelectionSet* newSelection);

private:

	/// This panel shows the properties of the current object being created.
	PropertiesPanel* propertiesPanel;

	/// Contains the available objects in the selected category.
    QWidget* objectTypePanel;

	/// Contains the object class selection buttons.
	QToolBar* objClassToolbar;

	/// Displays the available object categories.
	QComboBox* categoryBox;

	/// The creation mode that is currently active or NULL.
	CreationMode::SmartPtr currentMode;

	/// Contains one button per creation mode.
	QButtonGroup* createButtonGroup;

	/// The button that has been activated by the user.
	QAbstractButton* currentCreationButton;

	/// The list of actions for the object class toolbar.
	QActionGroup* objClassActions;

	/// The currently selected object class.
	ObjectClass currentObjClass;

	struct ObjectCategory {
		int identifier;
		QString displayName;
		QVector<PluginClassDescriptor*> classes;
	};

	/// Maps from Category Identifiers to categories.
    typedef map<int, ObjectCategory> CategoryList;

	/// The list of catgeories for each object class.
    CategoryList categoryLists[NUM_OBJ_CLASSES];

	/// Finds all object types provided by the installed plugins.
	void scanInstalledObjectTypes();

	/// Updates the displayed button in the object type panel.
	void rebuildObjectTypePanel();

	/// Deactivates the current creation mode.
	void endCreation();

protected Q_SLOTS:

	/// Is called when the user selects one of the object class buttons.
	void onObjectClassButton(QAction* action);

	/// Is called when the active viewport input handler has changed.
	void onInputModeChanged(ViewportInputHandler* oldMode, ViewportInputHandler* newMode);

	/// Is called when the user has clicked on one of the object creation buttons in the object type panel.
	void onCreateObjectButton(QAbstractButton* btn);

	/// Is called when the user has selected an object category.
	void onCategorySelected();
};


};
