///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/undo/UndoManager.h>
#include <core/data/DataSetManager.h>
#include <core/viewport/ViewportPanel.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/SelectionMode.h>
#include <core/viewport/input/MoveMode.h>
#include <core/viewport/input/RotationMode.h>
#include <core/viewport/input/ScalingMode.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager. 
EditingActionsHandler::EditingActionsHandler()
{
	connect(addCommandAction(ACTION_EDIT_UNDO, tr("Undo"), ":/core/main/edit_undo.png", QKeySequence::Undo), SIGNAL(triggered(bool)), this, SLOT(OnEditUndo()));
	connect(addCommandAction(ACTION_EDIT_REDO, tr("Redo"), ":/core/main/edit_redo.png", QKeySequence::Redo), SIGNAL(triggered(bool)), this, SLOT(OnEditRedo()));
	connect(addCommandAction(ACTION_EDIT_DELETE, tr("Delete"), ":/core/main/edit_delete.png", QKeySequence::Delete), SIGNAL(triggered(bool)), this, SLOT(OnEditDelete()));

	addXFormModeAction(ACTION_SELECTION_MODE, XFORM_MANAGER.objectSelectionMode(), tr("Select"), ":/core/main/mode_select.png");
	addXFormModeAction(ACTION_MOVE_MODE, XFORM_MANAGER.objectMoveMode(), tr("Move"), ":/core/main/mode_move.png");
	addXFormModeAction(ACTION_ROTATION_MODE, XFORM_MANAGER.objectRotationMode(), tr("Rotate"), ":/core/main/mode_rotate.png");
	addXFormModeAction(ACTION_SCALING_MODE, XFORM_MANAGER.objectScalingMode(), tr("Scale"), ":/core/main/mode_scale.png");
}

/// Creates and registers a single command action.
ActionProxy* EditingActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath, const QKeySequence& shortcut) 
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	if(!shortcut.isEmpty()) proxy->setShortcut(shortcut);	
	return proxy;
}

/// Creates and registers a single xform input mode action.
ViewportModeAction* EditingActionsHandler::addXFormModeAction(const QString& id, ViewportInputHandler* handler, const QString& title, const char* iconPath) 
{
	ViewportModeAction* action = new ViewportModeAction(id, handler, QColor(255, 255, 200));
	ACTION_MANAGER.addAction(action, title, iconPath);
	return action;
}

/// Handles ACTION_EDIT_UNDO command.
void EditingActionsHandler::OnEditUndo() 
{
	UNDO_MANAGER.undo();
}

/// Handles ACTION_EDIT_REDO command.
void EditingActionsHandler::OnEditRedo() 
{
	UNDO_MANAGER.redo();
}

/// Handles ACTION_EDIT_DELETE command.
void EditingActionsHandler::OnEditDelete() 
{	
	// Begin recording for undo.
	UNDO_MANAGER.beginCompoundOperation(tr("Delete"));

	// Delete all nodes in selection set.
	Q_FOREACH(SceneNode* node, DATASET_MANAGER.currentSelection()->nodes())
		node->deleteNode();

	// Stop recording for undo.
	UNDO_MANAGER.endCompoundOperation();
}

};
