///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __COMPRESSED_TEXT_PARSER_STREAM_H
#define __COMPRESSED_TEXT_PARSER_STREAM_H

#include <core/Core.h>
#include <atomviz/AtomViz.h>

#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_stream.hpp>

namespace AtomViz {

/**
 * \brief A stream class that decompresses gzipped text files on the fly.
 *
 * When opening the input file, it is decompressed if it has a .gz suffix.
 * Otherwise it is not processed in any special way.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT CompressedTextParserStream : public QObject
{
public:
	/// Constructor that opens the input stream.
	CompressedTextParserStream(const QString& filename) : _filename(filename), _lineNumber(0), _byteOffset(0) {
		std::ios_base::openmode openmode = std::ios_base::in;
		if(filename.endsWith(QLatin1String(".gz")))
			openmode |= std::ios_base::binary;
		_file_stream.open(filename.toUtf8().constData(), openmode);
		if(!_file_stream)
			throw Exception(tr("Failed to open file input %1.").arg(filename));
		if(filename.endsWith(QLatin1String(".gz"))) {
			_decompression_stream.push(boost::iostreams::gzip_decompressor());
			_decompression_stream.push(_file_stream);
		}
	}

	/// Reads in the next line.
	const string& readline() {
		_lineNumber++;
		if(_decompression_stream.empty()) {
			if(_file_stream.eof()) throw Exception(tr("File parsing error. Unexpected end of file after line %1.").arg(_lineNumber));
			std::getline(_file_stream, _line);
			if(!_file_stream && !_file_stream.eof()) throw Exception(tr("File parsing error. An I/O error occured at line number %1.").arg(_lineNumber));
		}
		else {
			if(_decompression_stream.eof()) throw Exception(tr("File parsing error. Unexpected end of file after line %1.").arg(_lineNumber));
			std::getline(_decompression_stream, _line);
			if(!_decompression_stream && !_decompression_stream.eof()) throw Exception(tr("File parsing error. An I/O error occured at line number %1.").arg(_lineNumber));
			_byteOffset += _line.length() + 1;
		}
		return _line;
	}

	/// Extracts characters from the input sequence and stores them as a c-string.
	int getline(char* s, streamsize n) {
		_lineNumber++;
		if(_decompression_stream.empty()) {
			if(_file_stream.eof()) return -1;
			_file_stream.getline(s, n);
			if(!_file_stream && !_file_stream.eof()) return -1;
			return _file_stream.gcount();
		}
		else {
			if(_decompression_stream.eof()) return -1;
			_decompression_stream.getline(s, n);
			if(!_decompression_stream && !_decompression_stream.eof()) return -1;
			_byteOffset += _decompression_stream.gcount() + 1;
			return _decompression_stream.gcount();
		}
	}

	void read(void* buffer, streamsize n) {
		if(_decompression_stream.empty()) {
			_file_stream.read((char*)buffer, n);
			if(_file_stream.gcount() != n)
				throw Exception(tr("File parsing error. Unexpected end of file"));
			if(!_file_stream && !_file_stream.eof())
				throw Exception(tr("File parsing error. An I/O error occured."));
		}
		else {
			_decompression_stream.read((char*)buffer, n);
			if(_decompression_stream.gcount() != n)
				throw Exception(tr("File parsing error. Unexpected end of file"));
			if(!_decompression_stream && !_decompression_stream.eof())
				throw Exception(tr("File parsing error. An I/O error occured."));
		}
	}

	/// Checks whether the end of file is reached.
	bool eof() {
		if(_decompression_stream.empty())
			return _file_stream.eof();
		else
			return _decompression_stream.eof();
	}

	/// Returns the last line read.
	const string& line() const { return _line; }

	/// Returns the number of the last line read.
	int lineNumber() const { return _lineNumber; }

	/// Returns the current position in the file.
	streampos byteOffset() {
		if(_decompression_stream.empty())
			return _file_stream.tellg();
		else
			return _byteOffset;
	}

	/// Returns the current position in the compressed file if it is gzipped.
	streampos compressedByteOffset() {
		return _file_stream.tellg();
	}

	/// Jumps to the given position in the file.
	void seek(streampos pos) {
		if(_decompression_stream.empty()) {
			_file_stream.seekg(pos);
			if(!_file_stream)
				throw Exception(tr("Failed to seek to byte offset %2 in input file.").arg(pos));
		}
		else {
			if(_byteOffset != streampos(0)) {
				// First seek to the beginning of the file by closing and re-opening the file for reading.
				_decompression_stream.reset();
				_file_stream.close();
				_file_stream.open(_filename.toUtf8().constData(), std::ios_base::in|std::ios_base::binary);
				if(!_file_stream)
					throw Exception(tr("Failed to re-open input file %1.").arg(_filename));
				_decompression_stream.push(boost::iostreams::gzip_decompressor());
				_decompression_stream.push(_file_stream);
				_byteOffset = 0;
				_lineNumber = 0;
			}
			_decompression_stream.ignore(pos);
			if(!_decompression_stream)
				throw Exception(tr("Failed to seek to byte offset %2 in compressed input file.").arg(pos));
		}
	}

	void clearError() {
		if(_decompression_stream.empty())
			_file_stream.clear();
		else
			_decompression_stream.clear();
	}

private:
	QString _filename;
	/// The current text line.
	string _line;
	/// The current line number.
	int _lineNumber;
	/// The current position in the decompressed file.
	streampos _byteOffset;
	/// The plain input stream.
	boost::iostreams::stream<boost::iostreams::file_source> _file_stream;
	/// The stream to read from when the file is gzipped.
	boost::iostreams::filtering_stream<boost::iostreams::input> _decompression_stream;

	Q_OBJECT
};

};	// End of namespace AtomViz

#endif // __COMPRESSED_TEXT_PARSER_STREAM_H
