///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __SELECT_EXPRESSION_MODIFIER_H
#define __SELECT_EXPRESSION_MODIFIER_H

#include <core/Core.h>

#include <atomviz/AtomViz.h>
#include "SelectionModifierBase.h"

class FunctionParser;

namespace AtomViz {

/******************************************************************************
* Selects atoms based on a user-defined expression.
******************************************************************************/
class ATOMVIZ_DLLEXPORT SelectExpressionModifier : public SelectionModifierBase
{
public:

	/// Default constructor.
	SelectExpressionModifier(bool isLoading = false) : SelectionModifierBase(isLoading) {
		INIT_PROPERTY_FIELD(SelectExpressionModifier, _expression);
	}

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time) { return TimeForever; }

	/// Sets the expression that is used to select atoms.
	void setExpression(const QString& expression) { _expression = expression; }

	/// Returns the expression that is used to select atoms.
	const QString& expression() const { return _expression; }

	/// \brief Returns the list of variables during the last evaluation.
	const QStringList& lastVariableNames() const { return _variableNames; }

public:

	Q_PROPERTY(QString expression READ expression WRITE setExpression)

protected:

	/// Modifies the atoms object.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// Compiles the stored expression string.
	/// Throws an exception on error.
	void compileExpression(FunctionParser& parser, int& numVariables);

	/// Determines the available variable names.
	QStringList getVariableNames();

	/// The expression that is used to select atoms.
	PropertyField<QString> _expression;

	/// The list of variables during the last evaluation.
	QStringList _variableNames;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(SelectExpressionModifier)
	DECLARE_PROPERTY_FIELD(_expression)
};

/******************************************************************************
* A properties editor for the SelectExpressionModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT SelectExpressionModifierEditor : public AtomsObjectModifierEditorBase
{
public:

	/// Sets the object being edited in this editor.
	virtual void setEditObject(RefTarget* newObject) {
		AtomsObjectModifierEditorBase::setEditObject(newObject);
		updateEditorFields();
	}

protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

private:

	/// Updates the enabled/disabled status of the editor's controls.
	void updateEditorFields();

	QLabel* variableNamesList;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(SelectExpressionModifierEditor)
};

};	// End of namespace AtomViz

#endif // __SELECT_EXPRESSION_MODIFIER_H
