///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __AMBIENT_LIGHTING_MODIFIER_H
#define __AMBIENT_LIGHTING_MODIFIER_H

#include <core/Core.h>
#include <core/scene/objects/Modifier.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <core/viewport/Viewport.h>

#include <atomviz/AtomViz.h>
#include "../AtomsObjectAnalyzerBase.h"

namespace AtomViz {

/******************************************************************************
* Computes the shading of the atoms.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AmbientLightingModifier : public AtomsObjectAnalyzerBase
{
public:

	/// Default constructor.
	AmbientLightingModifier(bool isLoading = false);

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// Returns the intensity of the shading.
	FloatType intensity() const { return _intensity ? _intensity->getCurrentValue() : (FloatType)1.0; }
	/// Sets the intensity of the shading.
	void setIntensity(FloatType newIntensity) { if(_intensity) _intensity->setCurrentValue(newIntensity); }
	/// Returns the controller for the intensity of the shading.
	FloatController* intensityController() const { return _intensity; }
	/// Sets the controller for the intensity of the shading.
	void setIntensityController(const FloatController::SmartPtr& ctrl) { _intensity = ctrl; }

	/// Returns the amount of spherical sampling points used in the shading computation.
	/// This must be an integer value between 0 and 4.
	int samplingLevel() const { return _samplingLevel; }
	/// Sets the amount of spherical sampling points used in the shading computation.
	/// This must be an integer value between 0 and 4.
	void setSamplingLevel(int level) { _samplingLevel = level; }

	/// Returns the buffer resolution level.
	/// This must be an integer value between 0 and 5.
	int bufferResolution() const { return _bufferResolution; }
	/// Sets the buffer resolution level.
	/// This must be an integer value between 0 and 5.
	void setBufferResolution(int res) { _bufferResolution = res; }

public:

	Q_PROPERTY(FloatType intensity READ intensity WRITE setIntensity)
	Q_PROPERTY(int samplingLevel READ samplingLevel WRITE setSamplingLevel)
	Q_PROPERTY(int bufferResolution READ bufferResolution WRITE setBufferResolution)

protected:

	// RefTarget virtual functions:
	virtual void saveToStream(ObjectSaveStream& stream);
	virtual void loadFromStream(ObjectLoadStream& stream);
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// This is the actual analysis method.
	/// It is responsible for storing the analysis results in the results container object that
	/// will be stored along with the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs);

	/// Applies the previously calculated analysis results to the atoms object.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval);

	/// Contains the calculated brightness values.
	QVector<unsigned int> ambientArray;

	/// The maximum brightness value.
	unsigned int maxAmbient;

	/// This controls the intensity of the ambient lighting effect.
	ReferenceField<FloatController> _intensity;

	/// Controls the quality of the lighting computation.
	PropertyField<int> _samplingLevel;

	/// Controls the resolution of the offscreen rendering buffer.
	PropertyField<int> _bufferResolution;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AmbientLightingModifier)
	DECLARE_REFERENCE_FIELD(_intensity)
	DECLARE_PROPERTY_FIELD(_samplingLevel)
	DECLARE_PROPERTY_FIELD(_bufferResolution)
};

/******************************************************************************
* A properties editor for the AmbientLightingModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AmbientLightingModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user presses the Recalculate button.
	void onRecalculate();

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AmbientLightingModifierEditor)
};

};	// End of namespace AtomViz

#endif // __AMBIENT_LIGHTING_MODIFIER_H
