///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOMS_OBJECT_ANALYSER_BASE_H
#define __ATOMS_OBJECT_ANALYSER_BASE_H

#include <core/Core.h>
#include "AtomsObjectModifierBase.h"
#include <atomviz/utils/NearestNeighborList.h>

namespace AtomViz {

/******************************************************************************
* Abstract base class for modifiers that perform some kind of analysis
* on the atoms that cannot be done in realtime.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AtomsObjectAnalyzerBase : public AtomsObjectModifierBase
{
protected:

	/// Constructor.
	AtomsObjectAnalyzerBase(bool isLoading);

public:

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// This will recalculate the analysis for the first input object.
	/// Throws an exception on error.
	EvaluationStatus performAnalysis(TimeTicks time, bool suppressDialogs = false);

	/// This will recalculate the analysis for one input object of the modifier.
	/// Throws an exception on error.
	EvaluationStatus performAnalysis(TimeTicks time, ModifierApplication* modApp, bool suppressDialogs = false);

	/// Returns whether automatic updating on animation time changes is enabled.
	bool autoUpdateOnTimeChangeEnabled() const { return _autoUpdateOnTimeChange; }

	/// Sets whether automatic updating on animation time changes is enabled.
	void setAutoUpdateOnTimeChangeEnabled(bool enable) { _autoUpdateOnTimeChange = enable; }

	/// \brief Returns the nearest neighbor list used by the modifier.
	/// \return The internal neighbor list object.
	/// \sa setNeighborList()
	NearestNeighborList* nearestNeighborList() const { return _nearestNeighborList; }

	/// \brief Replaces the internal neighbor list object.
	/// \list The new neighbor list object.
	/// \undoable
	void setNearestNeighborList(const NearestNeighborList::SmartPtr& list) { _nearestNeighborList = list; }

public:

	Q_PROPERTY(bool autoUpdateOnTimeChangeEnabled READ autoUpdateOnTimeChangeEnabled WRITE setAutoUpdateOnTimeChangeEnabled)

protected:

	/// Modifies the atoms object. The time interval passed
	/// to the function is reduced to the interval where the modified object is valid/constant.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// \brief Analyses the input object.
	/// \param time The animation at which the input object should be analyzed.
	///
	/// This is the specific analysis method that must be implemented by sub-classes.
	/// It is responsible for storing the analysis results in the CustomAttributesContainer
	/// of the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs) = 0;

	/// \brief Applies the previously calculated analysis results to the atoms object.
	/// \param[in] time The animation at which the results should be stored in the AtomsObject.
	/// \param[in,out] validityInterval This time interval is reduced to the interval durcing which the modified object is valid/constant.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval) = 0;

	/// Gets the validity interval for this analysis result.
	const TimeInterval& analysisValidityInterval() const { return _analysisValidityInterval; }

	/// Specifies the validity interval of this analysis result.
	void setAnalysisValidityInterval(const TimeInterval& newInterval) { _analysisValidityInterval = newInterval; }

	/// Reduces the validity interval of this analysis result to the given interval.
	void intersectAnalysisValidityInterval(const TimeInterval& interval) { _analysisValidityInterval.intersect(interval); }

	/// \brief Builds the nearest neighbor list for the current input AtomsObject.
	/// \param noProgressIndicator If set to \c true then no progress indicator is created to
	///                            show the build progress. In this case the building cannot be canceled by the user.
	/// \return \c false when the build process has been canceled by the user;
	///         \c true on success.
	/// \throw Exception if an error has occured during neighbor list building.
	bool buildNeighborList(bool noProgressIndicator = false);

protected:

	// RefTarget virtual functions:
	virtual void saveToStream(ObjectSaveStream& stream);
	virtual void loadFromStream(ObjectLoadStream& stream);
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

private:

	/// This controls whether the analysis is automatically redone when the animation time has changed.
	PropertyField<bool> _autoUpdateOnTimeChange;

	/// Contains the validity interval of the analysis results.
	TimeInterval _analysisValidityInterval;

	/// This is the nearest neighbor list used by the analysis modifier.
	ReferenceField<NearestNeighborList> _nearestNeighborList;

	/// The status code returned by the last call to doAnalysis().
	EvaluationStatus _analysisStatus;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AtomsObjectAnalyzerBase)
	DECLARE_PROPERTY_FIELD(_autoUpdateOnTimeChange)
	DECLARE_REFERENCE_FIELD(_nearestNeighborList)
};

};	// End of namespace AtomViz

#endif // __ATOMS_OBJECT_ANALYSER_BASE_H
