///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/RolloutContainer.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/viewport/Window3DContainer.h>
#include <core/viewport/Window3D.h>
#include "AtomVizBenchmarkUtility.h"

namespace AtomViz {

IMPLEMENT_PLUGIN_CLASS(AtomVizBenchmarkUtility, UtilityPlugin)

/******************************************************************************
* Initializes the utility applet.
******************************************************************************/
AtomVizBenchmarkUtility::AtomVizBenchmarkUtility() : UtilityPlugin(), panel(NULL)
{
}

/******************************************************************************
* Shows the UI of the utility in the given RolloutContainer.
******************************************************************************/
void AtomVizBenchmarkUtility::openUtility(RolloutContainer* container, const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	panel = new QWidget();
	container->addRollout(panel, tr("OpenGL Benchmark"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(panel);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(4);

	benchmarkTypeGroup = new QButtonGroup(panel);
	QRadioButton* radio1 = new QRadioButton(tr("Small dataset / large atoms"), panel);
	radio1->setChecked(true);
	layout->addWidget(radio1);
	benchmarkTypeGroup->addButton(radio1, 0);
	QRadioButton* radio2 = new QRadioButton(tr("Large dataset / small atoms"), panel);
	layout->addWidget(radio2);
	benchmarkTypeGroup->addButton(radio2, 1);

	QPushButton* btn;
	layout->addWidget((btn = new QPushButton(tr("Run Benchmark"), panel)));
	connect(btn, SIGNAL(clicked(bool)), this, SLOT(onRunBenchmark()));

	resultLabel = new QLabel(panel);
	layout->addWidget(resultLabel);
	resultLabel->setText(tr("Click button to start benchmark."));
	resultLabel->setTextInteractionFlags(Qt::TextSelectableByMouse|Qt::TextSelectableByKeyboard);
}

/******************************************************************************
* Removes the UI of the utility from the rollout container.
******************************************************************************/
void AtomVizBenchmarkUtility::closeUtility(RolloutContainer* container)
{
	delete panel;
}

/******************************************************************************
* Handles the "Run Benchmark" command event.
******************************************************************************/
void AtomVizBenchmarkUtility::onRunBenchmark()
{
	QObjectCleanupHandler cleanupHandler;

	// Create the benchmark window.
	Window3DContainer* win3DContainer = new Window3DContainer(NULL);
	cleanupHandler.add(win3DContainer);
	win3DContainer->setWindowModality(Qt::ApplicationModal);
	win3DContainer->setWindowTitle(tr("OpenGL Benchmark"));
	win3DContainer->resize(600,600);

	int dataSet = benchmarkTypeGroup->checkedId();

	// Create the OpenGL window.
	Benchmark3DWindow* win3D = new Benchmark3DWindow(win3DContainer, dataSet);
	win3D->setGeometry(win3DContainer->rect());
	win3D->setViewportRectangle(win3D->rect());

	// Show window.
	win3DContainer->show();
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

	QTime t;
	t.start();
	int nFrames = 500;
	for(int frame = 0; frame < nFrames; frame++) {
		FloatType angle = (FloatType)frame / nFrames * FLOATTYPE_PI * 2.0;
		win3D->setAngle(angle);
		win3D->update();
		Window3D::processWindowUpdates();
		QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
	}
	int renderTime = t.elapsed();

	FloatType framesPerSecond = (FloatType)nFrames * 1000.0 / (FloatType)renderTime;
	FloatType atomsPerSecond = framesPerSecond * win3D->numAtoms;

	QString datasetResults = tr("<h3>Dataset:</h3><p><i>Number of atoms</i>: %1</p>").arg(win3D->numAtoms);
	QString benchmarkResults = tr("<h3>Benchmark results:</h3><p><i>Frames per second</i>: %2<br>"
			"<i>Atoms per second</i>: %3<br><i>Initialization time</i>: %4 msec</p>").arg(framesPerSecond).arg(atomsPerSecond).arg(win3D->preparationTime);
	QString openglResults = tr("<h3>OpenGL:</h3><p><i>Vendor</i>: %1<br><i>Renderer</i>: %2")
		.arg((const char*)glGetString(GL_VENDOR))
		.arg((const char*)glGetString(GL_RENDERER));
	QString ext1 = tr("<font color=green>supported</font>");
	QString ext0 = tr("<font color=red>not supported</font>");
	openglResults.append(tr("<br>Direct rendering: %1").arg(win3D->format().directRendering() ? ext1 : ext0));
	openglResults.append(tr("<br>Depth buffer bits: %1").arg(win3D->format().depthBufferSize()));
	openglResults.append(tr("<br>Extensions:<ul>"));
	openglResults.append(tr("<li>EXT_compiled_vertex_array: %1</li>").arg(win3D->hasCompiledVertexArraysExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>KTX_buffer_region: %1</li>").arg(win3D->hasBufferRegionsExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>WIN_swap_hint: %1</li>").arg(win3D->hasSwapHintExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>ARB_point_parameters: %1</li>").arg(win3D->hasPointParametersExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>ARB_vertex_buffer_object: %1</li>").arg(win3D->hasVertexBufferObjectsExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>EXT_framebuffer_object: %1</li>").arg(win3D->hasFrameBufferExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>ARB_shader_objects: %1</li>").arg(win3D->hasShaderObjectsExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>ARB_vertex_shader: %1</li>").arg(win3D->hasVertexShaderExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>ARB_fragment_shader: %1</li>").arg(win3D->hasFragmentShaderExtension() ? ext1 : ext0));
	openglResults.append(tr("<li>EXT_fog_coord: %1</li>").arg(win3D->hasFogCoordExtension() ? ext1 : ext0));
	openglResults.append(tr("</ul></p>"));

	resultLabel->setText(datasetResults + benchmarkResults + openglResults);
}

/******************************************************************************
* Renders the 3d benchmark scene.
******************************************************************************/
void AtomVizBenchmarkUtility::Benchmark3DWindow::renderWindow()
{
	clearBuffer();

	setProjectionMatrix(Matrix4::perspective(FLOATTYPE_PI/3.0, 1.0, 1, 1000));

	if(dataSet == 0)
		setViewMatrix(AffineTransformation::lookAt(Point3(50,-50,50), ORIGIN, Vector3(0,0,1)));
	else
		setViewMatrix(AffineTransformation::lookAt(Point3(150,-150,150), ORIGIN, Vector3(0,0,1)));

	atoms.prepare(this);
	if(!atoms.isFilled())
		createAtoms();

	setWorldMatrix(AffineTransformation::rotationZ(angle) *
			AffineTransformation::translation(-(atoms.boundingBox().center() - ORIGIN)));

	atoms.render(this);
}

/******************************************************************************
* Creates the atoms for the benchmark.
******************************************************************************/
void AtomVizBenchmarkUtility::Benchmark3DWindow::createAtoms()
{
	int nrows;
	FloatType spacing = 1.0;
	FloatType radius = 0.5f;
	if(dataSet == 0)
		nrows = 50;
	else
		nrows = 150;

	QTime t;
	t.start();
	numAtoms = nrows*nrows*nrows;
	atoms.beginAtoms(numAtoms);
	for(int ix=0; ix<nrows; ix++) {
		for(int iy=0; iy<nrows; iy++) {
			for(int iz=0; iz<nrows; iz++) {
				atoms.specifyAtom(Point3(spacing*ix, spacing*iy, spacing*iz), 200*ix/nrows+55, 200*iy/nrows+55, 200*iz/nrows+55, radius);
			}
		}
	}
	atoms.endAtoms();
	preparationTime = t.elapsed();
}

};
