///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/ColorControllerUI.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/StringPropertyUI.h>
#include <core/scene/animation/controller/StandardControllers.h>

#include "AtomType.h"
#include "AtomsObject.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AtomType, RefTarget)
DEFINE_REFERENCE_FIELD(AtomType, VectorController, "AtomColor", colorCtrl)
DEFINE_REFERENCE_FIELD(AtomType, FloatController, "AtomRadius", radiusCtrl)
DEFINE_PROPERTY_FIELD(AtomType, "Name", _name)
SET_PROPERTY_FIELD_LABEL(AtomType, colorCtrl, "Color")
SET_PROPERTY_FIELD_LABEL(AtomType, radiusCtrl, "Radius")
SET_PROPERTY_FIELD_LABEL(AtomType, _name, "Name")
SET_PROPERTY_FIELD_UNITS(AtomType, radiusCtrl, WorldParameterUnit)

/******************************************************************************
* Constructs a new AtomType.
******************************************************************************/
AtomType::AtomType(bool isLoading) : RefTarget(isLoading)
{
	INIT_PROPERTY_FIELD(AtomType, colorCtrl);
	INIT_PROPERTY_FIELD(AtomType, radiusCtrl);
	INIT_PROPERTY_FIELD(AtomType, _name);
	if(!isLoading) {
		colorCtrl = CONTROLLER_MANAGER.createDefaultController<VectorController>();		
		colorCtrl->setValue(0, Color(1,1,1));		
		radiusCtrl = CONTROLLER_MANAGER.createDefaultController<FloatController>();		
		radiusCtrl->setValue(0, 1);		
	}
}

IMPLEMENT_PLUGIN_CLASS(AtomTypeEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void AtomTypeEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Atom Type"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);
    
	QGridLayout* layout2 = new QGridLayout();
	layout2->setContentsMargins(0,0,0,0);
	layout2->setSpacing(0);
	layout2->setColumnStretch(1, 1);
	layout1->addLayout(layout2);
	
	// Text box for the name of atom type.
	StringPropertyUI* namePUI = new StringPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomType, _name));
	layout2->addWidget(new QLabel(tr("Name:")), 0, 0);
	layout2->addWidget(namePUI->textBox(), 0, 1, 1, 2);
	
	// Color parameter.
	ColorControllerUI* colorPUI = new ColorControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomType, colorCtrl));
	layout2->addWidget(colorPUI->label(), 1, 0);
	layout2->addWidget(colorPUI->colorPicker(), 1, 1);	
   
	// Atom radius parameter.
	FloatControllerUI* radiusPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomType, radiusCtrl));
	layout2->addWidget(radiusPUI->label(), 2, 0);
	layout2->addWidget(radiusPUI->textBox(), 2, 1);
	layout2->addWidget(radiusPUI->spinner(), 2, 2);
	radiusPUI->setMinValue(0);
}

};	// End of namespace AtomViz
