/*
 * libosengine - A synchronization engine for the opensync framework
 * Copyright (C) 2004-2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007       Daniel Gollub <dgollub@suse.org>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 * 
 */
 
#include "opensync.h"
#include "opensync_internals.h"

#include "opensync-group.h"
#include "opensync-engine.h"
#include "opensync-client.h"
#include "opensync-data.h"
#include "opensync-mapping.h"
#include "opensync-plugin.h"

#include "opensync_engine_internals.h"
#include "opensync_status_internals.h"
#include "opensync_obj_engine_internals.h"
#include "opensync_sink_engine_internals.h"
#include "opensync_mapping_entry_engine_internals.h"

#include "plugin/opensync_objtype_sink_internals.h"
#include "archive/opensync_archive_internals.h"
#include "data/opensync_change_internals.h"
#include "data/opensync_data_internals.h"
#include "client/opensync_client_proxy_internals.h"

#include "opensync_mapping_engine.h"
#include "opensync_mapping_engine_internals.h"


OSyncMappingEngine *osync_mapping_engine_new(OSyncObjEngine *parent, OSyncMapping *mapping, OSyncError **error)
{
	OSyncMappingEngine *engine = NULL;
	OSyncList *s = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parent, mapping, error);

	osync_assert(parent);
	osync_assert(mapping);
	
	engine = osync_try_malloc0(sizeof(OSyncMappingEngine), error);
	if (!engine)
		goto error;
	engine->ref_count = 1;
	
	engine->mapping = mapping;
	osync_mapping_ref(mapping);
	
	engine->parent = parent;
	engine->synced = TRUE;
	
	for (s = parent->sink_engines; s; s = s->next) {
		OSyncSinkEngine *sink_engine = s->data;
		OSyncMappingEntryEngine *entry_engine = NULL;
		
		OSyncMember *member = osync_client_proxy_get_member(sink_engine->proxy);
		OSyncMappingEntry *mapping_entry = osync_mapping_find_entry_by_member_id(mapping, osync_member_get_id(member));
		if (!mapping_entry) {
			osync_error_set(error, OSYNC_ERROR_GENERIC, "Inconsistency in Mapping Table "
					"for Object Type \"%s\" detected.",
					osync_obj_engine_get_objtype(engine->parent));
			goto error_free_engine;
		}
		
		entry_engine = osync_entry_engine_new(mapping_entry, engine, sink_engine, parent, error);
		if (!entry_engine)
			goto error_free_engine;

		engine->entries = osync_list_append(engine->entries, entry_engine);
	}
	
	osync_trace(TRACE_EXIT, "%s: %p", __func__, engine);
	return engine;

 error_free_engine:
	osync_mapping_engine_unref(engine);
 error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return NULL;
}

OSyncMappingEngine *osync_mapping_engine_ref(OSyncMappingEngine *engine)
{
	osync_assert(engine);
	
	g_atomic_int_inc(&(engine->ref_count));

	return engine;
}

void osync_mapping_engine_unref(OSyncMappingEngine *engine)
{
	osync_assert(engine);
		
	if (g_atomic_int_dec_and_test(&(engine->ref_count))) {
		if (engine->mapping)
			osync_mapping_unref(engine->mapping);
		
		if (engine->master)
			osync_entry_engine_unref(engine->master);
		
		while (engine->entries) {
			OSyncMappingEntryEngine *entry = engine->entries->data;
			osync_entry_engine_unref(entry);
			
			engine->entries = osync_list_remove(engine->entries, engine->entries->data);
		}
		
		osync_free(engine);
	}
}

static OSyncMappingEntryEngine *_osync_mapping_engine_find_entry(OSyncMappingEngine *engine, OSyncChange *change)
{
	OSyncList *e;
	for (e = engine->entries; e; e = e->next) {
		OSyncMappingEntryEngine *entry = e->data;
		if (change && entry->change == change)
			return entry;
	}
	
	return NULL;
}

OSyncMember *osync_mapping_engine_change_find_member(OSyncMappingEngine *engine, OSyncChange *change)
{
	OSyncMember *member = NULL;
	OSyncMappingEntryEngine *entry = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p, %p)", __func__, engine, change);

	entry = _osync_mapping_engine_find_entry(engine, change);
	if (!entry)
		goto end;

	member = osync_client_proxy_get_member(entry->sink_engine->proxy);

 end:	
	osync_trace(TRACE_EXIT, "%s: %p", __func__, member);
	return member;
}

static OSyncMappingEntryEngine *_osync_mapping_engine_get_latest_entry(OSyncMappingEngine *engine, OSyncError **error)
{
	OSyncMappingEntryEngine *latest_entry = NULL; 
	OSyncChange *latest_change = NULL;
	time_t latest = 0;
	unsigned int i;

	osync_trace(TRACE_ENTRY, "%s(%p, %p)", __func__, engine, error);
	osync_trace(TRACE_INTERNAL, "mapping number: %i", osync_mapping_engine_num_changes(engine));
	for (i=0; i < osync_mapping_engine_num_changes(engine); i++) {
		OSyncChange *change = osync_mapping_engine_nth_change(engine, i); 
		OSyncData *data = NULL;
		time_t cur = 0;

		if (osync_change_get_changetype(change) == OSYNC_CHANGE_TYPE_UNKNOWN)
			continue;

		data = osync_change_get_data(change);

		if (!osync_data_has_data(data))
			continue;

		cur = osync_data_get_revision(data, error);

		if (cur < 0)
			goto error;
		
		/* If there are several changes/entries having the
			 same _and_ the latest revision -> don't declare
			 a latest_change. Since it's not guranteed that those are
			 equal, even with the _same_ revision.
		*/
		if (cur == latest)
			latest_change = NULL;

		if (cur <= latest)
			continue;

		latest = cur;
		latest_change = change;
	}

	if (!latest_change) {
		osync_trace(TRACE_EXIT, "%s: Can't find the latest change.",
								__func__);
		return NULL;
	}

	latest_entry = _osync_mapping_engine_find_entry(engine, latest_change);

	if (!latest_entry) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "Can't find the latest entry of the latest change.");
		goto error;
	}

	osync_trace(TRACE_EXIT, "%s: %p", __func__, latest_entry);
	return latest_entry;

 error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return NULL;
}

osync_bool osync_mapping_engine_supports_ignore(OSyncMappingEngine *engine)
{
	OSyncObjEngine *parent = NULL;
	osync_bool ignore_supported = TRUE;
	OSyncList *s = NULL;

	osync_trace(TRACE_ENTRY, "%s(%p)", __func__, engine);
	osync_assert(engine);

	parent = engine->parent;
	for (s = parent->sink_engines; s; s = s->next) {
		OSyncSinkEngine *sink_engine = s->data;
		
		OSyncMember *member = osync_client_proxy_get_member(sink_engine->proxy);
		OSyncMappingEntryEngine *entry_engine = osync_mapping_engine_get_entry(engine, sink_engine);

		/* check if mapping could be solved by "ignore" conflict handler */
		const char *objtype = entry_engine->sink_engine->engine->objtype;
		OSyncObjTypeSink *objtype_sink = osync_member_find_objtype_sink(member, objtype);

		/* if there is no sink read function, ignore is not support for this mapping. */
		if (!objtype_sink || !osync_objtype_sink_get_function_read(objtype_sink))
			ignore_supported = FALSE; 

	}
	
	osync_trace(TRACE_EXIT, "%s: conflict handler ignore supported: %s", __func__, ignore_supported ? "TRUE" : "FALSE");
	return ignore_supported;
}

osync_bool osync_mapping_engine_supports_use_latest(OSyncMappingEngine *engine)
{
	osync_bool latest_supported = TRUE;
	OSyncMappingEntryEngine *latest_entry = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p)", __func__, engine);
	osync_assert(engine);
	/* we ignore the error for now ... it's just a test if it would be possible/supported. */
	latest_entry = _osync_mapping_engine_get_latest_entry(engine, NULL);

	if (!latest_entry)
		latest_supported = FALSE; 
	
	osync_trace(TRACE_EXIT, "%s: conflict handler \"latest entry\" supported: %s", __func__, latest_supported ? "TRUE" : "FALSE");
	return latest_supported;
}

osync_bool osync_mapping_engine_multiply(OSyncMappingEngine *engine, OSyncError **error)
{
	OSyncList *e = NULL;
	osync_assert(engine);
	osync_assert(engine->mapping);
	
	osync_trace(TRACE_ENTRY, "%s(%p(%lli), %p)", __func__, engine, osync_mapping_get_id(engine->mapping), error);
		
	if (engine->synced) {
		osync_trace(TRACE_EXIT, "%s: No need to multiply. Already synced", __func__);
		return TRUE;
	}

	if (!engine->master) {
		osync_error_set(error, OSYNC_ERROR_GENERIC, "No master set");
		goto error;
	}

	for (e = engine->entries; e; e = e->next) {
		OSyncChange *existChange = NULL, *masterChange = NULL;
		OSyncData *masterData = NULL, *newData = NULL;
		OSyncMappingEntryEngine *entry_engine = e->data;
		OSyncChangeType existChangeType = 0, newChangeType = 0;

		/* Skip if the current entry_engine is the master */
		if (entry_engine == engine->master)
			continue;
			
		/* Input is:
		 * masterChange -> change that solved the mapping
		 * masterData -> data of masterChange
		 * existChange -> change that will be overwritten (if any) */
		
		existChange = entry_engine->change;
		masterChange = osync_entry_engine_get_change(engine->master);
		masterData = osync_change_get_data(masterChange);

		/* Skip if masterChang the existing Change of the current
		 * Mapping Entry Engine are the same. The original change of
		 * the current Mapping Entry Engine got replaced by the conflict
		 * master in osync_mapping_engine_check_conflict().
		 *
		 * The purpose is to avoid unexpected write of changes which are
		 * equal on a multi-sync. Regression testcases:
		 *  - multisync_dual_new
		 *  - multisync_tripple_new
		 */
		if (existChange == masterChange)
			continue;

		osync_trace(TRACE_INTERNAL, "Propagating change %s to %p from %p", osync_mapping_entry_get_uid(entry_engine->entry), entry_engine, engine->master);
		
		/* Clone the masterData. This has to be done since the data
		 * might get changed (converted) and we dont want to touch the 
		 * original data */
		newData = osync_data_clone(masterData, error);
		if (!newData)
			goto error;
		
		if (!existChange) {
			existChange = osync_change_new(error);
			if (!existChange)
				goto error;
			
			osync_change_set_changetype(existChange, OSYNC_CHANGE_TYPE_UNKNOWN);
		} else {
			/* Ref the change so that we can unref it later */
			osync_change_ref(existChange);
		}
		
		/* Save the changetypes, so that we can calculate the correct changetype later */
		existChangeType = osync_change_get_changetype(existChange);
		newChangeType = osync_change_get_changetype(masterChange);
		
		osync_trace(TRACE_INTERNAL, "Orig change type: %i New change type: %i", existChangeType, newChangeType);

		/* Now update the entry with the change */
		osync_entry_engine_update(entry_engine, existChange);
		
		/* We have to use the uid of the entry, so that the member
		 * can correctly identify the entry 
		 *
		 * prahal: added a check if the entry has a uid to send the existing uid
		 * to the plugins in case we have a slow-sync (both are added and have a uid) 
		 * This to avoid creating duplicates by sending the plugins a different uid 
		 * with the same or merged data 
		 *
		 * dgollub: enhanced the check if the entry has a uid to send the existing uid
		 * to the plugins in case we have a slow-sync - both have changetype ADDED - and
		 * for odd plugins/protocolls which mark new entries as MODIFIED all the time.
		 * Changetype MODIFIED of new entries has atleast the IrMC plugin and likely some
		 * SE SyncML implementation...
		 * 
		 * ^^irmc hacks in the irmc plugin ;-)
		 *
		 * dgollub: Set masterChange UID for existChange if entry_engine->entry doesn't have 
		 * mapping uid, even if the newChangeType is UNKOWN. Bug: #571
		 *
		 * prahal : rely on the fact that there are no mapping nor the entry existed to detect new change
		 *          Also avoid changing the id of the change if one existed and there where no mapping :
		 *          this way we send the id known to the member in case of a "modify". Fixing syncml plugin
		 *          freezing the phone and mozilla sync receiving an id it cannot do anything with for modify
		 *          in slow sync (no existing mapping).
		 *
		 */
		if ((!osync_mapping_entry_get_uid(entry_engine->entry) && !osync_change_get_uid(existChange)) ) 
			osync_change_set_uid(existChange, osync_change_get_uid(masterChange));
		else if(osync_mapping_entry_get_uid(entry_engine->entry)) 
			osync_change_set_uid(existChange, osync_mapping_entry_get_uid(entry_engine->entry));

		osync_change_set_data(existChange, newData);
		osync_change_set_changetype(existChange, osync_change_get_changetype(masterChange));
		
		/* We also have to update the changetype of the new change */
		if (newChangeType == OSYNC_CHANGE_TYPE_ADDED && (existChangeType != OSYNC_CHANGE_TYPE_DELETED && existChangeType != OSYNC_CHANGE_TYPE_UNKNOWN)) {
			osync_trace(TRACE_INTERNAL, "Updating change type to MODIFIED");
			osync_change_set_changetype(existChange, OSYNC_CHANGE_TYPE_MODIFIED);
			/* Only adapt the change to ADDED if the existing Change got deleted. Don't update it to ADDED if existChangeType is UNKOWN.
				 The exitChangeType is at least also UNKOWN if the file-sync has only one modified entry. */
		} else if (newChangeType == OSYNC_CHANGE_TYPE_MODIFIED && (existChangeType == OSYNC_CHANGE_TYPE_DELETED)) {
			osync_trace(TRACE_INTERNAL, "Updating change type to ADDED");
			osync_change_set_changetype(existChange, OSYNC_CHANGE_TYPE_ADDED);
		}
		
		osync_change_unref(existChange);
		/* Also unref newData. Otherwise this cannot be freed when it is written. */
		osync_data_unref(newData);
			
		osync_entry_engine_set_dirty(entry_engine, TRUE);
	}
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

 error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

void osync_mapping_engine_set_master(OSyncMappingEngine *engine, OSyncMappingEntryEngine *entry)
{
	if (engine->master)
		osync_entry_engine_unref(engine->master);
	engine->master = entry;
	osync_entry_engine_ref(engine->master);
}

static unsigned int prod(unsigned int n)
{
	int x = 0;
	if (n == 0)
		return 0;
	x = ((n + 1) / 2) * n;
	if (!(n & 0x1))
		x += n / 2;
	return x;
}

osync_bool osync_mapping_engine_check_conflict(OSyncMappingEngine *engine)
{
	unsigned int is_same = 0;
	OSyncList *e = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p)", __func__, engine);
	osync_assert(engine != NULL);
	
	if (engine->master != NULL) {
		osync_trace(TRACE_EXIT, "%s: Already has a master", __func__);
		return TRUE;
	}
	
	if (engine->conflict) {
		osync_trace(TRACE_INTERNAL, "Detected conflict early");
		goto conflict;
	}
	
	for (e = engine->entries; e; e = e->next) {
		OSyncMappingEntryEngine *leftentry = e->data;
		OSyncMappingEntryEngine *rightentry = NULL;
		
		OSyncChange *leftchange = osync_entry_engine_get_change(leftentry);
		OSyncChange *rightchange = NULL;
		OSyncList *n = NULL;
		osync_trace(TRACE_INTERNAL, "change: %p: %i", leftchange, leftchange ? osync_change_get_changetype(leftchange) : OSYNC_CHANGE_TYPE_UNKNOWN);
		if (leftchange == NULL)
			continue;
			
		if (osync_change_get_changetype(leftchange) == OSYNC_CHANGE_TYPE_UNKNOWN)
			continue;
		
		osync_mapping_engine_set_master(engine, leftentry);
		for (n = e->next; n; n = n->next) {
			rightentry = n->data;
			rightchange = osync_entry_engine_get_change(rightentry);
		
			if (rightchange == NULL)
				continue;
		
			if (osync_change_get_changetype(rightchange) == OSYNC_CHANGE_TYPE_UNKNOWN)
				continue;
			
			/* XXX: this compare is obsolate?! we do one complicated compare with dermging before this step! */
#if 1 
			if (osync_change_compare(leftchange, rightchange, NULL /* hopefully obsoalte*/) != OSYNC_CONV_DATA_SAME) {
				engine->conflict = TRUE;
				goto conflict;
			} else
#endif
			{
				is_same++;

				/* Update the change to the master change.
				 * This helps osync_mapping_engine_multiply() to identify
				 * them as equal changes which need not get multiplied,
				 * without an additional osync_change_compare()!
				 */
				osync_entry_engine_update(rightentry, leftchange);
			}
		}
	}
	
 conflict:
	if (engine->conflict) {
		//conflict, solve conflict
		osync_trace(TRACE_INTERNAL, "Got conflict for mapping_engine %p", engine);
		engine->parent->conflicts = osync_list_append(engine->parent->conflicts, engine);

		if (!osync_status_conflict(engine->parent->parent, engine))
			goto error;

		osync_trace(TRACE_EXIT, "%s: Got conflict", __func__);
		return TRUE;
	}
	osync_assert(engine->master);
	osync_status_update_mapping(engine->parent->parent, engine, OSYNC_ENGINE_MAPPING_EVENT_SOLVED, NULL);
	
	if (is_same == prod(osync_list_length(engine->entries) - 1)) {
		OSyncList *e = NULL;
		osync_trace(TRACE_INTERNAL, "No need to sync. All entries are the same");
		for (e = engine->entries; e; e = e->next) {
			OSyncMappingEntryEngine *entry = e->data;
			entry->dirty = FALSE;
		}
		engine->synced = TRUE;
	}
	
	osync_trace(TRACE_EXIT, "%s: No conflict", __func__);
	return TRUE;

error:
	osync_trace(TRACE_EXIT_ERROR, "%s", __func__);
	return FALSE;
}



OSyncMappingEntryEngine *osync_mapping_engine_get_entry(OSyncMappingEngine *engine, OSyncSinkEngine *sinkengine)
{
	OSyncList *e = NULL;
	for (e = engine->entries; e; e = e->next) {
		OSyncMappingEntryEngine *entry_engine = e->data;
		if (sinkengine == entry_engine->sink_engine)
			return entry_engine;
	}
	
	return NULL;
}



unsigned int osync_mapping_engine_num_changes(OSyncMappingEngine *engine)
{
	unsigned int num = 0;
	OSyncList *e = NULL;
	osync_assert(engine);
	for (e = engine->entries; e; e = e->next) {
		OSyncMappingEntryEngine *entry = e->data;
		if (entry->change)
			num++;
	}
	
	return num;
}

OSyncChange *osync_mapping_engine_nth_change(OSyncMappingEngine *engine, unsigned int nth)
{
	unsigned int num = 0;
	OSyncList *e = NULL;
	osync_assert(engine);
	for (e = engine->entries; e; e = e->next) {
		OSyncMappingEntryEngine *entry = e->data;
		if (entry->change) {
			if (num == nth)
				return entry->change;
			num++;
		}
	}
	
	return NULL;
}

OSyncChange *osync_mapping_engine_member_change(OSyncMappingEngine *engine, long long int memberid)
{
	OSyncList *e = NULL;
	osync_assert(engine);
	for (e = engine->entries; e; e = e->next) {
		OSyncMappingEntryEngine *entry = e->data;
		if (entry->change) {
			if (osync_mapping_entry_get_member_id(entry->entry) == memberid)
				return entry->change;
		}
	}
	
	return NULL;
}

osync_bool osync_mapping_engine_solve(OSyncMappingEngine *engine, OSyncChange *change, OSyncError **error)
{
	OSyncMappingEntryEngine *entry = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p, %p)", __func__, engine, change);
	
	entry = _osync_mapping_engine_find_entry(engine, change);
	engine->conflict = FALSE;
	osync_mapping_engine_set_master(engine, entry);
	osync_status_update_mapping(engine->parent->parent, engine, OSYNC_ENGINE_MAPPING_EVENT_SOLVED, NULL);
	engine->parent->conflicts = osync_list_remove(engine->parent->conflicts, engine);

	if (!osync_obj_engine_command(engine->parent, OSYNC_ENGINE_COMMAND_END_CONFLICTS, error))
		goto error;
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

osync_bool osync_mapping_engine_ignore(OSyncMappingEngine *engine, OSyncError **error)
{
	OSyncObjEngine *objengine = NULL;
	OSyncArchive *archive = NULL;
	char *objtype = NULL;
	long long int id = 0;
	OSyncList *c = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p, %p)", __func__, engine, error);
	
	engine->conflict = FALSE;
	engine->synced = TRUE;

	objengine = engine->parent;
	archive = objengine->archive;
	objtype = objengine->objtype;
	id = osync_mapping_get_id(engine->mapping);

	for (c = engine->entries; c; c = c->next) {
		OSyncMappingEntryEngine *entry = c->data;
		osync_archive_save_ignored_conflict(archive, objtype, id, osync_change_get_changetype(entry->change), error);
	}

	osync_status_update_mapping(engine->parent->parent, engine, OSYNC_ENGINE_MAPPING_EVENT_SOLVED, NULL);
	engine->parent->conflicts = osync_list_remove(engine->parent->conflicts, engine);

	if (!osync_obj_engine_command(engine->parent, OSYNC_ENGINE_COMMAND_END_CONFLICTS, error))
		goto error;
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

osync_bool osync_mapping_engine_use_latest(OSyncMappingEngine *engine, OSyncError **error)
{
	OSyncMappingEntryEngine *latest_entry = NULL;
	osync_trace(TRACE_ENTRY, "%s(%p, %p)", __func__, engine, error);
	
	latest_entry = _osync_mapping_engine_get_latest_entry(engine, error);

	if (!latest_entry)
		goto error;

	osync_mapping_engine_set_master(engine, latest_entry);

	engine->conflict = FALSE;
	osync_status_update_mapping(engine->parent->parent, engine, OSYNC_ENGINE_MAPPING_EVENT_SOLVED, NULL);
	engine->parent->conflicts = osync_list_remove(engine->parent->conflicts, engine);

	if (!osync_obj_engine_command(engine->parent, OSYNC_ENGINE_COMMAND_END_CONFLICTS, error))
		goto error;
	
	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

static osync_bool _osync_change_elevate(OSyncChange *change, int level, osync_bool *dirty, OSyncError **error)
{
	int i = 0;
	for (i = 0; i < level; i++) {
		if (!osync_change_duplicate(change, dirty, error))
			return FALSE;
	}
	return TRUE;
}

osync_bool osync_mapping_engine_duplicate(OSyncMappingEngine *existingMapping, OSyncError **error)
{
	int elevation = 0;
	OSyncObjEngine *objengine = NULL;
	OSyncList *entries = NULL, *e = NULL, *mappings = NULL;

	osync_trace(TRACE_ENTRY, "%s(%p, %p)", __func__, existingMapping, error);
	g_assert(existingMapping);
	
	objengine = existingMapping->parent;
	
	/* Remove all deleted items first and copy the changes to a list */
	e = existingMapping->entries;
	for (; e; e = e->next) {
		OSyncMappingEntryEngine *entry = e->data;
		if (entry->change) {
			if (osync_change_get_changetype(entry->change) == OSYNC_CHANGE_TYPE_MODIFIED || osync_change_get_changetype(entry->change) == OSYNC_CHANGE_TYPE_ADDED) {
				osync_trace(TRACE_INTERNAL, "Appending entry %s, changetype %i from member %lli", osync_change_get_uid(entry->change), osync_change_get_changetype(entry->change), osync_member_get_id(osync_client_proxy_get_member(entry->sink_engine->proxy)));
		
				entries = osync_list_append(entries, entry);
			} else {
				osync_trace(TRACE_INTERNAL, "Removing entry %s, changetype %i from member %lli", osync_change_get_uid(entry->change), osync_change_get_changetype(entry->change), osync_member_get_id(osync_client_proxy_get_member(entry->sink_engine->proxy)));
				osync_entry_engine_update(entry, NULL);
			}
		} else {
			osync_trace(TRACE_INTERNAL, "member %lli does not have a entry", osync_member_get_id(osync_client_proxy_get_member(entry->sink_engine->proxy)));
		}
	}
	
	/* Create a list with mappings. In the beginning, only the exisiting mapping is in the list */
	mappings = osync_list_append(NULL, existingMapping);
	osync_mapping_engine_ref(existingMapping);
	
	while (entries) {
		OSyncMappingEntryEngine *existingEntry = entries->data;
		
		/* Now lets see which mapping is the correct one for the entry */
		OSyncList *m = NULL;
		OSyncMappingEngine *mapping = NULL;
		OSyncChange *existingChange = NULL;
		osync_bool dirty = FALSE;
		OSyncMappingEntryEngine *newEntry = NULL;

		elevation = 0;
		for (m = mappings; m; m = m->next) {
			OSyncList *e = NULL;
			OSyncChange *change = NULL;
			OSyncMappingEntryEngine *entry = NULL;
			OSyncConvCmpResult cmpret;
			mapping = m->data;
			
			/* Get the first change of the mapping to test. Compare the given change with this change.
			 * If they are not the same, we have found a new mapping */
			for (e = mapping->entries; e; e = e->next) {
				entry = e->data;
				change = entry->change;
				if (change)
					break;
			}
			
			if (change)
				cmpret = osync_change_compare(existingEntry->change, change, error);

			if (!change  || cmpret == OSYNC_CONV_DATA_SAME) {
				existingChange = existingEntry->change;
				osync_change_ref(existingChange);
				osync_assert(osync_change_get_uid(existingChange));
				break;
			} else if (cmpret == OSYNC_CONV_DATA_UNKNOWN) {
				/* This is an error during comparing */
				goto error;
				break;
			}


			mapping = NULL;
			elevation++;
		
			existingChange = osync_change_clone(existingEntry->change, error);
			osync_assert(osync_change_get_uid(existingChange));
		}
		
		
		if (!mapping) {
			/* Unable to find a mapping. We have to create a new one */
			mapping = _osync_obj_engine_create_mapping_engine(objengine, error);
			if (!mapping)
				goto error;
			mappings = osync_list_append(mappings, mapping);
			objengine->mapping_engines = osync_list_append(objengine->mapping_engines, mapping);
			osync_mapping_engine_ref(mapping);
		}
		
		/* update the uid and the content to suit the new level */
		if (!_osync_change_elevate(existingChange, elevation, &dirty, error))
			goto error;

		/* Lets add the entry to the mapping */
		newEntry = osync_mapping_engine_get_entry(mapping, existingEntry->sink_engine);
		osync_assert(newEntry);
		osync_entry_engine_update(newEntry, existingChange);
		osync_mapping_entry_set_uid(newEntry->entry, osync_change_get_uid(existingChange));
		osync_change_unref(existingChange);
		
		/* Set the last entry as the master */
		osync_mapping_engine_set_master(mapping, newEntry);
		
		/* Update the dirty status. If the duplicate function said
		 * that the returned item needs to be written, we will set
		 * this information here */
		newEntry->dirty = dirty;
		
		entries = osync_list_remove(entries, existingEntry);
	}
	
	
	while (mappings) {
		OSyncMappingEngine *mapping = mappings->data;
		osync_mapping_engine_unref(mapping);
		mappings = osync_list_remove(mappings, mapping);
	}
	
	objengine->conflicts = osync_list_remove(objengine->conflicts, existingMapping);
	osync_status_update_mapping(objengine->parent, existingMapping, OSYNC_ENGINE_MAPPING_EVENT_SOLVED, NULL);

	if (!osync_obj_engine_command(objengine, OSYNC_ENGINE_COMMAND_END_CONFLICTS, error))
		goto error;

	osync_trace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	while (mappings) {
		OSyncMappingEngine *mapping = mappings->data;
		osync_mapping_engine_unref(mapping);
		mappings = osync_list_remove(mappings, mapping);
	}
	osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(error));
	return FALSE;
}

OSyncList *osync_mapping_engine_get_changes(OSyncMappingEngine *engine) {
	OSyncList *list = engine->entries;
	OSyncList *new_list = NULL;
	OSyncMappingEntryEngine *entry = NULL;
	
	if (list) {
		OSyncList *last;

		new_list = osync_list_alloc();
		entry = list->data;
		new_list->data = entry->change;
		new_list->prev = NULL;
		last = new_list;
		list = list->next;
		while (list) {
			last->next = osync_list_alloc();
			last->next->prev = last;
			last = last->next;
			entry = list->data;
			last->data = entry->change;
			list = list->next;
		}
		last->next = NULL;
	}

	return new_list;
}

