#!/usr/bin/env bash
#############################################################################################

VG=$(which valgrind)
VGOPT="--trace-children=yes --leak-check=full --show-reachable=yes --leak-resolution=high"
VERBOSE=0
DEFAULT_REGEX='^.*(oscap-scan|test_probes|probe_).*$'

#############################################################################################
function imsg {
    if (( $VERBOSE == 1 )); then
        echo -n '[i] '
        echo $*
    fi
}

function wmsg {
    echo -n '[w] '
    echo $*
}

function emsg {
    echo -n '[e] '
    echo $*
}

function cpy {
    local src=$1
    local dst=$2
    
    imsg "$src ==> $dst"
    cp   "$src"   "$dst"
}
#############################################################################################

prog="$(basename "$0")"
asdf=$(getopt -o r:vh --long regex,verbose,help -n vgrun.sh -- "$@")

if (( $? != 0 )); then
    exit 1
fi

eval set -- "$asdf"

while true; do
    case "$1" in
        -r|--regex)
            REGEX="$2"
            shift 2
            ;;
        -v|--verbose)
            VERBOSE=1
            shift
            ;;
        -h|--help)
            echo "Usage: $prog [-hr] <command>"
            echo "       -r, --regex    Filename regex (extended)"
            echo "       -v, --verbose  Be verbose"
            echo "       -h, --help     This help"
            echo ""
            exit 0
            ;;
        --)
            shift; break;;
        *)
            emsg "wtf?"
            exit 1
            ;;
    esac
done

COMM="$1"

if [[ -z "$VG" ]]; then
    emsg "Please intall valgrind first or correct your \$PATH."
    exit 1
else
    VER="$($VG --version)"
    case "$VER" in
        valgrind-3.[345].*)
            ;;
        *)
            wmsg "The version of Valgrind ($VG) you are using wasn't tested with this script. Don't trust the output of this script."
            ;;
    esac
fi

if [[ -z "$COMM" ]]; then
    emsg "Executing rm -rf / ... this may take a while"
    exit 2
fi

if [[ -z "$REGEX" ]]; then
    REGEX="$DEFAULT_REGEX"
    imsg "Using default filename regex: $REGEX"
fi

TMPDIR=$(mktemp -d)

imsg " outdir: $TMPDIR"
imsg "command: $VG $VGOPT --log-file=\"$TMPDIR/output.%p\" -- $COMM"

PREFIX="$$-$(date +%H%M)"
SUFFIX=".vglog"

echo "=== ID: $PREFIX === CMD: $COMM ==="

$VG $VGOPT --log-file="$TMPDIR/output.%p" -- $COMM > /dev/null 2>&1

if (( $? == 134 )); then
    exit 134
fi

c=0
LOG=()

for log in "$TMPDIR"/output.*; do
    log_cmd0="$(sed -n 's|^.*Command:[[:space:]]*\(.*\)[[:space:]]*$|\1|p' "$log")"

    if [[ -z "$log_cmd0" ]]; then
    	# Different output format
    	log_cmd0="$(grep -A 1 '^.*My PID = [0-9]*, parent PID = [0-9]*\.[[:space:]]*Prog and args are:.*$' "$log")"

    	if [[ -z "$log_cmd0" ]]; then
    	    emsg "Don't know how to parse valgrind output :["
    		# TODO: xml output parsing?
    		exit 3
    	fi

    	log_cmd0="$(echo "$log_cmd0" | tail -n 1 | sed -n 's|^==[0-9]*==[[:space:]]*\(.*\)$|\1|p')"
    fi
    
    log_cmd1="$(basename "$(echo "$log_cmd0" | sed -n 's|^\([^[:space:]]*\).*$|\1|p')")"

    if echo "$log_cmd1" | egrep -q "$REGEX"; then
        outfile="$PREFIX-$c.$log_cmd1$SUFFIX"
        cpy "$log" "$outfile"
        LOG[$c]="$outfile"
        CMD[$c]="$log_cmd1"
        c=$(($c + 1))        
    fi
done

for ((i=0; i < ${#LOG[@]}; i++)); do
    num_le="$(egrep -ci "(lost in loss record|still reachable in loss record)" "${LOG[$i]}")"
    num_ir="$(grep -ci  "invalid read of size"                                 "${LOG[$i]}")"
    num_iw="$(grep -ci  "invalid write of size"                                "${LOG[$i]}")"
    num_ij="$(grep -ci  "conditional jump or move depends"                     "${LOG[$i]}")"
    num_iv="$(grep -ci  "use of uninitialized value"                           "${LOG[$i]}")"

    sum=$(($num_le + $num_ir + $num_iw + $num_ij + $num_iv))

    printf '[ %28s ] leaks=%3d, inv.r/w=%3d/%3d, inv.jumps=%3d, inv.vals=%3d .... E= %d\n' "${CMD[i]}" $num_le $num_ir $num_iw $num_ij $num_iv $sum

    if (( $sum == 0 )); then
        rm "${LOG[$i]}"
    fi
done
echo ""
