# Copyright (C) 2000-2001 The OpenRPG Project
#
#    openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: mapper/gird.py
# Author: OpenRPG Team
# Maintainer:
# Version:
#   $Id: grid.py,v 1.17 2005/06/05 20:12:38 digitalxero Exp $
#
# Description:
#
__version__ = "$Id: grid.py,v 1.17 2005/06/05 20:12:38 digitalxero Exp $"

from base import *
from isometric import *

# Grid mode constants
GRID_RECTANGLE = 0
GRID_HEXAGON = 1
GRID_ISOMETRIC = 2
LINE_NONE = 0
LINE_DOTTED = 1
LINE_SOLID = 2

RATIO_DEFAULT = 2.0

##-----------------------------
## grid layer
##-----------------------------
class grid_layer(layer_base):

    def __init__(self, canvas):
        layer_base.__init__(self)
        self._protected_attr = ["unit_size","snap","color","mode","line"]
        self.canvas = canvas
        self.iso_ratio = RATIO_DEFAULT  #2:1 isometric ratio
        self.mapscale = 1
        self.unit_size = 100
        self.unit_size_y = 100
        #unit_widest and unit_offset are for the Hex Grid only. these are mathmatics to figure out the exact center of the hex
        self.unit_widest = 100
        self.unit_offset = 100
        #size_ratio is the size ajustment for Hex and ISO to make them more accurate
        self.size_ratio = 1.5

        self.snap = true
        color = wxBLACK
        (r,g,b) = color.Get()
        self.color = cmpColour(r,g,b)
        self.r_h = RGBHex()
        self.mode = GRID_RECTANGLE
        self.line = LINE_NONE
        # Keep logic for different modes in different functions
        self.grid_hit_test = self.grid_hit_test_rect
        self.get_top_corner = self.get_top_corner_rect
        self.draw = self.draw_rect
        self._clean_all_attr()

    def get_unit_size(self):
        return self.unit_size

    def get_iso_ratio(self):
        return self.iso_ratio

    def get_mode(self):
        return self.mode

    def get_color(self):
        return self.color

    def get_line_type(self):
        return self.line

    def is_snap(self):
        return self.snap

    def set_rect_mode(self):
        "switch grid to rectangular mode"
        self.mode = GRID_RECTANGLE
        self.grid_hit_test = self.grid_hit_test_rect
        self.get_top_corner = self.get_top_corner_rect
        self.draw = self.draw_rect
        self.unit_size_y = self.unit_size

    def set_hex_mode(self):
        "switch grid to hexagonal mode"
        self.mode = GRID_HEXAGON
        self.grid_hit_test = self.grid_hit_test_hex
        self.get_top_corner = self.get_top_corner_hex
        self.draw = self.draw_hex
        self.unit_size_y = self.unit_size
        self.unit_offset = sqrt(pow((self.unit_size/self.size_ratio ),2)-pow((self.unit_size/2),2))
        self.unit_widest = (self.unit_offset*2)+(self.unit_size/self.size_ratio )

    def set_iso_mode(self):
        "switch grid to hexagonal mode"
        self.mode = GRID_ISOMETRIC
        self.grid_hit_test = self.grid_hit_test_iso
        self.get_top_corner = self.get_top_corner_iso
        self.draw = self.draw_iso
        self.unit_size_y = self.unit_size

    def set_line_none(self):
        "switch to no line mode for grid"
        self.line = LINE_NONE

    def set_line_dotted(self):
        "switch to dotted line mode for grid"
        self.line = LINE_DOTTED

    def set_line_solid(self):
        "switch to solid line mode for grid"
        self.line = LINE_SOLID

    def grid_hit_test_rect(self,pos):
        "return grid pos (w,h) on rect map from pos"
        if self.unit_size and self.snap:
            return (pos.x/self.unit_size,pos.y/self.unit_size)
        else:
            return None

    def grid_hit_test_hex(self,pos):
        "return grid pos (w,h) on hex map from pos"
        if self.unit_size and self.snap:
            # Which row is this is? Rows start at wierd multiples of
            # unit_size, and we need to shift our value so that things
            # stick to the intuitiviely correct cell
            temp_w = int(pos.x/self.unit_size)
            # Are we in an even or odd column? Adjust accordingly
            #print temp_w
            if temp_w % 2:
                temp_h = int((pos.y-self.unit_size_y/2)/self.unit_size_y)
            else:
                temp_h = int(pos.y/self.unit_size_y)
            #print temp_h
            return (temp_w,temp_h)
        else:
            return None

    def grid_hit_test_iso(self,pos):
        "return grid pos (w,h) on isometric map from pos"
        if self.unit_size and self.snap:
            # create IsoGrid helper object
            IG = IsoGrid(self.unit_size*self.size_ratio)
            IG.Ratio(self.iso_ratio)
            dx,dy = IG.CornerOffset()

            #This code is required to make the snap work with ISO Grid
            #It is some what clumsy, but I could figure out no other way to get the snap working correctly
            #Dj Gilcrease
            sz = self.canvas.size
            rows = int(min(pos.y+0,sz[1])/IG.height)
            cols = int(min(pos.x+0,sz[0])/IG.width)
            for y in range(rows+1):
                for x in range(cols+1):
                    IG.BoundPlace((x*IG.width),(y*IG.height))
                    x1,y1 = IG.Top()
                    x2,y2 = IG.Left()
                    m = (y2-y1)/(x2-x1)
                    ul_y = (pos.y-y1)
                    ul_x = m*(pos.x-x1)

                    x1,y1 = IG.Left()
                    x2,y2 = IG.Bottom()
                    m = (y2-y1)/(x2-x1)
                    bl_y = (pos.y-y1)
                    bl_x = m*(pos.x-x1)

                    x1,y1 = IG.Bottom()
                    x2,y2 = IG.Right()
                    m = (y2-y1)/(x2-x1)
                    br_y = (pos.y-y1)
                    br_x = m*(pos.x-x1)

                    x1,y1 = IG.Right()
                    x2,y2 = IG.Top()
                    m = (y2-y1)/(x2-x1)
                    ur_y = (pos.y-y1)
                    ur_x = m*(pos.x-x1)


            temp = IG.Center()
            temp_w = temp[0]
            temp_h = temp[1]

            if ul_y < ul_x:
                temp_w -= dx
                if bl_y > bl_x:
                    temp_h += dy
                else:
                    temp_h -= dy
            elif bl_y > bl_x and ur_y > ur_x:
                temp_h += dy
                if br_y > br_x:
                    temp_w += dx
                else:
                    temp_w -= dx
            elif ur_y < ur_x and br_y < br_x and bl_y < bl_x and ul_y > ul_x:
                temp_w += dx
                temp_h -= dy
            elif ur_y > ur_x and br_y > br_x and bl_y < bl_x and ul_y > ul_x:
                temp_w += dx
                temp_h += dy

            return (int(temp_w),int(temp_h))
        else:
            return None

    def get_top_corner_iso(self,grid_pos):
        "return upper left of a iso grid pos"
        if self.unit_size:
            return cmpPoint(grid_pos[0],grid_pos[1])
        else:
            return None

    def get_top_corner_rect(self,grid_pos):
        "return upper left of a rect grid pos"
        if self.unit_size:
            return cmpPoint(grid_pos[0]*self.unit_size,grid_pos[1]*self.unit_size)
        else:
            return None

    def get_top_corner_hex(self,grid_pos):
        "return upper left of a hex grid pos"
        if self.unit_size:
            # We can get our x value directly, y is trickier
            temp_x = (1.5*self.unit_size*grid_pos[0])
            temp_y = self.unit_size_y*grid_pos[1]
            # On odd columns we have to slide down slightly
            if grid_pos[0] % 2:
                temp_y += self.unit_size_y/2
            return cmpPoint(temp_x,temp_y)
        else:
            return None

    def set_grid(self,unit_size,snap,color,mode,line,ratio = None):
        self.unit_size = unit_size
        if ratio != None:
            self.iso_ratio = ratio
        self.snap = snap
        self.set_color(color)
        self.SetMode(mode)
        self.SetLine(line)

    def SetLine(self,line):
        if line == LINE_NONE:
            self.set_line_none()
        elif line == LINE_DOTTED:
            self.set_line_dotted()
        elif line == LINE_SOLID:
            self.set_line_solid()

    def SetMode(self, mode):
        if mode == GRID_RECTANGLE:
            self.set_rect_mode()
        elif mode == GRID_HEXAGON:
            self.set_hex_mode()
        elif mode == GRID_ISOMETRIC:
            self.set_iso_mode()

    def return_grid(self):
        return self.canvas.size

    def set_color(self,color):
        (r,g,b) = color.Get()
        self.color = cmpColour(r,g,b)

    def draw_iso(self,dc,topleft,clientsize):
        if not self.unit_size: return
        if self.line == LINE_NONE: return


        if self.line == LINE_SOLID:
            dc.SetPen(wxPen(self.color,1,wxSOLID))

        else:
            dc.SetPen(wxPen(self.color,1,wxDOT))


        sz = self.canvas.size

        # Enable DC optimizations if available on a platform
        dc.BeginDrawing()

        # create IsoGrid helper object
        IG = IsoGrid(self.unit_size*self.size_ratio)
        IG.Ratio(self.iso_ratio)

        rows = int(min(clientsize[1]+topleft[1],sz[1])/IG.height)
        cols = int(min(clientsize[0]+topleft[0],sz[0])/IG.width)


        for y in range(rows+1):
            for x in range(cols+1):
                IG.BoundPlace((x*IG.width),(y*IG.height))
                x1,y1 = IG.Top()
                x2,y2 = IG.Left()
                dc.DrawLine(x1,y1,x2,y2)
                x1,y1 = IG.Left()
                x2,y2 = IG.Bottom()
                dc.DrawLine(x1,y1,x2,y2)
                x1,y1 = IG.Bottom()
                x2,y2 = IG.Right()
                dc.DrawLine(x1,y1,x2,y2)
                x1,y1 = IG.Right()
                x2,y2 = IG.Top()
                dc.DrawLine(x1,y1,x2,y2)


        # Enable DC optimizations if available on a platform
        dc.EndDrawing()

        dc.SetPen(wxNullPen)

        # Disable pen/brush optimizations to prevent any odd effects elsewhere




    def draw_rect(self,dc,topleft,clientsize):
        if self.unit_size:
            draw = 1

            # Enable pen/brush optimizations if available on a platform

            if self.line == LINE_NONE:
                draw = 0

            elif self.line == LINE_SOLID:
                dc.SetPen(wxPen(self.color,1,wxSOLID))

            else:
                dc.SetPen(wxPen(self.color,1,wxDOT))

            if draw:
                sz = self.canvas.size

                # Enable DC optimizations if available on a platform
                dc.BeginDrawing()

                # Now, draw the map grid
                x = 0
                s = self.unit_size
                x = int(topleft[0]/s)*s
                mx = min(clientsize[0]+topleft[0],sz[0])
                my = min(clientsize[1]+topleft[1],sz[1])
                while x < mx:
                    dc.DrawLine(x,topleft[1],x,my)
                    x += self.unit_size

                y = 0
                y = int (topleft[1]/s)*s
                while y < my:
                    dc.DrawLine(topleft[0],y,mx,y)
                    y += self.unit_size

                # Enable DC optimizations if available on a platform
                dc.EndDrawing()

                dc.SetPen(wxNullPen)

            # Disable pen/brush optimizations to prevent any odd effects elsewhere



    def draw_hex(self,dc,topleft,clientsize):
        if self.unit_size:
            draw = 1

            # Enable pen/brush optimizations if available on a platform

            if self.line == LINE_NONE:
                draw = 0

            elif self.line == LINE_SOLID:
                dc.SetPen(wxPen(self.color,1,wxSOLID))

            else:
                dc.SetPen(wxPen(self.color,1,wxDOT))

            if draw:
                sz = self.canvas.size

                # Enable DC optimizations if available on a platform
                dc.BeginDrawing()
                x = 0
                us = self.unit_size/self.size_ratio
                usy = self.unit_size_y
                usxy = self.unit_size


                startx=int(topleft[0]/(3*us))*3*us
                starty=int(topleft[1]/usy)*usy



                y = starty
                mx = min(clientsize[0]+topleft[0],sz[0])
                my = min(clientsize[1]+topleft[1],sz[1])
                while y < my:
                    x = startx
                    lineArray = []
                    while x < mx:
                        lineArray.append(wxPoint(x,y))
                        lineArray.append(wxPoint(x+us,y))
                        lineArray.append(wxPoint(x+usxy,y+usy/2))
                        lineArray.append(wxPoint(x+usxy+us,y+usy/2))
                        dc.DrawLine(x+usxy,y+usy/2,x+us,y+usy)
                        dc.DrawLine(x+usxy+us,y+usy/2,x+3*us,y+usy)
                        x += 3*us
                    y += usy
                    dc.DrawLines(lineArray)


                # Enable DC optimizations if available on a platform
                dc.EndDrawing()

                dc.SetPen(wxNullPen)

            # Disable pen/brush optimizations to prevent any odd effects elsewhere



    def toxml(self,action = "update"):

        xml_str = ""

        if action == "new":
            self._dirty_all_attr()

        changed = self._changed_attr()

        if changed:
            xml_str = "<grid"

            for a in changed.keys():
                if a == "color":
                    if not (self.color is None):
                        (red,green,blue) = self.color.Get()
                        hexcolor = self.r_h.hexstring(red, green, blue)
                        xml_str += " color='" + hexcolor + "'"
                elif a == "unit_size":
                    if not (self.unit_size is None):
                        xml_str += " size='" + str(self.unit_size) + "'"
                elif a == "iso_ratio":
                    if not (self.iso_ratio is None):
                        xml_str += " ratio='" + str(self.iso_ratio) + "'"
                elif a == "snap":
                    if not (self.snap is None):
                        xml_str += " snap='" + str(self.snap) + "'"
                elif a == "mode":
                    if not (self.mode is None):
                        xml_str+= "  mode='" + str(self.mode) + "'"
                elif a == "line":
                    if not (self.line is None):
                        xml_str+= " line='" + str(self.line) + "'"

            xml_str += "/>"

        self._clean_all_attr()
        return xml_str


    def takedom(self,xml_dom):
        color = xml_dom.getAttribute("color")
        if color <> "":
            r,g,b = self.r_h.rgb_tuple(color)
            self.set_color(cmpColour(r,g,b))
            self._clean_attr("color")

        size = xml_dom.getAttribute("size")

        #backwards compatible with non-isometric map formated clients
        try:
            ratio = xml_dom.getAttribute("ratio")
        except:
            ratio = RATIO_DEFAULT

        if size <> "":
            self.unit_size = int(size)
            if self.mode == GRID_RECTANGLE:
                self.unit_size_y = self.unit_size
            if self.mode == GRID_ISOMETRIC:
                self.unit_size_y = self.unit_size
            elif self.mode == GRID_HEXAGON:
                self.unit_size_y = self.unit_size*(3**0.5)
            self._clean_attr("unit_size")

        snap = xml_dom.getAttribute("snap")

        if snap <> "":
            if (snap == 'True') or (snap == "1"):
                snap = 1
            else:
                snap = 0
            self.snap = snap
            self._clean_attr("snap")


        mode = xml_dom.getAttribute("mode")
        if mode <> "":
            self.SetMode(int(mode))
            self._clean_attr("mode")

        line = xml_dom.getAttribute("line")
        if line <> "":
            self.SetLine(int(line))
            self._clean_attr("line")

