/*
 *
 * olpc-kbdshim.c -- low-level support for XO:
 *  - mouse-based scrolling use the XO grab keys
 *  - touchpad and dpad rotation (to match screen rotation)
 *  - user activity monitoring

NOTE:  this version of the code is a standalone daemon,
pre-dating the HAL addon version.  There are newer features
implemented in olpc-kbdshim-hal which are not implemented here.
In particular, the -hal version can monitor USB devices automatically,
where this version cannot.  There are other smaller differences
as well.

 *
 * Copyright (C) 2009, Paul G Fox, inspired in places
 *  by code from mouseemu, by Colin Leroy and others.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <wait.h>
#include <syslog.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sched.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <linux/input.h>
#include <linux/uinput.h>
#include <time.h>
#include <utime.h>
#include <limits.h>

static void inject_uinput_event(unsigned int type, unsigned int code,
        int value);

/* are we still in the foreground? */
int daemonized;

/* log to syslog (instead of stderr) */
int logtosyslog;

/* higher values for more debug */
int debug;

/* suppress any actual tranmission or injection of data */
int noxmit;

/* which keys do grabbing? */
#define NGRABKEYS 4
struct keypress {
    int code;
    int pass;
    int pressed;
} grabkey[NGRABKEYS];
int ngrab;

/* command fifo */
char *command_fifo;
int fifo_fd = -1;

/* sysactive path -- where we send user activity/idle events */
char *sysactive_path = "/var/run/powerevents";
#define MAXTIMERS 3
/* list of idle deltas.  last timer should stay zero */
int idledeltas[MAXTIMERS + 1] = { 10 * 60, 5 * 60, 5 * 60, 0 };;

/* are we in user-idle state? */
int idleness;
int wasidle;

/* external commands bound to rotate and brightness keys:
 *
 * the rotate command should inform us via our command fifo when
 * a rotate has happened, so that we can adjust the touchpad and
 * d-pad rotation.
 *
 * the brightness command should accept a single argument of
 * "up", "down", "min", or "max" (the last two when the alt
 * modifier is in effect).
 *
 * these bindings are done here for convenience, and could be
 * done anywhere else in the system just as well.  binding
 * them here effectively "steals" them from sugar, which has
 * internal implementations of rotation and brightness control
 * which are no longer correct when this package and olpc-powerd
 * are used.
 */
char *brightness_cmd; /*  probably "olpc-brightness" */
char *volume_cmd;     /*  probably "olpc-volume" */
char *rotate_cmd;     /*  probably "olpc-rotate" */

int reflect_x, reflect_y;  /* inverted mouse (e.g., for ebook mode) */
enum rotation {
    ROT_NORMAL,
    ROT_RIGHT,
    ROT_INVERT,
    ROT_LEFT
} rotation;

/* are we scrolling, or not */
int scrolling;

#define SCROLL_QUANTUM 20
#define SCROLL_FILTER_PERCENT 33
int cumul_x, cumul_y;

/* "distance traveled" before we emulate a scroll button event */
int quantum = SCROLL_QUANTUM;
int ratio = SCROLL_FILTER_PERCENT;

/* if true, finger moves scrollbars, instead of window contents */
int reverse;

/* product, vendor id for keyboard and touchpad */
struct input_id kbd_id = {
    bustype: BUS_I8042,
    product: 0xffff,
    vendor: 0xffff
};
struct input_id tpad_id = {
    bustype: BUS_I8042,
    product: 0xffff,
    vendor: 0xffff
};

extern char *optarg;
extern int optind, opterr, optopt;

/* output events device */
int uinp_fd = -1;

/* input event devices */
int kbd_fd = -1;
int tpad_fd = -1;

/* bit array ops */
#define bits2bytes(x) ((((x)-1)/8)+1)
#define test_bit(bit, array) ( array[(bit)/8] & (1 << (bit)%8) )

#define MAX(a, b) (((a) > (b)) ? (a) : (b))

/* how many /dev/input/eventX devices to check */
#define EVENTDEVICES 20

typedef struct input_id input_id_t;


char *me;

void
usage(void)
{
    fprintf(stderr,
        "usage: %s [options]\n"
        " Grab-scroll configuration:\n"
        "   '-g N','-G N' Specify which key(s) are used for grabbing.\n"
        "        A maximum of %d keys can be specified.  (Use \"showkey\"\n"
        "        at the console to find the keycode.)  With '-g', the\n"
        "        key will be processed normally (in addition to instituting\n"
        "        scrolling).  '-G' will cause scrolling but suppress the\n"
        "        key's normal action.  Default is '-g 125 -g 126'.  (The\n"
        "        \"meta\" keys, labeled as \"grab\" keys on the XO laptop.\n"
        "        Use '-g 0' to suppress grab-scoll support.\n"
        "   '-v' to reverse the relative motion of mouse and window\n"
        "        By default the mouse/finger slides the window, but\n"
        "        with -r it effectively slides the scrollbars instead.\n"
        "   '-q N' to set how far the mouse/finger must move before a\n"
        "        scrolling event is generated"
                   " (default is %d).\n"
        "   '-n N' If the x or y motion is less than N percent of the\n"
        "        other (default N is %d) then the smaller of the two is\n"
        "        dropped. This reduces sideways jitter when scrolling\n"
        "        vertically,and vice versa.\n"
        "\n"
        " Touchpad/D-pad rotation:\n"
        "   '-R <fifoname>'  If set, this gives the name of a fifo which\n"
        "        will be created and monitored for the purpose of rotating\n"
        "        and reflecting the touchpad and the D-pad.  This can be\n"
        "        used to make their function match the screen's rotation.\n"
        "        This fifo will also receive setup commands related to\n"
        "        user activity monitoring (see below).\n"
        "\n"
        " Keyboard/touchpad identification:\n"
        "   '-K BB:VV:PP' Specifies the bustype, vendor, and product id\n"
        "        for the keyboard to be monitored.  BB, VV, and PP are hex\n"
        "        numbers.  ffff can be used as a wildcard for any of them.\n"
        "        Default is '11:ffff:ffff' for the i8042 bus, which will find\n"
        "        the local keyboard.  '03:ffff:ffff' finds any USB keyboard.\n"
        "   '-T BB:VV:PP' Same as -K, but for the pointer device.\n"
        "\n"
        " User activity monitor:\n"
        "   '-A PATH' Specifies path where user idle/activity events will\n"
        "        be written (default is /var/run/powerevents).\n"
        "\n"
        " Key bindings:\n"
        "   '-r rotate-cmd'  If set, the command to be run when the XO\n"
        "        'rotate' button is detected.  (Will be passed no arguments.)\n"
        "   '-b brightness-cmd'  If set, the command to be run when the XO\n"
        "        brightness keys are used.  (Should accept 'min', 'max', 'up'\n"
        "        or 'down' as the sole argument.)\n"
        "   '-V volume-cmd'  If set, the command to be run when the XO\n"
        "        volume keys are used.  (Should accept 'min', 'max', 'up'\n"
        "        or 'down' as the sole argument.)\n"
        "\n"
        " Daemon options:\n"
        "   '-f' to keep program in foreground.\n"
        "   '-l' use syslog, rather than stderr, for messages.\n"
        "   '-s' to run with elevated (sched_fifo) scheduling priority.\n"
        "   '-d' for debugging (repeate for more verbosity).\n"
        "   '-X' don't actually pass on received keystrokes (for debug).\n"
        "(olpc-kbdshim version %d)\n"
        , me, NGRABKEYS, SCROLL_QUANTUM, SCROLL_FILTER_PERCENT, VERSION);
    exit(1);
}

static void
report(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    if (logtosyslog && debug <= 1) {
        vsyslog(LOG_NOTICE, fmt, ap);
    } else {
        fprintf(stderr, "%s: ", me);
        vfprintf(stderr, fmt, ap);
        fputc('\n', stderr);
    }
}

static void
dbg(int level, const char *fmt, ...)
{
    va_list ap;

    if (debug < level) return;

    va_start(ap, fmt);
    if (logtosyslog && debug <= 1) {
        vsyslog(LOG_NOTICE, fmt, ap);
    } else {
        fputc(' ', stderr);
        vfprintf(stderr, fmt, ap);
        fputc('\n', stderr);
    }
}

void
die(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    if (logtosyslog && debug <= 1) {
        vsyslog(LOG_ERR, fmt, ap);
        syslog(LOG_ERR, "exiting -- %m");
    } else {
        fprintf(stderr, "%s: ", me);
        vfprintf(stderr, fmt, ap);
        fprintf(stderr, " - %s", strerror(errno));
        fputc('\n', stderr);
    }
    exit(1);
}

/* Manage the uinput device */

void
deinit_uinput_device(void)
{
    if (uinp_fd < 0) return;

    /* Destroy the input device */
    if (ioctl(uinp_fd, UI_DEV_DESTROY) < 0)
        die("destroy of uinput dev failed\n");

    /* Close the UINPUT device */
    close(uinp_fd);
    uinp_fd = -1;
}

int
setup_uinput(void)
{
    struct uinput_user_dev uinp;
    int e, i;

    uinp_fd = open("/dev/input/uinput", O_WRONLY | O_NDELAY);
    if (uinp_fd < 0) {
        uinp_fd = open("/dev/uinput", O_WRONLY | O_NDELAY);
        if (uinp_fd < 0) {
            report(
                "Unable to open either /dev/input/uinput or /dev/uinput\n");
            return -1;
        }
    }

    atexit(deinit_uinput_device);

    memset(&uinp, 0, sizeof(uinp));

    strncpy(uinp.name, "olpc-kbdshim virtual input", UINPUT_MAX_NAME_SIZE);
    uinp.id.bustype = BUS_VIRTUAL;
    uinp.id.vendor  = 'p';
    uinp.id.product = 'g';
    uinp.id.version = 'f';

    for (i = 0; i <= KEY_UNKNOWN; i++) {
        if (ioctl(uinp_fd, UI_SET_KEYBIT, i) != 0) {
            report("uinput setup failed, code %d\n", i);
            return -1;
        }
    }

    e = 0;
    if (!++e || ioctl(uinp_fd, UI_SET_EVBIT, EV_KEY) < 0 || // keys
        !++e || ioctl(uinp_fd, UI_SET_EVBIT, EV_REP) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_EVBIT, EV_REL) < 0 || // mouse
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_X) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_Y) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_WHEEL) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_HWHEEL) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_KEYBIT, BTN_LEFT) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_KEYBIT, BTN_RIGHT) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_KEYBIT, BTN_MIDDLE) < 0 ||
        !++e || write(uinp_fd, &uinp, sizeof(uinp)) < 0 ||  // device
        !++e || ioctl(uinp_fd, UI_DEV_CREATE) < 0) {
            report("uinput setup failed, step %d\n", e);
            return -1;
    }

    /* disable timer-based autorepeat, see http://dev.laptop.org/ticket/9690 */
    inject_uinput_event(EV_REP, REP_DELAY, 0);
    inject_uinput_event(EV_REP, REP_PERIOD, 0);

    return 0;
}

int
match_input_id(input_id_t *id, input_id_t *tmpl)
{
    return id->bustype == tmpl->bustype &&
        (id->vendor == tmpl->vendor || tmpl->vendor == 0xffff) &&
        (id->product == tmpl->product || tmpl->product == 0xffff);
}

int
setup_input()
{
    int i;
    int dfd;
    char devname[128];
    unsigned char bit[bits2bytes(EV_MAX)];
    struct input_id id;

    for (i = 0; i < EVENTDEVICES; i++) {

        snprintf(devname, sizeof(devname), "/dev/input/event%d", i);
        if ((dfd = open(devname, O_RDONLY)) < 0)
            continue;

        if (ioctl(dfd, EVIOCGID, &id) < 0) {
            report("failed ioctl EVIOCGID on %d", i);
            close(dfd);
            continue;
        }

        if (ioctl(dfd, EVIOCGBIT(0, EV_MAX), bit) < 0) {
            report("failed ioctl EVIOCGBIT on %d", i);
            close(dfd);
            continue;
        }

        if (kbd_fd < 0 &&
                test_bit(EV_KEY, bit) &&
                test_bit(EV_REP, bit) &&
                match_input_id(&id, &kbd_id)) {
            kbd_fd = dfd;
            if (ioctl(kbd_fd, EVIOCGRAB, 1) < 0)
                die("ioctl EVIOCGRAB on keyboard");
            report("found keyboard %s (%02x:%02x:%02x)", devname,
                id.bustype, id.vendor, id.product);
            continue;
        }

        if (tpad_fd < 0 &&
                test_bit(EV_REL, bit) &&
                match_input_id(&id, &tpad_id)) {
            tpad_fd = dfd;
            if (ioctl(tpad_fd, EVIOCGRAB, 1) < 0)
                die("ioctl EVIOCGRAB on touchpad");
            report("found touchpad %s (%02x:%02x:%02x)", devname,
                id.bustype, id.vendor, id.product);
            continue;
        }

        close(dfd);
    }

    if (kbd_fd == -1)
        report("didn't find keyboard");
    if (tpad_fd == -1)
        report("didn't find touchpad");

    return (kbd_fd >= 0 && tpad_fd >= 0);
}

void
inject_uinput_event(unsigned int type, unsigned int code, int value)
{
    struct input_event event;

    gettimeofday(&event.time, NULL);
    event.type = type;
    event.code = code;
    event.value = value;
    if (!noxmit && write(uinp_fd, &event, sizeof(event)) < 0) {
        report("warning: inject event failed, t/c/v 0x%x/0x%x/0x%x\n",
            type, code, value);
    }

}

void
send_a_scroll(int x, int y)
{
    if (reverse) {
        x = -x;
        y = -y;
    }

    if (x)  {
        dbg(1, "scroll %s", y > 0 ? "left" : "right");
        inject_uinput_event(EV_REL, REL_HWHEEL, x < 0 ? 1 : -1);
    }
    if (y) {
        dbg(1, "scroll %s", y > 0 ? "up" : "down");
        inject_uinput_event(EV_REL, REL_WHEEL, y < 0 ? -1 : 1);
    }
    inject_uinput_event(EV_SYN, SYN_REPORT, 0);
}

unsigned char dpad[] = { KEY_KP8, KEY_KP6, KEY_KP2, KEY_KP4,
                         KEY_KP8, KEY_KP6, KEY_KP2, KEY_KP4};

int keyboard_event(void)
{
    struct input_event ev[1];
    int pass = 1;
    int i;
    static int altdown;

    if (read(kbd_fd, ev, sizeof(ev)) != sizeof(ev))
        die("bad read from keyboard");

    dbg(3, "evtype %d code %d", ev->type, ev->code);

    if (ev->type == EV_KEY) {
        for (i = 0; i < ngrab; i++) {
            if (ev->code == grabkey[i].code) {
                /* keep track of how many grab keys are down */
                if (ev->value) {
                    if (grabkey[i].pressed++ == 0)
                        scrolling++;
                } else {
                    grabkey[i].pressed = 0;
                    if (--scrolling == 0)
                        cumul_x = cumul_y = 0;
                }

                pass = grabkey[i].pass;

                dbg(1, "scrolling %d", scrolling);
                break;
            }
        }

    }

    /* implement arrow key grab-scrolling */
    if (scrolling && ev->value) {
        switch(ev->code) {
        case KEY_UP:    send_a_scroll( 0,-1);  pass = 0; break;
        case KEY_DOWN:  send_a_scroll( 0, 1);  pass = 0; break;
        case KEY_LEFT:  send_a_scroll(-1, 0);  pass = 0; break;
        case KEY_RIGHT: send_a_scroll( 1, 0);  pass = 0; break;
        }
    }


    /* rotate the dpad keys on the screen bezel */
    i = 0;
    switch(ev->code) {
    case KEY_KP4: i++;
    case KEY_KP2: i++;
    case KEY_KP6: i++;
    case KEY_KP8:
        ev->code = dpad[rotation + i];
        break;

    case KEY_SWITCHVIDEOMODE:  // rotate button
        dbg(3, "might rotate");
        if (rotate_cmd && ev->value) {

            // command given, so don't pass the keystroke through.
            pass = 0;

            // if absolute, we can check for presence cheaply
            if (*rotate_cmd != '/' || access(rotate_cmd, X_OK) == 0) {
                dbg(1, "invoking %s", rotate_cmd);
                if (system(rotate_cmd) == 0)
                    ;
            }
        }
        break;

    case KEY_F9:        // brightness down key (or minimize, with alt)
    case KEY_F10:       // brightness up key (or maximize, with alt)
        dbg(3, "might brighten");
        if (brightness_cmd && ev->value) {
            char cmd[256];
            char *arg;

            // command given, so don't pass the keystroke through.
            pass = 0;

            // if absolute, we can check for presence cheaply
            if (*brightness_cmd != '/' ||
                        access(brightness_cmd, X_OK) == 0) {
                if (altdown) {
                    arg = (ev->code == KEY_F9) ? "min" : "max";
                } else {
                    arg = (ev->code == KEY_F9) ? "down" : "up";
                }
                if (snprintf(cmd, sizeof(cmd), "%s %s",
                                brightness_cmd, arg) <= sizeof(cmd)) {
                    dbg(1, "invoking %s", cmd);
                    if (system(cmd) == 0)
                        ;
                }
            }
        }
        break;

    case KEY_F11:       // volume down key (or minimize, with alt)
    case KEY_F12:       // volume up key (or maximize, with alt)
        dbg(3, "might adjust volume");
        if (volume_cmd && ev->value) {
            char cmd[256];
            char *arg;

            // command given, so don't pass the keystroke through.
            pass = 0;

            // if absolute, we can check for presence cheaply
            if (*volume_cmd != '/' ||
                        access(volume_cmd, X_OK) == 0) {
                if (altdown) {
                    arg = (ev->code == KEY_F11) ? "min" : "max";
                } else {
                    arg = (ev->code == KEY_F11) ? "down" : "up";
                }
                if (snprintf(cmd, sizeof(cmd), "%s %s",
                                volume_cmd, arg) <= sizeof(cmd)) {
                    dbg(1, "invoking %s", cmd);
                    if (system(cmd) == 0)
                        ;
                }
            }
        }
        break;

    case KEY_LEFTALT:
    case KEY_RIGHTALT:
        if (ev->value == 0)
            altdown--;
        else if (ev->value == 1)
            altdown++;
        else
            /* ignore repeats with ev->value == 2 */ ;
        if (altdown > 2) altdown = 2;       // safety
        else if (altdown < 0) altdown = 0;  // safety
        break;
    }

    if (pass && !noxmit && write(uinp_fd, ev, sizeof(ev)) < 0) {
        report("uinput write failed: %s\n", strerror(errno));
    }

    return !!(ev->value);  /* indicates press */
}

void touchpad_event(void)
{
    struct input_event ev[1];

    if (read(tpad_fd, ev, sizeof(ev)) != sizeof(ev))
        die("bad read from touchpad");

    dbg(3, "evtype %d code %d", ev->type, ev->code);

    if ((reflect_x && ev->code == REL_X) ||
        (reflect_y && ev->code == REL_Y)) {
        ev->value = -ev->value;
    }

    switch(rotation) {
    case ROT_NORMAL:
        break;
    case ROT_RIGHT:
        if (ev->code == REL_Y) {
            ev->code = REL_X;
        } else if (ev->code == REL_X) {
            ev->code = REL_Y;
            ev->value = -ev->value;
        }
        break;
    case ROT_LEFT:
        if (ev->code == REL_Y) {
            ev->code = REL_X;
            ev->value = -ev->value;
        } else if (ev->code == REL_X) {
            ev->code = REL_Y;
        }
        break;
    case ROT_INVERT:
        ev->value = -ev->value;
        break;
    }

    if (scrolling && ev->type == EV_REL && ev->code == REL_X) {
            cumul_x += ev->value;

    } else if (scrolling && ev->type == EV_REL && ev->code == REL_Y) {
            cumul_y += ev->value;

    } else if (scrolling && ev->type == EV_SYN) {
        /* emit our current scroll input */
        int x = 0, y = 0;

        if (abs(cumul_y) > quantum) {
            if (abs(cumul_x) < ratio * abs(cumul_y) / 100)
                cumul_x = 0;
            y = cumul_y;
            cumul_y = 0;
        }
        if (abs(cumul_x) > quantum) {
            if (abs(cumul_y) < ratio * abs(cumul_x) / 100)
                cumul_y = 0;
            x = cumul_x;
            cumul_x = 0;
        }
        send_a_scroll(x, y);

    } else {
        /* passthrough */
        if (!noxmit && write(uinp_fd, ev, sizeof(ev)) < 0) {
            report("uinput write failed: %s\n", strerror(errno));
        }
    }
}


void
send_event(char *e)
{
    int fd, n;
    char evtbuf[128];

    fd = open(sysactive_path, O_RDWR|O_NONBLOCK);
    if (fd >= 0) {
        n = snprintf(evtbuf, 128, "%s %d\n", e, (int)time(0));
        n = write(fd, evtbuf, n);
        close(fd);
    }
}

void
indicate_activity(int newactivity)
{
    static char useridle[] = "useridleX";

    if (newactivity) {

        if (wasidle)
            send_event("useractive");

        idleness = 0;
        wasidle = 0;

    } else { /* an idle timer expired */

        useridle[8] = idleness + '1';
        if (idleness < MAXTIMERS) idleness++;

        send_event(useridle);
        wasidle = 1;
    }
}

void
get_command(void)
{
    int n;
    unsigned int a, b, c;
    char event[128];
    n = read(fifo_fd, event, sizeof(event));

    if (n < 1) {
        if (n < 0)
            die("read from rotation fifo");
        close(fifo_fd);
        fifo_fd = -1;
    }

    switch (event[0]) {
        case 'x': reflect_x = 0; break;
        case 'y': reflect_y = 0; break;
        case 'z': reflect_x = 0;
                  reflect_y = 0; break;

        case 'X': reflect_x = 1; break;
        case 'Y': reflect_y = 1; break;
        case 'Z': reflect_x = 1;
                  reflect_y = 1; break;

        case 'n': rotation = ROT_NORMAL; break;
        case 'r': rotation = ROT_RIGHT; break;
        case 'i': rotation = ROT_INVERT; break;
        case 'l': rotation = ROT_LEFT; break;

        case 'I':
                n = sscanf(event, "I %u %u %u", &a, &b, &c);
                if (n < 1) {
                    a = b = c = 0;
                } else {
                    if (n < 2 || b < a + 1)
                        b = a + 1;
                    if (n < 3 || c < b + 1)
                        c = b + 1;

                }

		{ 
		    static unsigned int old_a, old_b, old_c;
		    if (a != old_a || b != old_b || c != old_c) {
                	report("idle timers changed to %d %d %d", a, b, c);
		    } else {
                	dbg(1,"idle timers set to %d %d %d", a, b, c);
		    }
		    old_a = a; old_b = b; old_c = c;
		}


                idledeltas[3] = 0; /* always, so our last sleep is untimed */
                idledeltas[2] = c - b;
                idledeltas[1] = b - a;
                idledeltas[0] = a;

                idleness = 0;
                wasidle = 1;

                break;

        default:
                rotation = ROT_NORMAL;
                reflect_x = 0; reflect_y = 0;
                break;
    }

    dbg(1, "r/x/y is %d/%d/%d", rotation, reflect_x, reflect_y);

}

void
data_loop(void)
{
    fd_set inputs, errors;
    int maxfd, r;
    struct timeval tv;
    struct timeval *tvp;
    int pressed;

    indicate_activity(1);

    maxfd = MAX(fifo_fd, MAX(kbd_fd, tpad_fd));

    while (1) {

        FD_ZERO(&inputs);
        FD_SET(kbd_fd, &inputs);
        FD_SET(tpad_fd, &inputs);
        if (fifo_fd >= 0) FD_SET(fifo_fd, &inputs);

        FD_ZERO(&errors);
        FD_SET(kbd_fd, &errors);
        FD_SET(tpad_fd, &errors);
        if (fifo_fd >= 0) FD_SET(fifo_fd, &errors);

        if (idledeltas[idleness]) {
            tv.tv_sec = idledeltas[idleness];
            tv.tv_usec = 0;
            tvp = &tv;
        } else {
            tvp = 0;
        }

        r = select(maxfd+1, &inputs, NULL, &errors, tvp);
        if (r < 0)
            die("select failed");

        if (r == 0) {
            indicate_activity(0);
            continue;
        }

        pressed = 0;

        if (fifo_fd >= 0 && FD_ISSET(fifo_fd, &errors))
            die("select reports error on rotation event fifo");

        if (FD_ISSET(kbd_fd, &errors))
            die("select reports error on keyboard");

        if (FD_ISSET(tpad_fd, &errors))
            die("select reports error on touchpad");

        if (fifo_fd >= 0 && FD_ISSET(fifo_fd, &inputs))
            get_command();

        if (FD_ISSET(kbd_fd, &inputs))
            pressed |= keyboard_event();

        if (FD_ISSET(tpad_fd, &inputs)) {
            touchpad_event();
            pressed = 1;
        }

        if (pressed)
            indicate_activity(1);

    }

}

int
init_fifo(char *fifo_node)
{
    int fd;
    struct stat sbuf;

#define fifomode 0622  /* allow anyone to adjust the rotation */

    if (mkfifo(fifo_node, fifomode)) {
        if (errno != EEXIST) {
            report("mkfifo of %s failed", fifo_node);
            return -1;
        }

        /* the path exists.  is it a fifo? */
        if (stat(fifo_node, &sbuf) < 0) {
            report("stat of %s failed", fifo_node);
            return -1;
        }

        /* if not, remove and recreate */
        if (!S_ISFIFO(sbuf.st_mode)) {
            unlink(fifo_node);
            if (mkfifo(fifo_node, fifomode)) {
                report("recreate of %s failed", fifo_node);
                return -1;
            }
        }
    }

    /* mkfifo was affected by umask */
    if (chmod(fifo_node, fifomode)) {
        report("chmod of %s to 0%o failed", fifo_node, fifomode);
        return -1;
    }

    /* open for read/write, since writers are itinerant */
    fd = open(fifo_node, O_RDWR);
    if (fd < 0) {
        report("open %s failed", fifo_node);
        return -1;
    }

    return fd;
}

void
deinit_fifo(void)
{
    close(fifo_fd);
    fifo_fd = -1;
    unlink(command_fifo);
}


void
sighandler(int sig)
{
    deinit_uinput_device();
    deinit_fifo();
    die("got signal %d", sig);
}

int
main(int argc, char *argv[])
{
    int foreground = 0;
    int sched_realtime = 0;
    unsigned short b, v, p;
    char *cp, *eargp;
    int c;

    me = argv[0];
    cp = strrchr(argv[0], '/');
    if (cp) me = cp + 1;

    while ((c = getopt(argc, argv, "flsdXvq:n:K:T:g:G:R:A:r:b:V:")) != -1) {
        switch (c) {

        /* daemon options */
        case 'f':
            foreground = 1;
            break;
        case 'l':
            logtosyslog = 1;
            break;
        case 's':
            sched_realtime = 1;
            break;
        case 'd':
            debug++;
            break;
        case 'X':
            noxmit = 1;
            break;

        /* algorithmic tuning */
        case 'v':
            reverse = 1;
            break;
        case 'q':
            quantum = strtol(optarg, &eargp, 10);
            if (*eargp != '\0' || quantum <= 0)
                usage();
            break;
        case 'n':
            ratio = strtol(optarg, &eargp, 10);
            if (*eargp != '\0' || ratio <= 0)
                usage();
            break;
        case 'K':
            if (sscanf(optarg, "%hx:%hx:%hx", &b, &v, &p) != 3)
                usage();
            kbd_id.bustype = b;
            kbd_id.vendor = v;
            kbd_id.product = p;
            break;
        case 'T':
            if (sscanf(optarg, "%hx:%hx:%hx", &b, &v, &p) != 3)
                usage();
            tpad_id.bustype = b;
            tpad_id.vendor = v;
            tpad_id.product = p;
            break;
        case 'g':
        case 'G':
            if (ngrab >= NGRABKEYS) {
                fprintf(stderr, "%s: too many grab keys specified, %d max\n",
                    me, NGRABKEYS);
                exit(1);
            }
            grabkey[ngrab].code = strtol(optarg, &eargp, 10);
            if (*eargp != '\0')
                usage();
            grabkey[ngrab].pass = (c == 'g');
            ngrab++;
            break;

        case 'R':
            command_fifo = optarg;
            break;

        case 'A':
            sysactive_path = optarg;
            break;

        case 'r':
            rotate_cmd = optarg;
            break;

        case 'b':
            brightness_cmd = optarg;
            break;

        case 'V':
            volume_cmd = optarg;
            break;

        default:
            usage();
            break;
        }
    }

    if (optind < argc) {
        report("found non-option argument(s)");
        usage();
    }

    /* default to XO grab keys, with passthrough.  since they're
     * just modifiers, nothing will happen with them unless X
     * wants it to.
     */
    if (ngrab == 0) {
        grabkey[0].code = KEY_LEFTMETA;
        grabkey[0].pass = 1;
        grabkey[1].code = KEY_RIGHTMETA;
        grabkey[1].pass = 1;
        ngrab = 2;
    }

    report("starting %s version %d", me, VERSION);

    if (setup_input() < 0)
        die("%s: unable to find input devices\n", me);

    /* initialize uinput, if needed */
    if (!noxmit && setup_uinput() < 0)
        die("%s: unable to find uinput device\n", me);

    if (command_fifo) {
        fifo_fd = init_fifo(command_fifo);
        if (fifo_fd < 0)
            report("no fifo for rotation control");
    }

    signal(SIGTERM, sighandler);
    signal(SIGHUP, sighandler);
    signal(SIGINT, sighandler);
    signal(SIGQUIT, sighandler);
    signal(SIGABRT, sighandler);
    signal(SIGUSR1, sighandler);
    signal(SIGUSR2, sighandler);

    if (sched_realtime) {
        struct sched_param sparam;
        int min, max;

        /* first, lock down all our memory */
        long takespace[1024];
        memset(takespace, 0, sizeof(takespace)); /* force paging */
        if (mlockall(MCL_CURRENT|MCL_FUTURE) < 0)
            die("unable to mlockall");

        /* then, raise our scheduling priority */
        min = sched_get_priority_min(SCHED_FIFO);
        max = sched_get_priority_max(SCHED_FIFO);

        sparam.sched_priority = (min + max)/2; /* probably always 50 */
        if (sched_setscheduler(0, SCHED_FIFO, &sparam))
            die("unable to set SCHED_FIFO");

        report("memory locked, scheduler priority set");

    }

    if (!foreground) {
        if (daemon(0, 0) < 0)
            die("failed to daemonize");
        daemonized = 1;
    }

    data_loop();

    return 0;
}

