#include "seqacqread.h"
#include "seqlist.h"


void SeqAcqRead::common_init() {
  SeqAcqInterface::set_marshall(&acq);
  SeqFreqChanInterface::set_marshall(&acq);
}

SeqAcqRead::SeqAcqRead(const STD_string& object_label,double sweepwidth,unsigned int nAcqPoints,
            float fov, direction gradchannel,float os_factor,const STD_string& nucleus,
            const dvector& phaselist,const dvector& freqlist,
            float timestep, rampType rampmode )
  : SeqParallel(object_label),
  acq(object_label+"_acq",nAcqPoints,sweepwidth,os_factor,nucleus,phaselist,freqlist),
  read(object_label+"_read",gradchannel, secureDivision(2.0*PII*acq.get_sweepwidth(),systemInfo->get_gamma(nucleus)*fov), secureDivision(double(nAcqPoints),acq.get_sweepwidth()),timestep,rampmode), // Use adjusted sweepwidth
  middelay(object_label+"_middelay",0.0),
  midgrad(object_label+"_midgrad",gradchannel,0.0),
  tozero(object_label+"_tozero",read.get_offramp_duration()+systemInfo->get_inter_grad_delay()) {
  common_init();
  build_seq();
}




SeqAcqRead::SeqAcqRead(const STD_string& object_label)
   : SeqParallel(object_label) {
  common_init();
}


SeqAcqRead::SeqAcqRead(const SeqAcqRead& sar) {
  common_init();
  SeqAcqRead::operator = (sar);
}


double SeqAcqRead::get_acquisition_center() const {
  double result=0.0;
  result+=SeqParallel::get_pulprogduration();
  result+=middelay.get_duration();
  result+=acq.get_acquisition_center();
  return result;
}


double SeqAcqRead::get_acquisition_start() const {
  double result=0.0;
  result+=SeqParallel::get_pulprogduration();
  result+=middelay.get_duration();
  result+=acq.get_acquisition_start();
  return result;
}


SeqAcqInterface& SeqAcqRead::set_sweepwidth(double sw, float os_factor) {
  Log<Seq> odinlog(this,"set_sweepwidth");
  ODINLOG(odinlog,warningLog) << "Ignoring request to change sweepwidth after construction" << STD_endl;
  return *this;
}


SeqAcqRead& SeqAcqRead::operator = (const SeqAcqRead& sar) {
  SeqParallel::operator = (sar);
  acq=sar.acq;
  read=sar.read;
  middelay=sar.middelay;
  midgrad=sar.midgrad;
  tozero=sar.tozero;
  readdephgrad=sar.readdephgrad;
  build_seq();
  return *this;
}


void SeqAcqRead::build_seq() {
  Log<Seq> odinlog(this,"build_seq");
  bool seqset=false;

  double ppgdur=SeqParallel::get_pulprogduration();
  double gradcent=read.get_gradpulse_center();
  double acqcent=ppgdur+acq.get_acquisition_center();

  double shift=gradcent-acqcent;

  ODINLOG(odinlog,normalDebug) << "ppgdur/gradcent/acqcent/shift=" << ppgdur << "/" << gradcent << "/" << acqcent << "/" << shift << STD_endl;

  if(shift>=systemInfo->get_min_duration(delayObj)) {
    ODINLOG(odinlog,normalDebug) << "acq is played out later, because gradient switching is delayed" << STD_endl;
    middelay.set_duration(shift);
    SeqParallel::operator = ( read / ( middelay + acq + tozero ) );
    seqset=true;
  }

  if((-shift)>=systemInfo->get_min_duration(gradObj)) {
    ODINLOG(odinlog,normalDebug) << "gradients are played out later, because acq is delayed" << STD_endl;
    midgrad.set_duration(-shift);
    SeqParallel::operator = ( (midgrad+read) / ( acq + tozero ) );
    seqset=true;
  }

  if(!seqset) {
    ODINLOG(odinlog,normalDebug) << "acq and gradients are played out simultaneously" << STD_endl;
    SeqParallel::operator = ( read / ( acq + tozero ) );
  }


  // Integral is the same for dephasing and rephasing
  float dephintegral=-(read.get_onramp_integral()+0.5*read.get_constgrad_integral());

  // Initialze once so that consecutive calls to get_dephgrad return always the same objects
  readdephgrad=SeqGradTrapez(STD_string(get_label())+"_readdephgrad",
                             dephintegral,
                             read.get_channel(),0.5 *read.get_constgrad_duration(),
                             read.get_timestep(),read.get_ramptype());

}


const SeqVector* SeqAcqRead::get_dephgrad(SeqGradChanParallel& dephobj, bool rephase) const {
  SeqGradTrapez* rdg=new SeqGradTrapez(readdephgrad); // Create deep copy so user can modify SeqAcqDeph safely
  rdg->set_temporary();
  dephobj+=(*rdg);
  return 0;
}


