#include "fileio.h"

#ifdef VISTASUPPORT

#ifdef HAVE_VIAIO_VLIB_H
  #include <viaio/Vlib.h>
#elif defined HAVE_VISTA_VLIB_H
  #include <vista/Vlib.h>
#else
  #error "Vista include not found. You will need vista/Vlib.h or viaio/Vlib.h in your default include path"
#endif

#ifdef HAVE_VIAIO_VIMAGE_H
  #include <viaio/VImage.h>
#elif defined HAVE_VISTA_VIMAGE_H
  #include <vista/VImage.h>
#else
  #error "Vista include not found. You will need vista/VImage.h or viaio/VImage.h in your default include path"
#endif

#include <stdexcept>

enum vista_dialect{unknown,common,odin,lipsia,timecourse,dwi};

/////////////////////////////////////////////////////////////

class LipsiaImages: public STD_list<STD_pair<Protocol,VImage> >{
public:
  unsigned short is_4d(iterator it,unsigned short &postfix, iterator &postfix_it){
    unsigned len=0;
    postfix=0;
    const Protocol &refProtocol=it->first;
    for(postfix_it=it;postfix_it!=end();postfix_it++){
      if(postfix_it->second && postfix_it->first == refProtocol)len++;
      else break;
    }
    if(len>1){//only 4Dim Datasets have a postfix
        for(;postfix_it!=end();postfix_it++){
          if(postfix_it->second==NULL)postfix++;
          else break;
      }
    }
    return len;
  }
  unsigned short is_prefix(iterator it,iterator &next_it)	{
    unsigned short ret=0;next_it=end();
    for(iterator i=it;i!=end();i++){
      if(i->second==NULL)ret++;
      else {next_it=i;break;}
    }
    return ret;
  }
  iterator addImage(Protocol prot,VImage img){
    STD_pair<Protocol,VImage> pip(prot,img);
    return insert(end(), pip);
  }
  iterator removeImage(iterator i){
      if(i->second)VDestroyImage(i->second);
    return erase(i);
  }
  iterator removeImage(iterator begin,iterator end){
    for(iterator i=begin;i!=end;i++)
      if(i->second)VDestroyImage(i->second);
    return erase(begin,end);
  }
};

struct VistaFormat : public FileFormat {

  Mutex mutex; // Vista lib is not thread safe

  STD_string description() const {return "Vista";}

  svector suffix() const  {
    svector result; result.resize(1);
    result[0]="v";
    return result;
  }
  svector dialects() const {
    svector result; result.resize(3);
    result[0]="common";
    result[1]="odin";
    result[2]="lipsia";
    return result;
  }
  static void swap34(FileIO::ProtocolDataMap::iterator pdit,FileIO::ProtocolDataMap &pdmap){
    STD_pair<Protocol, Data<float,4> > pdpair=*pdit;
    pdmap.erase(pdit);
    Protocol &prot=pdpair.first;
    Data<float,4> &dat=pdpair.second;

    dat.transposeSelf(secondDim,firstDim,thirdDim,fourthDim);
    const unsigned int slices=prot.geometry.get_nSlices();
    prot.geometry.set_nSlices(prot.seqpars.get_NumOfRepetitions());
    prot.seqpars.set_NumOfRepetitions(slices);

    pdmap.insert(pdpair);
  }

  template<typename T,VRepnKind K,typename R> static bool readAttr(const VAttrList &attr, const char *name,R &ret){
    T dummy;
    if(VGetAttr(attr, name, NULL, K, &dummy)==VAttrFound){ret=dummy;return true;}
    else return false;
  }
  template<VRepnKind K,typename R> static void storeAttr(const R &value,const VAttrList &attr, const char *name){
    VAppendAttr(attr, name, NULL, K,value);
  }

  static bool readScal(const VAttrList &attr, const char *name,STD_string &ret){return readAttr<char*,VStringRepn>(attr,name,ret);}
  static bool readScal(const VAttrList &attr, const char *name,VLong      &ret){return readAttr<VLong,VLongRepn>(attr,name,ret);}
  static bool readScal(const VAttrList &attr, const char *name,float      &ret){return readAttr<float,VFloatRepn>(attr,name,ret);}

  static void storeScal(const STD_string &value, const VAttrList &attr, const char *name){storeAttr<VStringRepn>(value.c_str(),attr,name);}
  static void storeScal(const VLong value,       const VAttrList &attr, const char *name){storeAttr<VLongRepn>(value,attr,name);}
  static void storeScal(const float value,      const VAttrList &attr, const char *name){storeAttr<VFloatRepn>(value,attr,name);}

  template<typename T> static bool attr2vector(const VAttrList &attr, const char *name,tjvector<T> &ret){
    Log<FileIO> odinlog("VistaFormat","attr2vector");
    STD_string attrib;
    if(readScal(attr,name,attrib)){
      svector toks(tokens(attrib));
      const int size=toks.size();
      ret.resize(size);
      for(int i=0;i<size;i++)
        TypeTraits::string2type(toks[i],ret[i]);
      ODINLOG(odinlog,normalDebug) << "parsed " << name << "=" << attrib << " as vector" << ret << STD_endl;
      return true;
    }else return false;
  }
  template<typename T> static bool vector2attr(const tjvector<T> &vec,VAttrList &attr, const char *name){
    Log<FileIO> odinlog("VistaFormat","vector2attr");
    if(!vec.length())return false;
    STD_ostringstream attrib;attrib << STD_setprecision(8) << (fabs(vec[0])<ODIN_GEO_CHECK_LIMIT ?0:vec[0]);
    for(size_t i=1;i<vec.length();i++)
      attrib << ' ' << (fabs(vec[i])<ODIN_GEO_CHECK_LIMIT ?0:vec[i]);
    ODINLOG(odinlog,normalDebug) << "Setting " << name << "=\"" << attrib.str() << "\" (from " << vec << ")" << STD_endl;
    storeScal(attrib.str(), attr,name);
    return attrib.str().length()!=0;
  }

  static vista_dialect attr2protocol(const VAttrList &attr, Protocol &prot,const unsigned short xsize,const unsigned short ysize,const unsigned short zsize)  {
    Log<FileIO> odinlog("VistaFormat","attr2protocol");
    int n=prot.numof_pars();
    STD_string value;
    vista_dialect ret=common;
    prot.geometry.set_Mode(slicepack);

    for(int i=0; i<n; i++){
      // Parse Odin protocol
      JcampDxClass& ldr=prot[i];
      if(ldr.get_filemode()!=exclude)
        if(readScal(attr, ldr.get_label().c_str(), value)){
          ODINLOG(odinlog,normalDebug) << "value(" << ldr.get_label() << ")=" << value << STD_endl;
          ldr.parsevalstring(value);
          ret=odin; //these params are only written in odin - so its odin
        }
    }
    if(ret==odin)return ret;//OK its odin, we're done here

    //if it wasn't odin ...

    //check for some obligatory attributes which only (??) appears in lipsia files
    if(readScal(attr, "convention", value))
      ret=lipsia;

    if(readScal(attr, "modality", value)){
      if(value=="fMRI")ret=timecourse;
      else if(value=="MRI")ret=lipsia;
    }

    //from here on we only select a dialect, if its unknown before

    //get date/time
    STD_string date,time;
    prot.study.get_DateTime(date,time);//get default
    if(readScal(attr, "date",date))
      date=date.substr(6,4)+date.substr(3,2)+date.substr(0,2);
    if(readScal(attr, "time",time))
      time=time.substr(0,2)+time.substr(3,2)+time.substr(6,2);
    prot.study.set_DateTime(date,time);//set it again

    STD_string coil;
    float echoTime,flipAngle;
    if(readScal(attr, "transmit_coil",coil))prot.system.set_transmit_coil_name(coil);
    if(readScal(attr, "echoTime",echoTime))prot.seqpars.set_EchoTime(echoTime);
    if(readScal(attr, "flipAngle",flipAngle))prot.seqpars.set_EchoTime(flipAngle);


    //set study-data
    STD_string description,scientist,dummy;
    int number;
    prot.study.get_Context(dummy,scientist); //get defaults
    prot.study.get_Series(description,number);
    readScal(attr,"description",description);
    readScal(attr,"scientist",scientist);
    prot.study.set_Context(dummy,scientist);
    prot.study.set_Series(description,number);

    //set Patient Data
    STD_string id,full_name,birth_date;
    char sex;STD_string ssex,sage;
    float weight;
    prot.study.get_Patient(id,full_name,birth_date,sex,weight);//get defaults
    readScal(attr,"patient",id);
    if(readScal(attr, "age",sage) && readScal(attr, "date",date))
       birth_date=TypeTraits::type2string(atoi(date.substr(6,4).c_str())-atoi(sage.c_str()))+"0101";//Just set birthdate to the 1st Jan of scandate-age
    if(readScal(attr,"sex", ssex)){
      if(ssex=="female")sex='f';
      else if(ssex=="male")sex='m';
      else if(ssex=="other")sex='o';
    }
    prot.study.set_Patient(id,full_name,birth_date,sex,weight);

    //get T_r
    float repetition_time;
    if(readScal(attr, "repetition_time",repetition_time))
      prot.seqpars.set_RepetitionTime(repetition_time);
    else if(readScal(attr, "MPIL_vista_0", value)){
      int ntimesteps =0;
      ret=ret?:lipsia;
      if(sscanf(value.c_str()," repetition_time=%f packed_data=1 %d ",&repetition_time,&ntimesteps)>0)//@todo possible overflow
        prot.seqpars.set_RepetitionTime(repetition_time);
    }

    //get resolution
    dvector res(3);
    if(attr2vector(attr,"voxel",res)){
      dvector latice;
      ret=ret?:lipsia;
      ODINLOG(odinlog,normalDebug) << "Found voxel size. Setting FOV " << res[0]*xsize << "x" << res[1]*ysize << STD_endl;
      prot.geometry.set_FOV(readDirection,res[0]*xsize);
      prot.geometry.set_FOV(phaseDirection,res[1]*ysize);
      prot.geometry.set_sliceDistance(res[2]);
      if(attr2vector(attr,"latice",latice)){
        ODINLOG(odinlog,normalDebug) << "Found latice size. Setting slicegap " << res[2]-latice[2] << STD_endl;
        prot.geometry.set_sliceThickness(latice[2]);
      }
    } else ODINLOG(odinlog,warningLog) << "Cannot find voxel size. Using default" << STD_endl;

    //get orientation and offset
    dvector readV(3),phaseV(3),sliceV(3),origin(3);
    readV[0]=phaseV[1]=sliceV[2]=1;

    //if we have the real orientation info ..
    if(attr2vector(attr,"columnVec",readV) && attr2vector(attr,"rowVec",phaseV) && attr2vector(attr,"sliceVec",sliceV)){
      ret=ret?:lipsia;
      attr2vector(attr,"indexOrigin",origin);
      // in lipsia (or dicom) x- and y-axis are inverted to the coord system of ODIN so we need a tranform
      dvector transform(3);transform[0]=-1.0; transform[1]=-1.0; transform[2]=1.0;
      readV*=transform;phaseV*=transform;sliceV*=transform;origin*=transform;

      const dvector ivector(readV*res[0]* (xsize-1)+phaseV*res[1]*(ysize-1)+sliceV*res[2]*(zsize-1));
      dvector center=origin+ ivector*.5;//center of the data is the Middle of the 0th Voxel (origin) plus the half of the diagonale through the data
      prot.geometry.set_orientation_and_offset(readV,phaseV,sliceV,center);

    } else if(readScal(attr, "orientation", value)) {//otherwise try the lipsia flag "orientation"
      ret=ret?:lipsia;
      ODINLOG(odinlog,warningLog) << "Not orientation matrix found using orientation " << value << STD_endl;
      if(STD_string("axial")==value)prot.geometry.set_orientation(axial);
      else if(STD_string("sagittal")==value)prot.geometry.set_orientation(sagittal);
      else if(STD_string("coronal")==value)prot.geometry.set_orientation(coronal);

    } else //or at least tell the user
      ODINLOG(odinlog,warningLog) << "No orientation information found using default" << value << STD_endl;
	
	//check for diffusion-data
	float bValue;
	if(readScal(attr,"diffusionBValue",bValue)){
		dvector bVector;
		if(!attr2vector(attr,"diffusionGradientOrientation",bVector))
			ODINLOG(odinlog,warningLog) <<  "no diffusion direction found for bValue"<<  bValue << STD_endl;
		bVector*=bValue;
		ODINLOG(odinlog,normalDebug) <<  "Found diffusion bVector: [" << bVector.printbody() << "]"<< STD_endl;
		prot.methpars.append_copy(JDXtriple(bVector[0],bVector[1],bVector[2],"Diffusion_bVector"));
	}

    return ret;
  }

  static int copyImage(const VImage &src,float *dest,const unsigned short slice)	{
    Log<FileIO> odinlog("VistaFormat","copyImage");
    const unsigned int bands=VImageNBands(src),rows=VImageNRows(src),columns=VImageNColumns(src);
    const unsigned int image_size=bands*rows*columns;
    dest+=image_size*slice;

    switch(VPixelRepn(src))
    {
      case VFloatRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VFloat),dest,image_size,image_size);break;
      case VDoubleRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VDouble),dest,image_size,image_size);break;
      case VBitRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VBit),dest,image_size,image_size);break;
      case VUByteRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VUByte),dest,image_size,image_size);break;
      case VSByteRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VSByte),dest,image_size,image_size);break;
      case VShortRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VShort),dest,image_size,image_size);break;
      case VLongRepn:
        Converter::convert_array(&VPixel(src, 0, 0, 0, VLong),dest,image_size,image_size);break;
      default:
        ODINLOG(odinlog,errorLog) << "pixel representation unknown - nothing written" << STD_endl;
        return -1;
    }
    return image_size;
  }

  void prepareImage(LipsiaImages& lImages,const VImage &src,const Protocol& protocol_template,vista_dialect &dialect) {
    Log<FileIO> odinlog("VistaFormat","prepareImage");
    const int nx=VImageNColumns(src);
    const int ny=VImageNRows(src);
    const int nz=VImageNBands(src);

    // Read atrribute of the image into protocol
    Protocol prot(protocol_template);
    if(dialect==unknown)dialect=attr2protocol(VImageAttrList(src),prot,nx,ny,nz);
    else attr2protocol(VImageAttrList(src),prot,nx,ny,nz);

    //if it could be (or is) timecourse dialect
    if(dialect == unknown || dialect == timecourse){
      if(nx == 1 && ny == 1 && nz == 1){//if its a "zero-image"
      //add an "_Zero_-Image" and leave
        lImages.addImage(protocol_template,NULL);
        dialect= timecourse; //only lipsia timecourse has "zero-images" - so we can say ist lipsia
        return;
      }
    }

    // overwrite extent in protocol so that datasets with different size have differing protocol, even if none was provided
    // keeps protocol of hyperplanes equal
    prot.seqpars.set_MatrixSize(phaseDirection,ny);
    prot.seqpars.set_MatrixSize(readDirection,nx);
    prot.geometry.set_nSlices(nz);

    switch(VPixelRepn(src))
    {
      case VFloatRepn:prot.system.set_data_type(TypeTraits::type2label((float)0));break;
      case VDoubleRepn:prot.system.set_data_type(TypeTraits::type2label((double)0));break;
      case VBitRepn:prot.system.set_data_type(TypeTraits::type2label((u8bit)0));break; //@todo really use "bit"
      case VUByteRepn:prot.system.set_data_type(TypeTraits::type2label((u8bit)0));break;
      case VSByteRepn:prot.system.set_data_type(TypeTraits::type2label((s8bit)0));break;
      case VShortRepn:prot.system.set_data_type(TypeTraits::type2label((s16bit)0));break;
      case VLongRepn:prot.system.set_data_type(TypeTraits::type2label((s32bit)0));break;
      default:
        ODINLOG(odinlog,errorLog) << "pixel representation unknown - nothing read" << STD_endl;break;
    }
    lImages.addImage(prot,src);//add real image
    return;
  }

  int read(FileIO::ProtocolDataMap& pdmap, const STD_string& filename, const FileReadOpts& opts, const Protocol& protocol_template)	{
    Log<FileIO> odinlog("VistaFormat","read");

    MutexLock lock(mutex);

    LipsiaImages lImages;

    VAttrListPosn posn;
    FILE *fp=fopen(filename.c_str(),"r");
    VAttrList attrlist=fp ? VReadFile(fp, NULL):0;
    vista_dialect dialect=unknown;
    const STD_string sdialect=tolowerstr(opts.dialect);
    if(sdialect=="odin")dialect=odin;
    else if(sdialect=="lipsia")dialect=lipsia;
    else if(sdialect=="common")dialect=common;

    if(attrlist==NULL) {
      ODINLOG(odinlog,errorLog) << "Cannot open file >" << filename << "<("<< strerror(errno) << ")" <<  STD_endl;
      return -1;
    }

    VImage imageBuf;
    for(VFirstAttr(attrlist,&posn);VAttrExists(&posn);VNextAttr(&posn))
      if(VGetAttrRepn(&posn) == VImageRepn)			{
          VGetAttrValue(&posn,NULL,VImageRepn,&imageBuf);
      prepareImage(lImages,imageBuf,protocol_template,dialect);
      }
    fclose(fp);

    switch(dialect){
      case unknown:
      case common:ODINLOG(odinlog,normalDebug) << "common dialect selected" << STD_endl;break;
      case odin:ODINLOG(odinlog,normalDebug) << "odin dialect selected" << STD_endl;break;
      case lipsia:ODINLOG(odinlog,normalDebug) << "lipsia dialect selected" << STD_endl;break;
      case timecourse:ODINLOG(odinlog,normalDebug) << "lipsia timecourse dialect selected" << STD_endl;break;
    }

    while(!lImages.empty())
    {
      LipsiaImages::iterator plit=lImages.begin(),prefix_begin=plit,postfix_end;
      unsigned short postfix;
      unsigned short prefix=lImages.is_prefix(prefix_begin,plit);
      assert(plit!=lImages.end());
      unsigned short len=lImages.is_4d(plit,postfix,postfix_end);
      Protocol prot=plit->first;
      VImage image=plit->second;

      const int nx=VImageNColumns(image);
      const int ny=VImageNRows(image);
      const int nz=VImageNBands(image);

      prot.seqpars.set_NumOfRepetitions(prefix+postfix+len);

      FileIO::ProtocolDataMap::iterator destIt=pdmap.find(prot);
      if(destIt==pdmap.end())
      {
        STD_pair<Protocol,Data<float,4> > p(prot,Data<float,4>(prefix+postfix+len,nz,ny,nx));
        destIt=pdmap.insert(pdmap.end(),p);
      }	else	{
        ODINLOG(odinlog,warningLog) << "multiple datasets with same protocol found - trying to append" << STD_endl;
        prefix+=destIt->second.shape()(0);
        destIt->second.resizeAndPreserve(prefix+postfix+len,nz,ny,nx);
      }
      float *dest=destIt->second.c_array();
      for(unsigned short n=0;n<len;n++){
        assert(plit!=lImages.end());
        image=plit->second;
        assert(nx==VImageNColumns(image) && ny==VImageNRows(image) && nz==VImageNBands(image));
        copyImage(image,dest,n+prefix);
        if(prefix_begin==plit)
          prefix_begin=plit=lImages.removeImage(plit);
        else
          plit=lImages.removeImage(plit);
      }
      if(dialect==lipsia && len>1){
        dialect=timecourse;
        ODINLOG(odinlog,normalDebug) << "lipsia dialect and 4D image detected - swapping 3rd and 4th dimension" << STD_endl;
      }
      switch(dialect){
        case odin:
          //odin ALLWAYS stores 4th dim in frames and 3rd Dim in images
          ODINLOG(odinlog,normalDebug) << "odin dialect detected - swapping 3rd and 4th dimension" << STD_endl;
        case unknown:
        case common://@todo silently swapping by default
          swap34(destIt,pdmap);//@todo nasty
          break;
        case lipsia:
            ODINLOG(odinlog,normalDebug) << "lipsia dialect and 3D image detected - not swapping 3rd and 4th dimension" << STD_endl;
          break;
        case timecourse:
          lImages.removeImage(prefix_begin,postfix_end);//remove zero-images
          //lipsia stores time in frames and 3rd dim in images dim for animated dataset, so swap 3rd end 4th dim
          swap34(destIt,pdmap);
          ODINLOG(odinlog,normalDebug) << "lipsia dialect timecourse detected - swapping 3rd and 4th dimension" << STD_endl;
          break;
      }
    }
    return pdmap.size();
  }

  template<typename T> Data<T,4> prepareImage(const Data<float,4> &src){
    Data<T,4> ret;
    src.convert_to(ret,noupscale);
    return ret;
  }
  template<class T> static VImage copyImage(const Data<T,4> &src,const int fourth_index,const VRepnKind &repnID)	{
    Log<FileIO> odinlog("VistaFormat","copyImage");
    const unsigned int columns = src.extent(3);
    const unsigned int rows = src.extent(2);

    ODINLOG(odinlog,normalDebug) << "columns/rows: " << columns << "/" << rows << STD_endl;

    Data<T,3> dat;
    const Range all=Range::all();
    unsigned int bands;
    if(fourth_index == -1){
      ODINLOG(odinlog,normalDebug) << "its 3D Data - just copy it" << STD_endl;
      bands = src.extent(1);
      assert(src.extent(0)==1);
      dat.reference(src(0,all,all,all));
    }else{
      ODINLOG(odinlog,normalDebug) << "its 2D Animated - peel it out of the 4D-Dataset (Slice " << fourth_index << ")" << STD_endl;
      bands = src.extent(0);
      dat.reference(src(all,fourth_index,all,all));
    }
    unsigned int size = bands*rows*columns;
    ODINLOG(odinlog,normalDebug) << "coping Data" << dat.shape() << " to VImage ["<< bands << "\t" << rows << "\t" << columns << "]" << STD_endl;
    VImage ret=VCreateImage(bands,rows,columns,repnID);//@todo shouldnt this be deleted somewhere ??
    memcpy(&VPixel(ret, 0, 0, 0, T), dat.c_array(), size*sizeof(T));
    return ret;
  }
  template<class T> unsigned short copyImages(FileIO::ProtocolDataMap::const_iterator pdit,VAttrList &attrlist,const VRepnKind &repnID,vista_dialect dialect)  {
    Data<T,4> src_=prepareImage<T>(pdit->second);
    unsigned short n;
    for(n=0;n<pdit->second.extent(1);n++){//3rd dim is amount of animated 2D slices
      VImage i=copyImage(src_,n,repnID);
      pdit2attr(pdit,VImageAttrList(i),dialect);
      if(dialect==lipsia || dialect==timecourse)
        storeScal("fMRI",VImageAttrList(i),"modality");//If it lipsia, tell People its timecourse
      VAppendAttr(attrlist, "image", NULL, VImageRepn, i);
    }
    return n;
  }

  static void pdit2attr(const Protocol &prot,const Data<float,4> data,VAttrList imageAttr,vista_dialect dialect)	{
    Log<FileIO> odinlog("VistaFormat","pdit2attr");
    const Geometry &geometry=prot.geometry;

    JDXtriple *diff=dynamic_cast<JDXtriple *>(const_cast<Protocol&>(prot).methpars.get_parameter("Diffusion_bVector"));
    if(dialect==lipsia){

      //scandate/time
      STD_string scandate,scantime;
      prot.study.get_DateTime(scandate,scantime);
      storeScal(scandate.substr(6,2)+"."+scandate.substr(4,2)+"."+scandate.substr(0,4),imageAttr, "date");
      storeScal(scantime.substr(0,2)+":"+scantime.substr(2,2)+":"+scantime.substr(4,2),imageAttr, "time");

      //study-data
      STD_string description,scientist,dummy;
      int number;
      prot.study.get_Context(dummy,scientist);
      prot.study.get_Series(description,number);
      storeScal(description.substr(0,description.find('_')),imageAttr, "description");
      storeScal(scientist,imageAttr, "scientist");
      storeScal(prot.system.get_transmit_coil_name(),imageAttr, "transmit_coil");
      storeScal((float)prot.seqpars.get_FlipAngle(),imageAttr, "flipAngle");
      storeScal((float)prot.seqpars.get_EchoTime(),imageAttr, "echoTime");

      //Set Patient Data
      STD_string id,full_name,birth_date;
      char sex;
      float weight;
      prot.study.get_Patient(id,full_name,birth_date,sex,weight);
      storeScal(id,imageAttr, "patient");
      switch(sex){
        case 'F':
        case 'f':storeScal("female",imageAttr, "sex");break;
        case 'M':
        case 'm':storeScal("male",imageAttr, "sex");break;
        default:storeScal("other",imageAttr, "sex");
      }
      if(birth_date!="00000000"){
        int born=atoi(birth_date.substr(0,4).c_str()),now=atoi(scandate.substr(0,4).c_str());
        storeScal(now-born,imageAttr, "age");
      }

      //voxelsize
      STD_string voxel;
      dvector extend(3);
      extend[0]=voxel_extent(geometry,readDirection,data.extent(3));
      extend[1]=voxel_extent(geometry,phaseDirection,data.extent(2));

      if(geometry.get_Mode()!=voxel_3d){
        extend[2]=geometry.get_sliceThickness();
        vector2attr(extend, imageAttr, "latice");
        extend[2]=geometry.get_sliceDistance();
      }else{
        extend[2]=geometry.get_FOV(sliceDirection)/data.extent(1);
        vector2attr(extend, imageAttr, "latice");
      }
      vector2attr(extend, imageAttr, "voxel");

      //PixelBandwidth
      storeScal((float)secureDivision(1000.0*prot.seqpars.get_AcqSweepWidth(),data.extent(readDim)),imageAttr, "pixelBandwidth");


      //convention -- @todo will this ever be radiological
      storeScal("natural",imageAttr, "convention");

      //orientation
      JDXstring orientation("axial","orientation");
      sliceOrientation sorient=geometry.get_orientation();
      if(sorient==coronal) orientation="coronal";
      else if(sorient==sagittal) orientation="sagittal";
      storeScal(orientation,imageAttr, "orientation");

      dvector center=geometry.get_center();
      const dvector readV=prot.geometry.get_readVector();
      const dvector phaseV=prot.geometry.get_phaseVector();
      const dvector sliceV=prot.geometry.get_sliceVector();
      const dvector ivector =data_diagonal(geometry,data.extent(3),data.extent(2));
      const dvector origin=center - ivector*.5;

      // in lipsia (or dicom) x- and y-axis are inverted to the coord system of ODIN so we need a tranform
      dvector transform(3);
      transform[0]=-1.0; transform[1]=-1.0; transform[2]=1.0;

      vector2attr(origin*transform,imageAttr,"indexOrigin");
      vector2attr(readV*transform, imageAttr,"columnVec");
      vector2attr(phaseV*transform,imageAttr, "rowVec");
      vector2attr(sliceV*transform,imageAttr, "sliceVec");

      if(diff){
        vector2attr(origin,imageAttr,"indexOrigin");
        storeScal(readV.printbody()+" "+phaseV.printbody(),imageAttr, "imageOrientationPatient");
      }
    }
    if(diff){
      storeScal("HFS",imageAttr, "patientPosition");
      JDXtriple *tdiff=dynamic_cast<JDXtriple *>(const_cast<Protocol&>(prot).methpars.get_parameter("Diffusion_bVector"));
      assert(tdiff);fvector diff(3);
      for(int i=0;i<3;i++)diff[i]=(*tdiff)[i];

      const float len=sqrt((diff*diff).sum());if(len)diff/=len;
      VAppendAttr(imageAttr, "diffusionBValue", NULL, VFloatRepn,len);
      vector2attr(diff,imageAttr, "diffusionGradientOrientation");
    } else if(data.extent(sliceDim)>1){
      storeScal(data.extent(0),imageAttr, "ntimesteps");
      storeScal((float)prot.seqpars.get_RepetitionTime(),imageAttr, "repetition_time");
    }


    if(dialect==odin){
      // Odin protocol
      for(unsigned int i=0; i<prot.numof_pars(); i++) {
        const JcampDxClass& ldr=prot[i];
        if(ldr.get_filemode()!=exclude) {
          STD_string valstr=ldr.printvalstring();
          ODINLOG(odinlog,verboseDebug) << "valstr(" << ldr.get_label() << ")=>" << valstr << "<" << STD_endl;
          if(valstr=="") valstr="NoValue"; // empty string not allowed
          storeScal(valstr,imageAttr, ldr.get_label().c_str());
        }
      }
    }
  }
  static void pdit2attr(FileIO::ProtocolDataMap::const_iterator pdit,VAttrList imageAttr,vista_dialect dialect) {
    pdit2attr(pdit->first,pdit->second,imageAttr,dialect);
  }

  VRepnKind selRepn(const Protocol &prot, const FileWriteOpts& opts){
    Log<FileIO> odinlog("VistaFormat","selRepn");
    STD_string datatype=selectDataType(prot,opts);
    VRepnKind type=VFloatRepn; // Use float as default if datatype is not available in Vista
    if      (datatype==TypeTraits::type2label((double)0)) type=VDoubleRepn;
    else if (datatype==TypeTraits::type2label((float)0)) type=VFloatRepn;
    else if (datatype==TypeTraits::type2label((u32bit)0)){
      ODINLOG(odinlog,warningLog) << "datatype >u32bit< is not supportet in vista. Using >s32bit< (VLong) instead" << STD_endl;
      type=VLongRepn;
    }
    else if (datatype==TypeTraits::type2label((s32bit)0))type=VLongRepn;
    else if (datatype==TypeTraits::type2label((u16bit)0)){
      ODINLOG(odinlog,warningLog) << "datatype >u16bit< is not supportet in vista. Using >s16bit< (VShort) instead" << STD_endl;
      type=VShortRepn;
    }
    else if (datatype==TypeTraits::type2label((s16bit)0))type=VShortRepn;
    else if (datatype==TypeTraits::type2label(( u8bit)0))type=VUByteRepn;
    else if (datatype==TypeTraits::type2label(( s8bit)0))type=VSByteRepn;
    else
      ODINLOG(odinlog,warningLog) << "datatype >" << datatype << "< unknown, using float" << STD_endl;
    return type;
  }

  int write(const FileIO::ProtocolDataMap& pdmap, const STD_string& filename, const FileWriteOpts& opts)	{
    Log<FileIO> odinlog("VistaFormat","write");

    MutexLock lock(mutex);

    VAttrList attrlist=VCreateAttrList();
#ifdef FILEIO_ONLY
    vista_dialect dialect=lipsia;
#else
    vista_dialect dialect=odin;
#endif
    const STD_string sdialect=tolowerstr(opts.dialect);
    if(sdialect=="odin")dialect=odin;
    else if(sdialect=="lipsia")dialect=lipsia;
    else if(sdialect=="timecourse")dialect=timecourse;
    else if(sdialect=="common")dialect=common;


    VImage i;
    int result=0;
//  put anatomic data at the beginning of the list
    if(dialect==lipsia) // lipsia stores anatomic data at the beginning of the file and with NOT swapped dimensions (look at read)
      for(FileIO::ProtocolDataMap::const_iterator pdit=pdmap.begin(); pdit!=pdmap.end(); ++pdit){
        const Data<float,4> &src=pdit->second;
        JDXtriple *diff=dynamic_cast<JDXtriple *>(const_cast<Protocol&>(pdit->first).methpars.get_parameter("Diffusion_bVector"));
        if(src.extent(0)==1 && !diff){ // Its anatomic Data
          switch(selRepn(pdit->first,opts))  {
            case VFloatRepn:i=copyImage(prepareImage<VFloat>(src),-1,VFloatRepn);break;
            case VDoubleRepn:i=copyImage(prepareImage<VDouble>(src),-1,VDoubleRepn);break;
            case VBitRepn:i=copyImage(prepareImage<VBit>(src),-1,VBitRepn);break;
            case VUByteRepn:i=copyImage(prepareImage<VUByte>(src),-1,VUByteRepn);break;
            case VSByteRepn:i=copyImage(prepareImage<VSByte>(src),-1,VSByteRepn);break;
            case VShortRepn:i=copyImage(prepareImage<VShort>(src),-1,VShortRepn);break;
            case VLongRepn:i=copyImage(prepareImage<VLong>(src),-1,VLongRepn);break;
            default:ODINLOG(odinlog,errorLog) << "pixel representation unknown" << STD_endl;return -1;
          }
          pdit2attr(pdit,VImageAttrList(i),dialect);
          VAppendAttr(attrlist, "image", NULL, VImageRepn, i);
          result++;
        }
      }

    //put functional data at the end of the list
    for(FileIO::ProtocolDataMap::const_iterator pdit=pdmap.begin(); pdit!=pdmap.end(); ++pdit){
      JDXtriple *diff=dynamic_cast<JDXtriple *>(const_cast<Protocol&>(pdit->first).methpars.get_parameter("Diffusion_bVector"));
      if(!diff && (pdit->second.extent(0)>1 || dialect!=lipsia)){// Its animated Data or its not lipsia (allways swaps 3rd and 4th dim)
        switch(selRepn(pdit->first,opts))  {
          case VFloatRepn:result+=copyImages<VFloat>(pdit,attrlist,VFloatRepn,dialect);break;
          case VDoubleRepn:result+=copyImages<VDouble>(pdit,attrlist,VDoubleRepn,dialect);break;
          case VBitRepn:result+=copyImages<VBit>(pdit,attrlist,VBitRepn,dialect);break;
          case VUByteRepn:result+=copyImages<VUByte>(pdit,attrlist,VUByteRepn,dialect);break;
          case VSByteRepn:result+=copyImages<VSByte>(pdit,attrlist,VSByteRepn,dialect);break;
          case VShortRepn:result+=copyImages<VShort>(pdit,attrlist,VShortRepn,dialect);break;
          case VLongRepn:result+=copyImages<VLong>(pdit,attrlist,VLongRepn,dialect);break;
          default:
            ODINLOG(odinlog,errorLog) << "pixel representation unknown" << STD_endl;
            return -1;
        }
			}
		}

    //deal with dwi
    STD_map<const double,STD_pair<Protocol,Data<float,4> > > directions;
    for(FileIO::ProtocolDataMap::const_iterator pdit=pdmap.begin(); pdit!=pdmap.end(); ++pdit){
      Protocol prot=pdit->first;
      const JDXtriple *diff=dynamic_cast<JDXtriple *>(prot.methpars.get_parameter("Diffusion_bVector"));
      if(diff){
        const Data<float,4> &dat=pdit->second;
        const TinyVector<int,4> shape(dat.shape());
        const SeqPars &seq=pdit->first.seqpars;
        if(shape(timeDim)>1){
          const double len1=seq.get_ExpDuration()*60 - seq.get_RepetitionTime()/1000;
          const double lx=len1/(shape(timeDim)-1);
          const Range all=Range::all();
          ODINLOG(odinlog,normalDebug) << "ShotTime: " << lx << "(*" << shape(timeDim) << ") for diffusion " << diff->printbody() << STD_endl;
          for(int i=0;i<shape(timeDim);i++){
            const double start=seq.get_AcquisitionStart() + lx*i;
            STD_pair<Protocol,Data<float,4> > &ref=directions[start];
            ref.second.resize(1,shape(sliceDim),shape(phaseDim),shape(readDim));
            ref.second=dat(Range(i),all,all,all);
            ref.first=pdit->first;
          }
        } else {
          STD_pair<Protocol,Data<float,4> > &ref=directions[seq.get_AcquisitionStart()];
          ref.first=pdit->first;
          ref.second.reference(pdit->second);
        }
      }
    }

    //vdtensor needs b0 in at "top", so lets write B0 first
    for(STD_map<const double,STD_pair<Protocol,Data<float,4> > >::const_iterator it=directions.begin();it!=directions.end(); ++it){
      Protocol prot=it->second.first;
      const Data<float,4> &src=it->second.second;
    	const JDXtriple *diff=dynamic_cast<const JDXtriple *>(prot.methpars.get_parameter("Diffusion_bVector"));
    	assert(diff);
      if(*diff!=JDXtriple(0,0,0))continue;
      assert(src.extent(timeDim)==1);
      ODINLOG(odinlog,infoLog) << "got B0 " << diff->printbody() << " at sec " << (it->first - directions.begin()->first) /1000 << STD_endl;
      i=copyImage(prepareImage<VShort>(src),-1,VShortRepn);
      pdit2attr(prot,src,VImageAttrList(i),dialect);
      VAppendAttr(attrlist, "image", NULL, VImageRepn, i);
      result++;
    } //and the rest afterwards
    for(STD_map<const double,STD_pair<Protocol,Data<float,4> > >::const_iterator it=directions.begin();it!=directions.end(); ++it){
      Protocol prot=it->second.first;
      const Data<float,4> &src=it->second.second;
      const JDXtriple *diff=dynamic_cast<const JDXtriple *>(prot.methpars.get_parameter("Diffusion_bVector"));
      assert(diff);
      if(*diff==JDXtriple(0,0,0))continue;
      assert(src.extent(timeDim)==1);
      ODINLOG(odinlog,infoLog) << "got diffusion " << diff->printbody() << " at sec " << (it->first - directions.begin()->first) /1000 << STD_endl;
      i=copyImage(prepareImage<VShort>(src),-1,VShortRepn);
      pdit2attr(prot,src,VImageAttrList(i),dialect);
      VAppendAttr(attrlist, "image", NULL, VImageRepn, i);
      result++;
    }


		FILE* fp=VOpenOutputFile(filename.c_str(), FALSE);
    if(fp==NULL || VWriteFile(fp,attrlist)==FALSE) {
			ODINLOG(odinlog,errorLog) << "Cannot write file >" << filename << "<("<< strerror(errno) << ")" << STD_endl;
			return -1;
		}
		else fclose(fp);
		return result;
	}
};

#endif

//////////////////////////////////////////////////////////////

void register_vista_format() {
#ifdef VISTASUPPORT
	static VistaFormat vf;
	vf.register_format();
#endif
}

