/* $Id: nws_messages.h,v 1.27 2005/05/27 00:49:18 graziano Exp $ */

/* We define here the messages ID that NWS will understand */

#ifndef NWS_MESSAGES_H
#define NWS_MESSAGES_H

#ifdef __cplusplus
extern "C" {
#endif

/* Ranges of message numbers assigned to modules that send and receive them. */

/**
 * Messages sent or recognized by the nws_memory
 *
 * STORE_STATE(state,contents) -- sent to the nws_memory process to
 *	request that the bytes #contents# should be stored with the
 *	characteristics described by #state#.  nws_memory replies with
 *	either a STATE_STORED or a MEMORY_FAILED message depending on the
 *	success of the store activity.
 * STATE_STORED(void) -- sent from the nws_memory process as the successful
 *	reply to a STORE_STATE message.
 * FETCH_STATE(state) -- sent to the nws_memory process to request that
 *	the contents of the state described by #state# be transmitted to
 *	the sender.  nws_memory replies with either a STATE_TYPE or or a
 *	MEMORY_FAILED message depending on the success of the retrieval.
 * STATE_FETCHED(state[,contents]) -- sent from the nws_memory process as
 *	the successful reply to a LOAD_STATE message.  #contents#
 *	contains the bytes of the loaded state; its characteristics are
 *	described by #state#.
 * AUTOFETCH_BEGIN(nameList) -- sent to the nws_memory process to request
 *	that it begin automatically forwarding STORE_STATE message for
 *	all of the state names listed in the tab-delimited,
 *	nul-terminated string #nameList#.  nws_memory replies with an
 *	AUTOFETCH_ACK message.  Whenever the memory receives a
 *	STORE_STATE message for any of these state names, it
 *	immediately**   transmits a STATE_FETCHED message, containing the
 *	newly-stored records, on the socket where the AUTOFETCH_BEGIN was
 *	received.  This continues until the client either sends another
 *	AUTOFETCH_BEGIN message with an empty name list or closes the
 *	connection.
 * AUTOFETCH_ACK(void) -- sent from the nws_memory process as the
 *	successful reply to a AUTOFETCH_BEGIN message.
 * MEMORY_CLEAN(seconds, series) -- send to the nws_memory process to
 * 	direct it to delete the #series# state files that have not been
 * 	accessed for the past #seconds# (an unsigned integer) seconds.
 * 	If #series# is the empty string, check all series known by the
 * 	memory.  #series# can be a comma separated list of series.
 * 	nws_memory replies with a MEMORY_CLEANED message.
 * TRANSFER_SERIES(series) -- send to the nws_memory process to instruct
 * 	it to fetch the series from another memory. #series# is a
 * 	tab-separated list of series preceded by the remote memory (in
 * 	host:port format). 
 * MEMORY_CLEANED(void) -- sent from the nws_memory process as the
 *	successful reply to a MEMORY_CLEAN message.
 * MEMORY_FAILED(void) -- sent from the nws_memory process as the
 *	unsuccessful reply to either a STORE_STATE or LOAD_STATE message.
 * STORE_AND_REGISTER(registration,contents) -- is the new version of
 * 	STORE_STATE. Instead of sending only the state (name), now you
 * 	can send the whole registration and let the memory deal with the
 * 	registration with the nameserver. Returns the same as
 * 	STORE_STATE. #registration# is a nameserver-ready registration
 * 	for the series in #contents#.
 */

#define MEMORY_FIRST_MESSAGE	100
#define MEMORY_LAST_MESSAGE	199
#define STORE_STATE		MEMORY_FIRST_MESSAGE
#define STATE_STORED		(STORE_STATE + 1)
#define FETCH_STATE		(STATE_STORED + 1)
#define STATE_FETCHED		(FETCH_STATE + 1)
#define AUTOFETCH_BEGIN		(STATE_FETCHED + 1)
#define AUTOFETCH_ACK		(AUTOFETCH_BEGIN + 1)
#define MEMORY_CLEAN		(AUTOFETCH_ACK + 1)
#define MEMORY_CLEANED		(MEMORY_CLEAN + 1)
#define MEMORY_LOGDEST		(MEMORY_CLEANED + 1)
#define MEMORY_LOGDEST_ACK	(MEMORY_LOGDEST + 1)
#define STORE_AND_REGISTER	(MEMORY_LOGDEST_ACK + 1)
#define TRANSFER_SERIES		(STORE_AND_REGISTER + 1)
#define MEMORY_FAILED		MEMORY_LAST_MESSAGE


/**
 * Messages recognized and/or generated by all NWS hosts:
 *
 * HOST_TEST(void) -- sent to the host process to request that it check
 *	its status with the name server.  The host replies with a
 *	HOST_TEST_RESULT message.
 * HOST_TEST_RESULT(testInfo) -- sent from the host process as the reply
 *	to a HOST_TEST message.  #testInfo# is a HostInfo struct
 *	containing information about the host.
 * HOST_DIAGNOSTICS(which) -- sent to the host process to request that
 *	the production of certain diagnostic messages be toggled on or
 *	off.  #which# is a signed integer that indicates which levels of
 *	diagnostics should be toggled. The host replies with a **
 *	HOST_DIAGNOSTICS_ACK message.
 * HOST_DIAGNOSTICS_ACK(void) -- sent from the host process as the
 *	successful reply to a HOST_DIAGNOSTICS message.
 * HOST_DIE([password]) -- sent to the host process to request that it
 *	exit.  The host replies with a HOST_DYING message and exits if it
 *	has no die password or if a matching #password# (a
 *	null-terminated string) accompanies the message; otherwise, it
 *	replies with a HOST_REFUSED message.
 * HOST_DYING(void) -- sent from the host process as the successful reply
 *	to a HOST_DIE message.
 * HOST_REGISTER(hostSpec) -- sent to a host to direct it to change its
 *	registration to the enclosed NWS name server.  The host responds
 *	with either a HOST_REGISTERED, a HOST_REFUSED, or a HOST_FAILED
 *	message.
 * HOST_REGISTERED(void) -- send from the host process as the successful
 *	reply to a HOST_REGISTER message.
 * HOST_LDAP_REGISTER(hostSpec) -- sent to a host to direct it to change its
 *   registration to the enclosed LDAP name server.  The host responds with
 *   either a HOST_LDAP_REGISTERED or a HOST_FAILED message.
 * HOST_LDAP_REGISTERED(hostSpec) -- send from the host process as the
 *   successful reply to a HOST_LDAP_REGISTER message
 * HOST_GET_NS(void) -- sent to the host process to request its name
 *	server.  The host replies with a HOST_GOT_NS message.
 * HOST_GOT_NS(nsInfo) -- sent from the host process as the reply to a
 *	HOST_GET_NS message.  #nsInfo# is a NsInfo struct containing
 *	information about the nameserver.
 * HOST_GET_MEMORY(nsInfo) -- sent from the host process as the reply to
 *	a HOST_GET_MEMORY message.  #nsInfo# is a NsInfo struct
 *	containing information about the memory.
 * HOST_CHANGE_MEMORY(hostSpec) -- sent to a host to direct it to change
 *	its memory to the enclosed NWS memory.  The host responds with
 *	either a HOST_MEMORY_CHANGED, a HOST_REFUSED, or a HOST_FAILED
 *	message.
 * HOST_MEMORY_CHANGED(void) -- send from the host process as the
 *	successful reply to a HOST_CHANGE_MEMORY message.
 * HOST_REGISTRATION(void) -- asks a host to return the objectset of its
 *	own registrations. Returns a HOST_MY_REGISTRATION message.
 * HOST_MY_REGISTRATION(registrations) -- answer to a HOST_REGISTRATION
 *	message.
 * HOST_FAILED(void) -- sent from the host as an unsuccessful reply to a
 *	request.
 * HOST_REFUSED(void) -- sent from the host as an unsuccessful reply to a
 *	request.  Presently sent only in response to a HOST_DIE message
 *	that does not include the proper password.
 */

#define ALLHOSTS_FIRST_MESSAGE 200
#define ALLHOSTS_LAST_MESSAGE 299
#define HOST_TEST             ALLHOSTS_FIRST_MESSAGE
#define HOST_TEST_RESULT      (HOST_TEST + 1)
#define HOST_DIAGNOSTICS      (HOST_TEST_RESULT + 1)
#define HOST_DIAGNOSTICS_ACK  (HOST_DIAGNOSTICS + 1)
#define HOST_DIE              (HOST_DIAGNOSTICS_ACK + 1)
#define HOST_DYING            (HOST_DIE + 1)
#define HOST_REGISTER         (HOST_DYING + 1)
#define HOST_REGISTERED       (HOST_REGISTER + 1)
#define HOST_LDAP_REGISTER    (HOST_REGISTERED + 1)
#define HOST_LDAP_REGISTERED  (HOST_LDAP_REGISTER + 1)
#define HOST_GET_NS           (HOST_LDAP_REGISTERED + 1)
#define HOST_GOT_NS           (HOST_GET_NS + 1)
#define HOST_GET_MEMORY       (HOST_GOT_NS + 1)
#define HOST_GOT_MEMORY       (HOST_GET_MEMORY + 1)
#define HOST_CHANGE_MEMORY    (HOST_GOT_MEMORY + 1)
#define HOST_MEMORY_CHANGED   (HOST_CHANGE_MEMORY + 1)
#define HOST_REGISTRATION     (HOST_MEMORY_CHANGED + 1)
#define HOST_MY_REGISTRATION  (HOST_REGISTRATION + 1)
#define HOST_FAILED           ALLHOSTS_LAST_MESSAGE
#define HOST_REFUSED          (HOST_FAILED - 1)


/**  
 * Messages recognized and/or generated by the periodic module:
 *
 * PERIODIC_EXPERIMENT([time, measurement, lenght], series) -- sent from
 * 	a child contains the #time# and #measurement# of an experiment.
 * 	#lenght# refers to the lenght of the registration contained in
 * 	#series#
 * 
 * (obsolete) PERIODIC_SERIES(name,series) -- sent from a child process
 * to request that the enclosed #series# (a registration Object) be
 * registered as a result of the activity named #name#.
 *
 */      

#define PERIODIC_FIRST_MESSAGE 300
#define PERIODIC_LAST_MESSAGE 399
#define PERIODIC_SERIES PERIODIC_FIRST_MESSAGE
#define PERIODIC_EXPERIMENT (PERIODIC_SERIES + 1)
#define PERIODIC_FAILED (PERIODIC_LAST_MESSAGE)


/**
 * Messages recognized and/or generated by the clique protocol:
 *
 * CLIQUE_ACTIVIATE(clique) -- sent to the sensor from a designated subprocess
 *	to indicate that the experiment for the enclosed #clique# is
 *	complete, and the next round may be scheduled.  No reply.
 * CLIQUE_DIE(clique) -- sent to a sensor from a fellow clique member to
 *	indicate that the sender is grabbing control of the enclosed
 *	#clique# token and, therefore, any held token should be
 *	discarded.  No reply.
 * CLIQUE_TOKEN_FWD(clique) -- sent to a sensor from a fellow clique member to
 *	indicate that the recipient should take control of the enclosed
 *	#clique# token and begin conducting experiments.  Alternatively,
 *	a host may be removed from a clique by sending it a
 *	CLIQUE_TOKEN_FWD along with a token which does *not* include it
 *	in the list of clique participants.  The receiver responds with a
 *	CLIQUE_ACK message.
 * CLIQUE_SERIES(name,object) -- sent to the sensor from a designated
 *	subprocess to indicate that the series designated by #object# has
 *	been generated by activity for clique #name#.  No reply.
 * CLIQUE_EXPERIMENT([time, measurement, lenght], series) -- sent from
 * 	a child contains the #time# and #measurement# of an experiment.
 * 	#lenght# refers to the lenght of the registration contained in
 * 	#series#
 * CLIQUE_NEW_MEMBER([cliqueName, host]) add #host# (the name of the
 *      host in host:port format) to the clique named #cliqueName#. The
 *      parameters are NULL terminated string sent as one.
 * CLIQUE_REMOVE_MEMBER([cliqueName, host]) remove #host# (the name of the
 *      host in host:port format) to the clique named #cliqueName#. The
 *      parameters are NULL terminated string sent as one.
 * CLIQUE_ACK(void) -- generic ack for the specific clique operation
 * 	(used as answet to CLIQUE_TOKEN_FWD and
 * 	CLIQUE_NEW_MEMBER/CLIQUE_REMOVE_MEMBER).
 * CLIQUE_FAILED(void) -- generic failure for the last clique
 * 	operation. Used as a NACK for the same messages that waits for a
 * 	CLIQUE_ACK.
 */
#define CLIQUE_FIRST_MESSAGE 400
#define CLIQUE_LAST_MESSAGE 499
#define CLIQUE_ACTIVATE   CLIQUE_FIRST_MESSAGE
#define CLIQUE_DIE        (CLIQUE_ACTIVATE + 1)
#define CLIQUE_ACK        (CLIQUE_DIE + 1)
#define CLIQUE_TOKEN_FWD  (CLIQUE_ACK + 1)
#define CLIQUE_SERIES     (CLIQUE_TOKEN_FWD + 1)
#define CLIQUE_EXPERIMENT (CLIQUE_SERIES + 1)
#define CLIQUE_NEW_MEMBER (CLIQUE_EXPERIMENT + 1)
#define CLIQUE_REMOVE_MEMBER (CLIQUE_NEW_MEMBER + 1)
#define CLIQUE_FAILED	(CLIQUE_LAST_MESSAGE)

/**
 * Messages recognized and/or generated by nws_sensor:
 *
 * ACTIVITY_START(name,control,skill,options) -- sent to the sensor to
 * 	request that it initiate a new activity.  The parameters are all
 * 	null-terminated strings.  #name# gives the registration name of
 * 	the activity, #control# the control in charge of the activity,
 * 	#skill# the skill to be invoked, and and #options# any
 * 	parameterizations of the control or skill.  The sensor replies
 * 	with either a ACTIVITY_STARTED or a SENSOR_FAILED message
 * 	depending on success.
 * ACTIVITY_STARTED(void) -- sent from the sensor as the successful reply
 * 	to a ACTIVITY_START message.
 * ACTIVITY_STOP(name) -- sent to the sensor to request that it terminate the
 * 	activity named #name#, which is a nul-terminated string.  The
 * 	sensor replies with either a ACTIVITY_STOPPED or a SENSOR_FAILED
 * 	message depending on success or failure.
 * ACTIVITY_STOPPED(void) -- sent from the sensor as the successful reply to a
 *	ACTIVITY_STOP message.
 * SENSOR_FAILED(void) -- sent from the sensor as the unsuccessful reply to a
 *	ACTIVITY_START or ACTIVITY_STOP message.
 */

#define SENSOR_FIRST_MESSAGE 600
#define SENSOR_LAST_MESSAGE 699
#define ACTIVITY_START   SENSOR_FIRST_MESSAGE
#define ACTIVITY_STARTED (ACTIVITY_START + 1)
#define ACTIVITY_STOP    (ACTIVITY_STARTED + 1)
#define ACTIVITY_STOPPED (ACTIVITY_STOP + 1)
#define SENSOR_FAILED    SENSOR_LAST_MESSAGE

/**
 *
 * Messages recognized and/or generated by nws_nameserver.
 *
 * NS_FAILED(void) -- sent from the nws_nameserver process as the
 *	unsuccessful response to either a NS_SEARCH or a NS_REGISTER message.
 * NS_REGISTER(object,timeOut) -- sent to the nws_nameserver process to
 *	request that it register #object#, a tab-delimited,
 *	double-tab-terminated list of attribute:value pairs, for the next
 *	#timeOut# (unsigned long) seconds.  A #timeOut# of zero indicates
 *	that the registration should never expire.  nws_nameserver
 *	responds with either NS_REGISTERED or NS_FAILED depending on
 *	success.
 * NS_REGISTERED(void) -- sent from the nws_nameserver process as the
 *	successful reply to a NS_REGISTER message.
 * NS_SEARCH(filter) -- sent to the nws_nameserver process to request
 *	that it search registered objects for those that match #filter#,
 *	a nul-terminated, LDAP-style search filter.  nws_nameserver
 *	replies with either NS_SEARCHED or NS_FAILED depending on
 *	success.
 * NS_SEARCHED(objectList) -- sent from the nws_nameserver process as the
 *	successful reply to a NS_SEARCH message.  #objectList# is a
 *	tab-delmited list of attribute:value pairs where each object is
 *	terminated by a pair of tabs (i.e. an empty attribute:value
 *	pair).  This list contains all registered objects that matched
 *	the search filter.
 * NS_UNREGISTER(filter) -- sent to the nws_nameserver process to request
 *	that it delete all registrations for items that match #filter#, a
 *	nul- terminated, LDAP-style search filter.  nws_nameserver
 *	replies with a NS_UNREGISTERED message.
 * NS_UNREGISTERED(void) -- sent from the nws_nameserver process as the
 *	successful reply to a NS_UNREGISTER message.
 */

#define NAMESERVER_FIRST_MESSAGE 700
#define NAMESERVER_LAST_MESSAGE 799
#define NS_REGISTER     NAMESERVER_FIRST_MESSAGE
#define NS_REGISTERED   (NS_REGISTER + 1)
#define NS_SEARCH       (NS_REGISTERED + 1)
#define NS_SEARCHED     (NS_SEARCH + 1)
#define NS_UNREGISTER   (NS_SEARCHED + 1)
#define NS_UNREGISTERED (NS_UNREGISTER + 1)
#define NS_FAILED     NAMESERVER_LAST_MESSAGE


/**
 * Messages sent/received related to the exercise of a particular skill 
 *
 * TCP_BW_REQ(sizes) --  sent to a sensor to request that it handle TCP
 * 	latency   and bandwidth experiments with the enclosed sizes (a
 * 	TcpBwCtrl -- see tcp_bw_exp.h) from the sender.
 * SKILL_EXPERIMENT(options) --  sent to a sensor to request that it
 * 	start a TCP latency and bandwidth experiments. #options# contains
 * 	the target(s), the experiment's parametersi and in particular the
 * 	skill to exercise. Returns SKILL_RESULT or SKILL_FAILED.  
 * SKILL_RESULT(resource, result) -- #result# refers to the #resource#
 * 	we've been asked to measured (by a previous SKILL_EXEPRIMENT).
 * 	(result is a double and resource is a NULL terminated string).
 */
#define SKILLS_FIRST_MESSAGE 800
#define SKILLS_LAST_MESSAGE 899
#define TCP_BW_REQ 		(SKILLS_FIRST_MESSAGE)
#define SKILL_EXPERIMENT	(TCP_BW_REQ + 1)
#define SKILL_RESULT		(SKILL_EXPERIMENT + 1)
#define SKILL_FAILED		(SKILLS_LAST_MESSAGE)

/**
 * Messages sent/received by the nws_proxy
 *
 * GET_FORECASTS(string) -- send a request to a proxy. #string#
 *	contains the resouce, options and hosts list. The proxy answers
 *	with a GOT_FORECASTS or PROXY_FAILED.
 * GOT_FORECASTS(howmany, list (int *), forecastCollections) -- returns
 * 	#howmany#, then a list of integers (howMany longs) with the index
 * 	of the hosts the proxy got forecasts of. #forecastCollection# in
 * 	answer to a GET_FORECASTS.  START_FETCHING(string) -- send a
 * 	request to a proxy. #string# contains the resouce, options and
 * 	hosts list. The proxy answers with a PROXY_ACK (it will start the
 * 	autofetching) or PROXY_FAILED.
 * START_FECTHING -- send a request to the proxy to start fetching series
 *	for a specific resource for a set of hosts. Returns ACK if
 *	succesful.
 * START_ACTIVITY_FECTHING -- send a request to the proxy to start fetching
 * 	series of an activity (typilcally a clique): all the resources
 * 	relatives to that activity will be fetched. Returns ACK if
 * 	succesful.
 */
#define PROXY_FIRST_MESSAGE	900
#define PROXY_LAST_MESSAGE	999
#define GET_FORECASTS		(PROXY_FIRST_MESSAGE)
#define GOT_FORECASTS		(GET_FORECASTS + 1)
#define START_FETCHING		(GOT_FORECASTS + 1)
#define START_ACTIVITY_FETCHING	(START_FETCHING + 1)
#define PROXY_ACK		(PROXY_LAST_MESSAGE - 1)
#define PROXY_FAILED		(PROXY_LAST_MESSAGE)

#ifdef __cplusplus
}
#endif 

#endif /* NWS_MESSAGES_H */
