/* $Id: register.c,v 1.111 2005/05/24 19:38:25 graziano Exp $ */

#include "config_nws.h"

#include <stdlib.h>
#include <string.h>         /* memcpy() strchr() strnlen() strstr() */

#include "diagnostic.h"     /* ERROR() FAIL() WARN() */
#include "strutil.h"        /* SAFESTRCPY() strmatch() vstrncpy() */
#include "messages.h"       /* Recv*() Send*() */
#include "osutil.h"         /* CurrentTime() */
#include "nws_messages.h" /* Nameserver-specific messages */
#include "register.h"

/*
** Note: We hold object sets in the format used by the name server: an object
** is a tab-delimited, tab-pair-terminated list of attribute:value pairs.  An
** object set is simply a list of these.
*/

/* it should be thread safe (given you are not overlapping memory used by
 * the parameters) */
void
AddNwsAttribute(	Object *toObject,
			const char *addName,
			const char *addValue) {

	Object expanded;
	size_t expandedLen, objectLen;
	const char *separator; 

	/* sanity check */
	if (addName == NULL || addValue == NULL || toObject == NULL) {
		return;
	}

	objectLen = strlen(*toObject);
	separator = (objectLen == (OBJECT_TERMINATOR_LEN)) ? "" : ATTRIBUTE_TERMINATOR;

	expandedLen = objectLen + strlen(separator) + strlen(addName) +
			(NAME_TERMINATOR_LEN) + strlen(addValue) + 1;

	expanded = (Object)REALLOC(*toObject, expandedLen, 1);
	
	/* Overwrite the existing object terminator. */
	vstrncpy(expanded + objectLen - (OBJECT_TERMINATOR_LEN),
		expandedLen - objectLen + (OBJECT_TERMINATOR_LEN), 5,
		separator, addName, NAME_TERMINATOR, addValue, 
		OBJECT_TERMINATOR);
	*toObject = expanded;
}


void
AddOptionToObject(	Object *toObject,
			const char *options) {
	char *name, *value, *next;
                                                                           
	for (next = NextNwsAttribute((Object)options, NO_ATTRIBUTE); next != NO_ATTRIBUTE; next = NextNwsAttribute((Object)options, next)) {
		name = NwsAttributeName_r(next);
		value = NwsAttributeValue_r(next);
		if (name == NULL || value == NULL) {
			INFO("AddOptionToObject: cannot get options\n");
		} else {
			AddNwsAttribute(toObject, name, value);
		}
		FREE(name);
		FREE(value);
	}
}


/* 
 * delete an NwsAttribute/NwsValue pair from an object 
 */
int
DeleteNwsAttribute(	Object *fromObject,
			const char *attr) {

	char *tmp, *tmp1;
	Object shrunk;

	/* is the attribute in object? */
	tmp = FindNwsAttribute(*fromObject, attr);
	if (tmp == NULL) {
		return 0;
	}

	/* let's remove it from the object */
	tmp1 = strstr(tmp, ATTRIBUTE_TERMINATOR);
	if (tmp1 == NULL) {
		/* something happended! */
		ERROR("DeleteNwsAttribute: couldn't find endOfAttribute!\n");
		return 0;
	}
	memmove(tmp, tmp1+1, strlen(tmp1+1)+1);	/* copy the \0 too */
	shrunk = (Object)REALLOC(*fromObject, sizeof(char)*(strlen(*fromObject)+1), 0);
	if (shrunk == NULL) {
		ERROR("DeleteNwsAttribute: realloc failed\n");
		return 0;
	}
	*fromObject = shrunk;
	
	return 1;
}

/* 
 * check if 2 objects are "equivalent". It should be thread safe (uses _r
 * version of NwsAttributeValue).
 *
 * returns 0 if they are different
 * 	   1 if they are equivalent
 */
int
AreObjectsEquivalent(	const Object lhs,
			const Object rhs) {

	const char *names[] = {"name", "objectclass", "activity", "port",
		"owner", "hostType", "controlName", "skillName", "option",
		"resource", "size", "period", "memory", NULL};
	char *val, *val1;
	int i, done = 1;

	/* sanity check */
	if (lhs == NO_OBJECT && rhs == NO_OBJECT) {
		return 1;
	}
	if (lhs == NO_OBJECT || rhs == NO_OBJECT) {
		return 0;
	}

	/* let's compare them */
	for (i=0; done == 1 && names[i] != NULL; i++) {
		val = NwsAttributeValue_r(FindNwsAttribute(lhs, names[i]));
		val1 = NwsAttributeValue_r(FindNwsAttribute(rhs, names[i]));
		if (val == NULL && val1 == NULL) 
			/* no such attribute in both objects */
			continue;
		if (val == NULL || val1 == NULL || strcmp(val, val1) != 0) {
			/* they are different */
			done = 0; 
		}
		FREE(val);
		FREE(val1);
	}
	return done;
}

int
AddObject(	ObjectSet *toSet,
		const Object addObject) {
	const char *endObject;
	ObjectSet expanded;
	size_t objectLen;
	size_t setLen;

	/* sanity checks */
	if (toSet == NULL || *toSet == NULL || addObject == NULL) {
		ERROR("AddObject: NULL parameter\n");
		return 0;
	}
	setLen = strlen(*toSet);
	endObject = strstr(addObject, OBJECT_TERMINATOR);
	if (endObject == NULL) {
		ERROR1("AddObject: %s doesn't have Object terminator\n", addObject);
		return 0;
	}
	objectLen = endObject - addObject + OBJECT_TERMINATOR_LEN;

	/* let's go on and add the object */
	expanded = (ObjectSet)REALLOC(*toSet, setLen + objectLen + 1, 1);

	strncat(expanded, addObject, objectLen);
	*toSet = expanded;

	return 1;
}

int
DeleteObject(	ObjectSet *fromSet,
		const Object delObject) {
	Object obj;
	ObjectSet thinned;
	int done;

	/* sanity check */
	if (fromSet == NULL || *fromSet == NULL) {
		ERROR("DeleteObject: NULL paramter\n");
		return 0;
	}

	/* let's see if we can find the object */
	thinned = NewObjectSet();
	done = 0;
	for (obj = NextObject(*fromSet, NULL); obj != NULL; obj = NextObject(*fromSet, obj)) {
		if (!AreObjectsEquivalent(obj, delObject)) {
			if (!AddObject(&thinned, obj)) {
				ERROR("DeleteObject: failed to copy object\n");
				FreeObjectSet(&thinned);
				return 0;
			}
			continue;
		} else {
			/* we found the object to delete */
			done = 1;
		}
	}
	if (!done) {
		FreeObjectSet(&thinned);
		DDEBUG("DeleteObject: no object to delete have been found\n");
		done = 0;
	} else {
		FreeObjectSet(fromSet);
		*fromSet = thinned;
	}

	return done;
}


char *
NwsAttributeName_r(const NwsAttribute ofNwsAttribute) {
	const char *endOfName;
	size_t nameLen;
	char *name = NULL;

	/* sanity check */
	if (ofNwsAttribute == NULL)
		return NULL;

	endOfName = strstr(ofNwsAttribute, NAME_TERMINATOR);

	if(endOfName != NULL) {
		nameLen = endOfName - ofNwsAttribute;
		name = (char *)MALLOC(nameLen + 1, 1);
		memcpy(name, ofNwsAttribute, nameLen);
		name[nameLen] = '\0';
	}
	return(name);
}

char *
NwsAttributeValue_r(const NwsAttribute ofNwsAttribute) {

	const char *endOfName;
	const char *endOfValue;
	size_t valueLen;
	char *value = NULL;

	/* sanity check */
	if (ofNwsAttribute == NULL)
		return NULL;

	endOfName = strstr(ofNwsAttribute, NAME_TERMINATOR);
	if(endOfName == NULL) {
		return NULL;
	}

	endOfValue = strstr(endOfName + 1, ATTRIBUTE_TERMINATOR);
	if(endOfValue == NULL) {
		/* there is no terminator: let's get to the end of the
		 * string */
		endOfValue = endOfName + strlen(endOfName);
	}

	valueLen = endOfValue - endOfName - 1;
	value = (char *)MALLOC(valueLen + 1, 1);
	memcpy(value, endOfName + 1, valueLen);
	value[valueLen] = '\0';

	return(value);

}


/* it should be thread safe */
NwsAttribute
FindNwsAttribute(const Object ofObject,
                 const char *name) {
	size_t nameLen;
	NwsAttribute nextAttr;

	/* some sanity check */
	if (name == NULL || ofObject == NO_OBJECT) {
		return NULL;
	}

	nameLen = strlen(name);

	for(nextAttr = NextNwsAttribute(ofObject, NO_ATTRIBUTE); nextAttr != NO_ATTRIBUTE; nextAttr = NextNwsAttribute(ofObject, nextAttr)) {
		if((strncasecmp(nextAttr, name, nameLen) == 0) &&
				(*(nextAttr + nameLen) == ':')) {
			break;
		}
	}
	return nextAttr;
}


/* it should be thread safe */
NwsAttribute
NextNwsAttribute(	const Object ofObject,
			const NwsAttribute preceding) {
	char *nextOne;
	
	/* sanity check */
	if (ofObject == NO_OBJECT || ofObject[0] == '\0') {
		return NO_ATTRIBUTE;
	}

	/* if there is no preceding returns the pointer to the object
	 * itself */
	if (preceding == NULL) {
		nextOne = ofObject;
	} else {
		nextOne = strstr(preceding, ATTRIBUTE_TERMINATOR);
		if (nextOne == NULL) {
			return NO_ATTRIBUTE;
		}
		nextOne += ATTRIBUTE_TERMINATOR_LEN;
	}

	return((*nextOne == '\t') ? NO_ATTRIBUTE : nextOne);
}


/* thread safe: just playing with pointer inside an Object itself */
Object
NextObject(const ObjectSet ofSet,
           const Object preceding) {
	char *nextOne, *tmp;
	
	/* sanity check */
	if (ofSet == NULL)  {
		return NO_OBJECT;
	}
	
	/* if no preceding, return the object itself */
	if (preceding == NO_OBJECT) {
		nextOne = ofSet;
	} else {
		tmp = strstr(preceding, OBJECT_TERMINATOR);
		if (tmp == NULL) {
			return NO_OBJECT;
		}
		nextOne = tmp + (OBJECT_TERMINATOR_LEN);
	}

	return((*nextOne == '\0') ? NO_OBJECT : nextOne);
}

void
FreeObject(Object *toBeFreed) {
	if(*toBeFreed != NO_OBJECT) {
		FREE(*toBeFreed);
	}
}


void
FreeObjectSet(ObjectSet *toBeFreed) {
	FREE(*toBeFreed);
}


Object
NewObject(void) {
	return(strdup(OBJECT_TERMINATOR));
}


ObjectSet
NewObjectSet(void) {
	return(strdup(""));
}

void
MakeHostCookie(	const char *host_name,
		short host_port,
		struct host_cookie *host_c) {
	SAFESTRCPY(host_c->name, host_name);
	host_c->port = host_port;
	host_c->sd = NO_SOCKET;
}

const char *
HostCImage(const struct host_cookie *host_c) {
	static char returnValue[255];

	sprintf(returnValue, "%s:%d", host_c->name, host_c->port);
	return &returnValue[0];
}

int
Host2Cookie(const char *image,
           unsigned short defaultPort,
           struct host_cookie *cookie) {

	char *c;
	char *fullName = NULL;
	IPAddress hostAddress;

	/* sanity check */
	if (image == NULL || cookie == NULL) {
		WARN("Host2Cookie: invalid parameters\n");
		return 0;
	}

	if (cookie->name != image) {
		SAFESTRCPY(cookie->name, image);
	}
	cookie->port = defaultPort;

	c = strchr(cookie->name, ':');
	if (c != NULL) {
		/* remove the port part ... */
		*c = '\0';
		/* ... and override the default port */
		cookie->port = (unsigned short)strtol(c + 1, NULL, 10);
	}

	/* Normalize to an all-lowercase, fully-qualified DNS name. */
	if(IPAddressValue(cookie->name, &hostAddress)) {
		/* we got an IP address: let's convert to a name */
		fullName = IPAddressMachine_r(hostAddress);
		if (fullName && *fullName == '\0') {
			FREE(fullName);
		}
	}
	if (fullName == NULL) {
		WARN1("Host2Cookie: couldn't resolve %s\n", cookie->name);
	} else {
		/* we got something useful */
		SAFESTRCPY(cookie->name, fullName);
		FREE(fullName);
	}
	strcase(cookie->name, ALL_LOWER);

	/* no socket yet */
	cookie->sd = NO_SOCKET;

	return 1;
}

int
ConnectToHost(	struct host_cookie *host_c,
		int *sd) {

	int addrCount;
	IPAddress addrs[MAX_ADDRESSES], addr;
	int i, ret;
	char *tmp;

	/* sanity check */
	if (host_c == NULL) {
		ERROR("ConnectToHost: NULL cookie\n");
		if (sd != NULL) {
			*sd = NO_SOCKET;
		}
		return 0;
	}

	/* we use strlen so we put a EOS at the end (should already be
	 * terminated anyway */
	host_c->name[sizeof(host_c->name) - 1] = '\0';

	/* now it gets somethwat weird: we are getting all the possible
	 * inet address we can find for this host. We then check if the
	 * socket belongs to the right host. 
	 */
	addrCount = IPAddressValues(host_c->name, addrs, MAX_ADDRESSES);
	if(addrCount == 0) {
		WARN1("ConnectToHost: failed to resolve %s\n", host_c->name);
	}

	ret = 0;
	if(IsOkay(host_c->sd)) {
		/* did we resolve the name? */
		if (addrCount > 0) {
			/* let's check if the socket is indeed the for
			 * the right peer */
			addr = Peer(host_c->sd);
			for (i = 0; i < addrCount; i++) {
				if (addr.addr != 0 && addrs[i].addr == addr.addr) {
					ret = 1;
					break;
				}
			}
		} else if (addrCount == 0) {
			/* addCount is 0, so we have no idea what is the
			 * inet address of this host: we hope this is
			 * a good socket */
			ret = 1;
		}
	}

	/* do we have a good socket? */
	if (ret == 0) {
		/* nope: trying to get a new one.  let's drop a possibly
		 * old socket */
		DROP_SOCKET(&host_c->sd);
		for(i = 0; i < addrCount; i++) {
			if(CallAddr(addrs[i], host_c->port, &host_c->sd, -1)) {
				/* got a good socket */
				ret = 1;
				break;
			}
		}
	}

	if (ret == 1) {
		/* assign the socket */
		if (sd != NULL) {
			*sd = host_c->sd;
		}

		/* we have a good socket: let's check if the cookie is
		 * good (name and port are set that is) */
		if (host_c->name[0] == '\0') {
			/* name */
			tmp = PeerName_r(host_c->sd);
			SAFESTRCPY(host_c->name, tmp);
			FREE(tmp);
			/* port */
			host_c->port = PeerNamePort(host_c->sd);
		}
	} else {
		/* no socket here */
		ERROR1("ConnectToHost: connect to %s failed\n", HostCImage(host_c));
	}

	return ret;
}

int
ConnectToObject(Object reg, struct host_cookie *cookie) {
	const char *ip;
	char *name, tmp[64];
	int ret;
	struct host_cookie tmpCookie;

	/* sanity check */
	if (reg == NULL || cookie == NULL) {
		ERROR("ConnectToObject: registration or cookie is NULL\n");
		return 0;
	}

	/* let's get the port */
	name = NwsAttributeValue_r(FindNwsAttribute(reg, "port"));
	if (name == NULL) {
		ERROR("ConnectToObject: object doesn't have 'port'\n");
		return 0;
	}
	tmpCookie.port = (unsigned short)strtol(name, NULL, 10);
	FREE(name);
	
	/* we can have multiple IPs so let's try to connect to all of
	 * them: we take the first one to be succesfull */
	name = NwsAttributeValue_r(FindNwsAttribute(reg, "ipAddress"));
	if (name == NULL) {
		ERROR("ConnectToObject: object doesn't have 'ipAddress'\n");
		return 0;
	}
	ret = 0;
	for (ip = name; GETTOK(tmp, ip, ",", &ip) != 0; ) {
		/* let's see if we talk to this IP */
		SAFESTRCPY(tmpCookie.name, tmp);
		tmpCookie.sd = cookie->sd;

		if (ConnectToHost(&tmpCookie, NULL)) {
			/* got it: we have a connection */
			SAFESTRCPY(cookie->name, tmpCookie.name);
			cookie->port = tmpCookie.port;
			
			/* did we get a new socket? */
			if (cookie->sd != tmpCookie.sd) {
				/* let's drop a possibly old socket */
				DROP_SOCKET(&cookie->sd);

				/* and get the new one in */
				cookie->sd = tmpCookie.sd;
			}
			ret = 1;
			break;
		}
	}
	FREE(name);

	if (ret == 0) {
		 /* no socket here */
		name = NwsAttributeValue_r(FindNwsAttribute(reg, "name"));
		if (name != NULL) {
			ERROR1("ConnectToObject: failed to connect to %s\n", name);
		} else {
			ERROR1("ConnectToObject: failed to connect to %s\n", reg);
		}
		FREE(name);
	}
	
	return ret;
}

int
Register(struct host_cookie *withWho,
         const Object object,
         double forHowLong) {

	unsigned long expiration;
	DataDescriptor objectDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	Socket sd;
	DataDescriptor timeDescriptor = SIMPLE_DATA(UNSIGNED_LONG_TYPE, 1);
	char *tmp;

	/* sanity check */
	tmp = strstr(object, OBJECT_TERMINATOR);
	if (tmp == NULL) {
		ERROR1("Register: 'object' is not terminated (%s)\n", object);
		return (0);
	}

	/* we pick only the first object (in case there are multiple) */
	objectDescriptor.repetitions = tmp - object + (OBJECT_TERMINATOR_LEN);
	if(!ConnectToHost(withWho, &sd)) {
		FAIL1("Register: failed to connect to %s\n", HostCImage(withWho));
	}

	expiration = (unsigned long)forHowLong;

	if(!SendMessageAndDatas(sd,
                          NS_REGISTER,
                          object,
                          &objectDescriptor,
                          1,
                          &expiration,
                          &timeDescriptor,
                          1,
                          -1)) {
		DROP_SOCKET(&withWho->sd);
		FAIL("Register: send message failed\n");
	}

	/* we don't check for the ack here anymore: look in
	 * host_protcol.c */

	return(1);
}

int
RetrieveObjects(struct host_cookie *whoFrom,
                const char *filter,
                ObjectSet *retrieved,
		int timeout) {

	DataDescriptor filterDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	ObjectSet filtered;
	DataDescriptor objectsDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	size_t replySize;
	Socket sd;

	/* sanity check */
	if (whoFrom == NULL || filter == NULL || retrieved == NULL) {
		WARN("RetrieveObjects: NULL parameter(s)\n");
		return 0;
	}

	if(!ConnectToHost(whoFrom, &sd)) {
		FAIL("RetrieveObject: unable to contact name server.\n");
	}

	filterDescriptor.repetitions = strlen(filter) + 1;
	if(!SendMessageAndData(sd,
                         NS_SEARCH,
                         filter,
                         &filterDescriptor,
                         1,
                         timeout)) {
		DROP_SOCKET(&whoFrom->sd);
		FAIL("RetrieveObjects: send failed\n");
	} else if(!RecvMessage(sd,
                       NS_SEARCHED,
                       &replySize,
                       timeout)) {
		DROP_SOCKET(&whoFrom->sd);
		FAIL("RetrieveObjects: message receive failed\n");
	}

	filtered = (ObjectSet)MALLOC(replySize + 1, 0);
	if(filtered == NULL) {
		FAIL("RetrieveObjects: out of memory\n");
	}

	objectsDescriptor.repetitions = replySize;
	if(!RecvData(sd,
			filtered,
			&objectsDescriptor,
			1,
			timeout)) {
		FREE(filtered);
		DROP_SOCKET(&whoFrom->sd);
		FAIL("RetrieveObjects: data receive failed\n");
	}
	filtered[objectsDescriptor.repetitions] = '\0';
	*retrieved = filtered;

	return(1);
}


int
Unregister(struct host_cookie *withWho,
           const char *filter) {

	DataDescriptor filterDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	Socket sd;

	if(!ConnectToHost(withWho, &sd)) {
		FAIL("Unregister: unable to contact name server.\n");
	}

	filterDescriptor.repetitions = strlen(filter) + 1;
	if(!SendMessageAndData(sd,
                         NS_UNREGISTER,
                         filter,
                         &filterDescriptor,
                         1,
                         -1)) {
		DROP_SOCKET(&withWho->sd);
		FAIL("Unregister: send failed\n");
	}

	/* we don't check for the ack here anymore: look in
	 * host_protcol.c */

	return 1;
}


/*-------------- struct registrations functions ---------*/
/* the following functions operates on elements of struct registrations */

/*
 * initialize a registrations structure, Returns 0 on failure (out of
 * memory).
 */
int
InitRegistrations(registrations **r) {
	*r = (registrations *) MALLOC(sizeof(registrations), 0);
	if (*r == NULL) {
		return 0;
	}
	(*r)->howMany = 0;
	(*r)->dirty = 0;
	(*r)->vals = NULL;
	(*r)->expirations = NULL;

	return 1;
}
	

/* 
 * simply check if there is enough space for an extra element in #r#
 * (registrations). If not, adds REG_CHUNCK elelments to it
 */
#define REG_CHUNK (512)
void
CheckRegistrations(registrations *r) {

	/* do we have extra elements handy? */
	if (r->dirty <= 0) {
		/* we don't have extra one: we have to realloc */
		r->vals = (char **)REALLOC((void *)r->vals, sizeof(char *)*(r->howMany+REG_CHUNK), 1);
		/* let's make room for the expirations */
		r->expirations = (unsigned long *)REALLOC((void*)r->expirations, sizeof(long)*(r->howMany+REG_CHUNK), 1);

		/* let's zeroes the pointers */
		memset(&(r->vals[r->howMany]), 0, sizeof(char *)*REG_CHUNK);
		r->dirty += REG_CHUNK;
	}
}

/* insert #registration# with #expiration# in #r#. It keeps the list ordered:
 * objects that AreObjectsEquivalent will be substituted with the new
 * one. If there is no such object, it will be added in the right place.
 * Returns 0 on failure, on success returns 1. */
int
InsertRegistration(	registrations *r,
			const Object object,
			unsigned long expiration,
			int i) {
	char *name, *newName, *obj;
	char **tmp;
	unsigned long *tmp_l;

	/* sanity check */
	if (r == NULL || object == NULL) {
		ERROR("InsertRegistration: wrong parameter\n");
		return 0;
	}
	name = strstr(object, OBJECT_TERMINATOR);
	if (name == NULL) {
		ERROR("InsertRegistration: object with no termination\n");
		return 0;
	}

	/* we get only the first object (in case there are more then one) */
	obj = (char *)MALLOC(name - object + OBJECT_TERMINATOR_LEN + 1, 1);
	memcpy(obj, object, name - object + OBJECT_TERMINATOR_LEN);
	obj[name - object + OBJECT_TERMINATOR_LEN] = '\0';
	newName = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
	if (newName == NULL) {
		FREE(obj);
		ERROR("InsertRegistration: registration with no name!\n");
		return 0;
	}
	FREE(newName);

	/* let's check we have enough room for an extra registration */
	CheckRegistrations(r);

	/* let's move the pointers one up to make room for the
	 * new registration */
	tmp = r->vals;
	tmp += i;
	memmove(tmp + 1, tmp, sizeof(char*)*(r->howMany - i));
	/* move the expirations */
	tmp_l = r->expirations;
	tmp_l += i;
	memmove(tmp_l + 1, tmp_l, sizeof(long)*(r->howMany - i));

	/* we have an extra registration */
	r->howMany++;
	r->dirty--;

	/* let's copy the registration */
	r->vals[i] = obj;
	r->expirations[i] = expiration;

	return 1;
}

/* delete the registration at #ind# from #r#. Returns 0 on failure */
int
DeleteRegistration(	registrations *r,
			int ind) {
	char **tmp;
	unsigned long *tmp_l;

	/* sanity check */
	if (r == NULL || ind >= r->howMany) {
		ERROR("DeleteRegistration: wrong parameters\n");
		return 0;
	}

	/* free the memory */
	FREE(r->vals[ind]);

	/* move all the registrations */
	tmp = r->vals;
	tmp += ind;
	memmove(tmp, tmp + 1, sizeof(char *)*(r->howMany - ind - 1));
	/* moved the expirations */
	tmp_l = r->expirations;
	tmp_l += ind;
	memmove(tmp_l, tmp_l + 1, sizeof(long)*(r->howMany - ind - 1));

	/* adjust the numbers of current registrations */
	r->howMany--;
	r->dirty++;

	return 1;
}




/* it looks for #name# in the values of the 'name' attribute in the
 * registrations #r#. It returns the positions where the #name# belongs
 * in #r# in #index# (it can returns r->howMany!): it returns 1 if at
 * least an objects was found (and the higher index if more than one was
 * present) or 0 if the objects would need to be inserted. If
 * #matchExact# is not set, returns success when matching up to
 * strlen(name), not the length of the registration's name.
 */
int 
SearchForName(	registrations *r,
		const char *name,
		int matchExact,
		int *ind) {
	int i, ln, done, len;
	double offset;
	char *tmp;

	/* sanity check */
	if (name == NULL || r == NULL || ind == NULL) {
		WARN("SearchForName: NULL parameters\n");
		return 0;
	}
	len = strlen(name);
	if (len == 0) {
		WARN("SearchForName: empty parameters\n");
		return 0;
	}

	offset = (double)r->howMany/2;
	done = -1;
	for (tmp = NULL, i = -1; (int)(offset + 0.5) != 0; ) {
		/* adjust the element we are going to check */
		if (done < 0) {
			/* we have to sum offset */
			i += (int)(offset + 0.5);
		} else if (done > 0) {
			/* we have to subtract the offset */
			i -= (int)(offset + 0.5);
		}

		/* let's check the limits */
		if (i < 0) {
			/* done here */
			offset = 0;
			i = 0;
		} else if (i >= r->howMany) {
			/* same here */
			offset = 0;
			i = r->howMany;
		}

		/* pick the name of the element */
		tmp = NwsAttributeValue_r(FindNwsAttribute(r->vals[i], "name"));
		if (matchExact) {
			done = strcmp(tmp, name);
		} else {
			/* let's see if we match it */
			if (strmatch(tmp, name)) {
				/* found it */
				done = 0;
			} else {
				done = strncmp(tmp, name, len);
			}
		}
		FREE(tmp);
		if (done == 0) {
			/* found it */
			break;
		}

		/* halve the distance we are using */
		offset = offset/2;
	}

	/* now let's look for the higher index (if we found a
	 * registration) of the same name registration (ie series may
	 * have the same name) */
	if (done == 0) {
		while ((i + 1) < r->howMany) {
			tmp = NwsAttributeValue_r(FindNwsAttribute(r->vals[i+1], "name"));
			if (matchExact) {
				ln =  strcmp(tmp, name);
			} else {
				if (strmatch(tmp, name)) {
					ln = 0;
				} else {
					ln = strncmp(tmp, name, len);
				}
			}
			FREE(tmp);
			if (ln == 0) {
				/* another similar registration */
				i++;
			} else {
				/* we are done: no more same name 
				 * registrations */
				break;
			}
		}
	} else if (done < 0) {
		/* we are down one */
		i++;
	}
	*ind = i;

	return (done == 0);
}

/*
 * Searches for an object equivalent to #obj#. Returns 1 on success and
 * the position #ind# in registration #r#, 0 otherwise and #ind# contains
 * the index into which the object should be inserted to keep the #r#
 * ordered.
 */
int
SearchForObject(	registrations *r,
			Object obj,
			int *ind) {
	char *name, *newName;
	int i, ret;

	/* sanity check */
	if (ind == NULL) { 
		ERROR("SearchForObject: null parameters\n");
		return 0;
	}
	*ind = -1;		/* we default to error */
	if (r == NULL || obj == NULL) {
		ERROR("SearchForObject: null parameters\n");
		return 0;
	}

	/* let's get the name and search for it */
	newName = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
	if (newName == NULL) {
		ERROR("SearchForObject: object with no name\n");
		return 0;
	}
	if (!SearchForName(r, newName, 1, &i)) {
		FREE(newName);
		/* nope, there is no such a beast */
		*ind = i;
		return 0;
	}

	/* we need to look if they are equivalent */
	name = NwsAttributeValue_r(FindNwsAttribute(r->vals[i],"name"));
	if (name == NULL) {
		FREE(newName);
		ERROR("SearchForObject: out of memory\n");
		return 0;
	}
	*ind = i;	/* this is a good spot to insert :) */
	/* now we walk down looking for the same (similar)
	 * registration and reuse its slot if we find it */
	ret = 0;
	while (strcmp(name, newName) == 0) {
		if (AreObjectsEquivalent(r->vals[i], obj)) {
			/* found a similar registration */
			*ind = i;
			ret = 1;
			break;
		}
		/* nope: let's get lower if we can */
		if (i == 0) {
			/* we are at the beginning of the list */
			break;
		}
		i--;

		/* get the next name */
		FREE(name);
		name = NwsAttributeValue_r(FindNwsAttribute(r->vals[i],"name"));
		if (name == NULL) {
			ERROR("SearchForObject: out of memory\n");
			break;
		}
	}

	/* nope, we didn't find the beast */
	FREE(name);
	FREE(newName);

	return ret;
}

/* utility function to get a tab-separated list of names from a
 * registrations structure. Returns a char * that needs to be freed */
char *
GenerateList(const registrations *r) {
	char *tmp, *name;
	int len, i, l;

	/* sanity check */
	if (r == NULL) { 
		ERROR("GenerateList: NULL parameter\n");
		return NULL;
	}


	/* let's generate the list */
	for (tmp = NULL, len = 0, i = 0;  i < r->howMany; i++) {
		name = NwsAttributeValue_r(FindNwsAttribute(r->vals[i], "name"));
		if (name == NULL) {
			WARN("GenerateList: no name??\n");
			continue;
		}
		l = strlen(name);
		tmp = REALLOC(tmp, len + l + 2, 1);
		if (len > 0) {
			tmp[len] = '\t';
			len++;
		}
		memcpy(tmp + len, name, l + 1);
		len += l;
		FREE(name);
	}

	return tmp;
}

