/* $Id: experiments.c,v 1.9 2005/07/06 18:47:47 graziano Exp $ */

#include "config_nws.h"

#include <stdlib.h>
#include <string.h>

#include "diagnostic.h"    /* FAIL() WARN() */
#include "host_protocol.h" /* Host connection & registration routines */
#include "strutil.h"       /* GETTOK() GETWORD() SAFESTRCPY() vstrncpy() */
#include "messages.h"      /* message send/receive */
#include "osutil.h"        /* CurrentTime() */
#include "nws_memory.h"    /* Memory-specific messages */


/**
 * Packs the #count#-long array of experiments #from# into #to#,
 * returning the record size in #rec_size#.
 */
static int
PackExperiments(	const NWSAPI_Measurement *from,
			size_t count,
			char **to,
			size_t *rec_size) {
	char *curr;
	int i;
	char rec_buff[63 + 1];

	/* make a test record to see how big they will be */
	memset(rec_buff, 0, sizeof(rec_buff));
	sprintf(rec_buff, WEXP_FORMAT, from->timeStamp, from->measurement);
	*rec_size = strlen(rec_buff);

	*to = (char *)MALLOC(*rec_size * count + 1, 0);
	if(*to == NULL) {
		FAIL("PackExperiments: out of memory\n");
	}
	memset(*to, ' ', *rec_size * count);

	curr = *to;

	for(i = 0; i < count; i++) {
		memset(rec_buff, 0, sizeof(rec_buff));
		sprintf(curr, WEXP_FORMAT, from->timeStamp, from->measurement);
		from++;
		curr += *rec_size;
	}

	return(1);
}


/**
 * Extracts experiment information from #from# and stores it in the
 * #max_count#-long array #to#.  Returns the number of experiments stored in
 * #count#.
 *
 */
int
UnpackExperiments(	const struct nws_memory_state *s,
			const char *from,
			size_t max_count,
			NWSAPI_Measurement *to,
			size_t *count) {
	const char *curr;
	int i;
	char word[128];

	curr = from;

	for(i = 0; (i < s->rec_count) && (i < max_count); i++) {
		if(!GETWORD(word, curr, &curr)) {
			WARN1("UnpackExperiments: formatting failed: %s\n", curr);
			break;
		}
		to[i].timeStamp = strtod(word, NULL);
		if(!GETWORD(word, curr, &curr)) {
			WARN1("UnpackExperiments: formatting failed: %s\n", curr);
			break;
		}
		to[i].measurement = strtod(word, NULL);
	}
	*count = i;

	return(1);
}


int
LoadExperiments(	struct host_cookie *mem_c,
			const char *exp_name,
			NWSAPI_Measurement *meas,
			size_t count,
			double seq_no,
			size_t *out_count,
			double *out_seq_no,
			double timeout) {
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 0);
	char *expContents;
	struct nws_memory_state expState;
	Socket memorySock;
	size_t recvSize;

	/* sanity check */
	if (exp_name == NULL || mem_c == NULL || out_count == NULL) {
		FAIL("LoadExperiments: invalid parameter\n");
	}
	*out_count = 0;
	if (out_seq_no) {
		*out_seq_no = 0.0;
	}

	/* let's contact the memory */
	if(!ConnectToHost(mem_c, &memorySock)) {
		FAIL2("LoadExperiments: couldn't contact server %s at port %d\n", mem_c->name, mem_c->port);
	}

	memset(&expState, 0, sizeof(struct nws_memory_state));
	SAFESTRCPY(expState.id, exp_name);
	expState.rec_count = count;

	/* a negative seq_no means to read count number of records
	 * regardless */
	expState.seq_no = (seq_no >= 0.0) ? seq_no : 0.0;

	if(!SendMessageAndData(	memorySock,
				FETCH_STATE,
				&expState,
				stateDescriptor,
				stateDescriptorLength,
				timeout)) {
		DisconnectHost(mem_c);
		FAIL("LoadExperiments: error making request\n");
	} else if(!RecvMessage(memorySock, STATE_FETCHED, &recvSize, timeout)) {
		DisconnectHost(mem_c);
		FAIL("LoadExperiments: message receive failed\n");
	} else if(recvSize == 0) {
		WARN("LoadExperiments: no records in memory?\n");
	} else if(!RecvData(	memorySock,
				&expState,
				stateDescriptor,
				stateDescriptorLength,
				timeout)) {
		DisconnectHost(mem_c);
		FAIL("LoadExperiments: state receive failed\n");
	} else if(expState.rec_count == 0) {
		if (seq_no >= 0.0) {
			LOG("LoadExperiments: no new data available?\n");
		} else {
			WARN("LoadExperiments: no records in memory?\n");
		}
	} else {
		des.repetitions = expState.rec_size * expState.rec_count;
		expContents = (char *)MALLOC(des.repetitions + 1, 0);
		if (expContents == NULL) {
			DisconnectHost(mem_c);
			FAIL("LoadExperiments: out of memory\n");
		}
		if (!RecvData(	memorySock,
				expContents,
				&des,
				1,
				timeout)) {
			FREE(expContents);
			DisconnectHost(mem_c);
			FAIL("LoadExperiments: data receive failed\n");
		}
		expContents[des.repetitions] = '\0';
		if (!UnpackExperiments(&expState, expContents, count, meas, out_count)) {
			FREE(expContents);
			FAIL("LoadExperiments: unable to unpack experiment data.\n");
		}
		FREE(expContents);
	}

	if (out_seq_no) {
		*out_seq_no = expState.seq_no;
	}

	return(1);
}

int
StoreNewExperiments(	struct host_cookie *mem_c,
			const char *id,
			const NWSAPI_Measurement *exper,
			size_t count,
			double forHowLong) {
	char *content;
	DataDescriptor contentsDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	struct nws_memory_new_state expState;
	size_t recSize;
	int ret;
	MessageHeader header;

	if (!ConnectToHost(mem_c, NULL)) {
		INFO2("StoreNewExperiments: couldn't contact server %s at port %d\n", mem_c->name, mem_c->port);
		return 0;
	}

	if (!PackExperiments(exper, count, &content, &recSize)) {
		FAIL("StoreNewExperiments: packing failed\n");
	}

	/* as of version 2.9, we piggy back the id (that now is the full
	 * experiment registration) in the content */
	expState.id_len = strlen(id);
	expState.rec_count = count;
	expState.seq_no = CurrentTime();
	expState.rec_size = recSize;
	expState.time_out = forHowLong;
	contentsDescriptor.repetitions = count * recSize;
	
	/* now let's piggy back the registration */
	contentsDescriptor.repetitions += expState.id_len;
	content = (char *)REALLOC(content, contentsDescriptor.repetitions, 1);
	memmove(content + expState.id_len, content, count * recSize);
	memcpy(content, id, expState.id_len);

	ret = SendMessageAndDatas(mem_c->sd,
       	                   STORE_AND_REGISTER,
       	                   &expState,
       	                   newStateDescriptor,
       	                   newStateDescriptorLength,
       	                   content,
       	                   &contentsDescriptor,
       	                   1,
       	                   -1);
	FREE(content);
	if (!ret) {
		DisconnectHost(mem_c);
		FAIL("StoreNewExperiments: error storing experiment\n");
	}

	/* let's get the ack */
	if (!RecvHeader(mem_c->sd, &header, -1)) {
		DisconnectHost(mem_c);
		FAIL("StoreNewExperiments: error receiving ack\n");
	}
	/* let's check is the right one */
	if (header.message == MEMORY_FAILED) {
		DisconnectHost(mem_c);
		FAIL("StoreNewExperiments: received MEMORY_FAILED\n");
	} else if (header.message != STATE_STORED) {
		DisconnectHost(mem_c);
		FAIL("StoreNewExperiments: received unknown message\n");
	}

	return ret;
}

/**
 * This function send a MEMORY_CLEAN message to the specified memory.
 */
int
MemoryClean(	struct host_cookie *mem,
		const char *series,
		unsigned long idle) {
	DataDescriptor contentsDes = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor idleDes = SIMPLE_DATA(UNSIGNED_LONG_TYPE, 1);
	MessageHeader header;
	int ret;

	/* sanity check */
	if (mem == NULL || series == NULL) {
		ERROR("MemoryClean: NULL parameter!\n");
		return 0;
	}

	if (!ConnectToHost(mem, NULL)) {
		INFO2("MemoryClean: couldn't talk to memory %s at port %d\n", mem->name, mem->port);
		return 0;
	}

	contentsDes.repetitions = strlen(series);
	ret = SendMessageAndDatas(mem->sd,
       	                   MEMORY_CLEAN,
       	                   &idle,
       	                   &idleDes,
       	                   1,
       	                   series,
       	                   &contentsDes,
       	                   1,
       	                   -1);
	if (!ret) {
		DisconnectHost(mem);
		FAIL("MemoryClean: failed to commicate with memory\n");
	}

	/* let's get the ack */
	if (!RecvHeader(mem->sd, &header, -1)) {
		DisconnectHost(mem);
		FAIL("MemoryClean: error receiving ack\n");
	}
	/* let's check is the right one */
	if (header.message != MEMORY_CLEANED) {
		DisconnectHost(mem);
		FAIL("MemoryClean: received MEMORY_FAILED\n");
	}

	return ret;
}
