# -*- coding: utf-8 -*-
#
#  aya5.py - an aya.dll(Ver.5) compatible Shiori module for ninix
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
#

# TODO:
# - str()ではなくunicode()が必要な箇所が残っている.
# - 文字列内埋め込み要素の展開の動作が非互換.
# - システム関数:
# - たくさん.

import os
import sys
import logging
import random
import time
import math
import shutil
import re

from ninix.home import get_normalized_path


class AyaError(ValueError):
    pass


def encrypt_char(char):
    c = ord(char)
    j = 0
    while j < 3:
        msb = c & 0x80
        c <<= 1
        c &= 0xff
        if msb:
            c |= 0x01
        else:
            c &= 0xfe
        j += 1
    c ^= 0xd2
    return chr(c)

def decrypt_char(char):
    c = ord(char)
    c ^= 0xd2
    j = 0
    while j < 3:
        lsb = c & 0x01
        c >>= 1
        if lsb:
            c |= 0x80
        else:
            c &= 0x7f
        j += 1
    return chr(c)

def decrypt_readline(f):
    line = ''
    while 1:
        c = f.read(1)
        if c == '':
            break
        line = ''.join((line, decrypt_char(c)))
        if line.endswith(chr(10)) or \
           line.endswith(''.join((chr(13), chr(10)))):
            break
    return line

def find_not_quoted(line, token):
    position = 0
    while 1:
        pos_new = line.find(token, position)
        if pos_new < 0:
            break
        elif pos_new == 0:
            break
        position = line.find('"', position)
        if 0 <= position < pos_new:
            position += 1
            while position < len(line) - 1:
                if line[position] == '"':
                    position += 1
                    break
                else:
                    position += 1
                    continue
        else:
            break
    return pos_new

def find_comment(line):
    if line.startswith('//'):
        return 0, len(line)
    start = len(line) # not len(line) - 1
    end = -1
    for token in ['//', '/*']:
        pos_new = find_not_quoted(line, token)
        if 0 <= pos_new < start:
            start = pos_new
            if token == '/*':
                end = find_not_quoted(line, '*/')
                if end >= 0:
                    end += 2
            else:
                end = len(line)
    if start == len(line):
        start = -1
    return start, end

def get_aya_version(filelist):
    if not filelist:
        return 0
    dic_files = filelist
    for filename in dic_files:
        if filename.lower().endswith('_shiori3.dic'): # XXX
            with open(filename) as f:
                for line in f:
                    try:
                        line = unicode(line, 'Shift_JIS') # XXX
                        if line.find(unicode('for 文 version 4', 'utf-8')) > 0:
                            return 4
                        elif line.find('for AYA5') > 0:
                            return 5
                    except:
                        return 5
    else:
        return 3

def find_dict(aya_dir, f):
    comment = 0
    dic_files = []
    for line in f:
        line = unicode(line, 'Shift_JIS', 'ignore') # XXX
        if comment:
            end = find_not_quoted(line, '*/')
            if end < 0:
                continue
            else:
                line = line[end + 2:]
                comment = 0
        while 1:
            start, end = find_comment(line)
            if start < 0:
                break
            if end < 0:
                comment = 1
                line = line[:start]
                break
            line = ' '.join((line[:start], line[end:]))
        line = line.strip()
        if not line:
            continue
        if ',' not in line:
            continue
        key, value = [x.strip() for x in line.split(',', 1)]
        if key == 'dic':
            filename = get_normalized_path(value, encode=0)
            path = os.path.join(aya_dir, filename)
            dic_files.append(path)
    return dic_files

def check_version(top_dir, dll_name):
    filename = None
    if os.path.isfile(os.path.join(top_dir, 'aya.txt')):
        filename = os.path.join(top_dir, 'aya.txt')
    elif os.path.isfile(os.path.join(top_dir, 'yaya.txt')):
        return 6 # XXX: YAYA
    elif dll_name is not None and \
         os.path.isfile(os.path.join(top_dir, ''.join((dll_name[:-3], 'txt')))):
        filename = os.path.join(top_dir, ''.join((dll_name[:-3], 'txt')))
    if filename is not None:
        with open(filename) as f:
            version = get_aya_version(find_dict(top_dir, f))
    else:
        version = 0
    return version


class Shiori(object):

    __AYA_TXT = 'aya.txt'
    __DBNAME = 'aya_variable.cfg'

    def __init__(self, dll_name):
        self.dll_name = dll_name
        if dll_name is not None:
            self.__AYA_TXT = ''.join((dll_name[:-3], 'txt'))
            self.__DBNAME = ''.join((dll_name[:-4], '_variable.cfg'))
        self.saori = None
        self.dic_files = []

    def use_saori(self, saori):
        self.saori = saori

    def find(self, top_dir, dll_name):
        result = 0
        version = check_version(top_dir, dll_name)
        if version == 5:
            result = 200
        return result

    def show_description(self):
        logging.info(
            'Shiori: AYA5 compatible module for ninix\n'
            '        Copyright (C) 2002-2011 by Shyouzou Sugitani\n'
            '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko')

    def reset(self):
        self.boot_time = time.time()
        self.aitalk = 0
        self.dic_files = []
        self.dic = AyaDictionary(self)
        self.global_namespace = AyaGlobalNamespace(self)
        self.system_functions = AyaSystemFunctions(self)
        self.logfile = None
        self.filelist = {}

    def load(self, aya_dir=None):
        self.aya_dir = aya_dir
        self.dbpath = os.path.join(self.aya_dir, self.__DBNAME)
        self.saori_library = AyaSaoriLibrary(self.saori, self.aya_dir)
        self.reset()
        try:
            path = os.path.join(self.aya_dir, self.__AYA_TXT)
            with open(path) as aya_txt:
                self.load_aya_txt(aya_txt)
        except IOError:
            logging.debug('cannot read aya.txt')
            return 0
        except AyaError as error:
            logging.debug(error)
            return 0
        self.global_namespace.load_database(self)
        # default setting
        if not self.global_namespace.exists('log'):
            self.global_namespace.put('log', '')
        for path in self.dic_files:
            basename, ext = os.path.splitext(path)
            ext = ext.lower()
            if ext == '.ayc':
                encrypted = 1
            else:
                encrypted = 0
            try:
                with open(path, 'r') as dicfile:
                    self.dic.load(dicfile, encrypted)
            except:
                logging.debug('cannnot read {0}'.format(path))
                continue
        func = self.dic.get_function('load')
        if func:
            func.call([self.aya_dir.replace('/', '\\')])
        return 1

    def load_aya_txt(self, f):
        comment = 0
        self.charset = 'Shift_JIS' # default
        for line in f:
            line = unicode(line, self.charset, 'ignore')
            if comment:
                end = find_not_quoted(line, '*/')
                if end < 0:
                    continue
                else:
                    line = line[end + 2:]
                    comment = 0
            while 1:
                start, end = find_comment(line)
                if start < 0:
                    break
                if end < 0:
                    comment = 1
                    line = line[:start]
                    break
                line = ' '.join((line[:start], line[end:]))
            line = line.strip()
            if not line:
                continue
            if ',' not in line:
                continue
            key, value = [x.strip() for x in line.split(',', 1)]
            self.evaluate_config(key, value)

    def evaluate_config(self, key, value):
        if key == 'charset':
            if value in ['Shift_JIS', 'ShiftJIS', 'SJIS']:
                self.charset = 'Shift_JIS'
            elif value == 'UTF-8':
                self.charset = 'UTF-8'
            else: # default and error
                self.charset = 'Shift_JIS'
        elif key == 'dic':
            filename = get_normalized_path(value, encode=0)
            path = os.path.join(self.aya_dir, filename)
            self.dic_files.append(path)
        elif key == 'msglang':
            pass ## FIXME
        elif key == 'log':
            assert isinstance(value, (str, unicode))
            if isinstance(value, unicode):
                filename = value.encode('utf-8', 'ignore')
            else:
                filename = value
            path = os.path.join(self.aya_dir, filename)
            try:
                f = open(path, 'w')
            except:
                logging.debug('cannnot open {0}'.format(path))
            else:
                if self.logfile:
                    self.logfile.close()
                self.logfile = f
                self.global_namespace.put('log', value)
        elif key == 'iolog':
            pass ## FIXME
        elif key == 'fncdepth':
            pass ## FIXME

    def get_dictionary(self):
        return self.dic

    def get_ghost_dir(self):
        return self.aya_dir

    def get_global_namespace(self):
        return self.global_namespace

    def get_system_functions(self):
        return self.system_functions

    def get_boot_time(self):
        return self.boot_time

    def unload(self):
        func = self.dic.get_function('unload')
        if func:
            func.call([])
        self.global_namespace.save_database()
        self.saori_library.unload()
        if self.logfile is not None:
            self.logfile.close()
        for key in self.filelist.keys():
            self.filelist[key].close()

    # SHIORI API
    def request(self, req_string):
        result = ''
        func = self.dic.get_function('request')
        if func:
            result = func.call([unicode(req_string, self.charset, 'ignore')])
        if result is None:
            result = ''
        return result.encode(self.charset, 'ignore')


class AyaDictionary(object):

    def __init__(self, aya):
        self.aya = aya
        self.functions = {}
        self.global_macro = {}

    def get_function(self, name):
        return self.functions.get(name, None)

    def load(self, f, encrypted):
        all_lines = []
        local_macro = {}
        logical_line = ''
        comment = 0
        while 1:
            if encrypted:
                line = decrypt_readline(f)
            else:
                line = f.readline()
            line = unicode(line, self.aya.charset, 'ignore')
            if not line:
                break # EOF
            if comment:
                end = find_not_quoted(line, '*/')
                if end < 0:
                    continue
                else:
                    line = line[end + 2:]
                    comment = 0
            line = line.strip()
            if not line:
                continue
            if line.endswith('/') and \
               not line.endswith('*/') and not line.endswith('//'):
                logical_line = ''.join((logical_line, line[:-1]))
            else:
                logical_line = ''.join((logical_line, line))
                while 1:
                    start, end = find_comment(logical_line)
                    if start < 0:
                        break
                    if end < 0:
                        comment = 1
                        logical_line = logical_line[:start]
                        break
                    logical_line = ''.join((logical_line[:start], ' ', logical_line[end:]))
                logical_line = logical_line.strip()
                if not logical_line:
                    continue
                buf = logical_line
                # preprosess
                if buf.startswith('#'):
                    buf = buf[1:].strip()
                    for (tag, target) in [('define', local_macro),
                                          ('globaldefine', self.global_macro)]:
                        if buf.startswith(tag):
                            buf = buf[len(tag):].strip()
                            i = 0
                            while i < len(buf):
                                if buf[i] == ' ' or buf[i] == '\t' or \
                                   buf[i:i + 2] == unicode('　', 'utf-8'):
                                    key = buf[:i].strip()
                                    target[key] = buf[i:].strip()
                                    break
                                i += 1
                            break
                    logical_line = '' # reset
                    continue
                for macro in [local_macro, self.global_macro]:
                    logical_line = self.preprosess(macro, logical_line)
                # multi statement
                list_lines = self.split_line(logical_line.strip())
                if list_lines:
                    all_lines.extend(list_lines)
                logical_line = '' # reset
        self.evaluate_lines(all_lines, os.path.split(f.name)[1])

    def split_line(self, line):
        lines = []
        while 1:
            if not line:
                break
            pos = len(line) # not len(line) - 1
            token = ''
            for x in ['{', '}', ';']:
                pos_new = find_not_quoted(line, x)
                if 0 <= pos_new < pos:
                    pos = pos_new
                    token = x
            new = line[:pos].strip()
            line = line[pos + len(token):].strip()
            if new:
                lines.append(new)
            if token not in ['', ';']:
                lines.append(token)
        return lines

    def preprosess(self, macro, line):
        for key, value in macro.items():
            line = line.replace(key, value)
        return line

    __SPECIAL_CHARS = [r']', r'(', r')', r'[', r'+', r'-', r'*', r'/', r'=',
                       r':', r';', r'!', r'{', r'}', r'%', r'&', r'#', r'"',
                       r'<', r'>', r',', r'?']

    def evaluate_lines(self, lines, file_name):
        prev = None
        name = None
        function = []
        option = None
        block_nest = 0
        for i in range(len(lines)):
            line = lines[i]
            if line == '{':
                if name is not None:
                    function.append(line)
                    block_nest += 1
                else:
                    if prev is None:
                        logging.debug(
                            'syntax error in {0}: unbalanced "{" at '
                            'the top of file'.format(file_name))
                    else:
                        logging.debug(
                            'syntax error in {0}: unbalanced "{" at '
                            'the bottom of function "{1}"'.format(file_name, prev))
            elif line == '}':
                if name is not None:
                    block_nest -= 1
                    function.append(line)
                    if block_nest == 0:
                        self.functions[name] = AyaFunction(self, name,
                                                           function, option)
                        # reset
                        prev = name
                        name = None
                        function = []
                        option = None
                else:
                    if prev is None:
                        logging.debug(
                            'syntax error in {0}: unbalanced "}" at '
                            'the top of file'.format(file_name))
                    else:
                        logging.debug(
                            'syntax error in {0}: unbalanced "}" at '
                            'the bottom of function "{1}"'.format(file_name, prev))
                    block_nest = 0
            elif name is None:
                if ':' in line:
                    name, option = [x.strip() for x in line.split(':', 1)]
                else:
                    name = line
                for char in self.__SPECIAL_CHARS:
                    if char in name:
                        logging.debug(
                            'illegal function name "{0}" in {1}'.format(
                                name, file_name))
                function = []
            else:
                if name is not None and block_nest > 0:
                    function.append(line)
                else:
                    logging.debug('syntax error in {0}: {1}'.format(file_name, line))


class AyaFunction(object):

    __TYPE_INT = 10
    __TYPE_FLOAT = 11
    __TYPE_DECISION = 12
    __TYPE_RETURN = 13
    __TYPE_BLOCK = 14
    __TYPE_SUBSTITUTION = 15
    __TYPE_INC = 16
    __TYPE_DEC = 17
    __TYPE_IF = 18
    __TYPE_WHILE = 19
    __TYPE_FOR = 20
    __TYPE_BREAK = 21
    __TYPE_CONTINUE = 22
    __TYPE_SWITCH = 23
    __TYPE_CASE = 24
    __TYPE_STRING_LITERAL = 25
    __TYPE_STRING = 26
    __TYPE_OPERATOR = 27
    __TYPE_STATEMENT = 28
    __TYPE_CONDITION = 29
    __TYPE_SYSTEM_FUNCTION = 30
    __TYPE_FUNCTION = 31
    __TYPE_ARRAY_POINTER = 32
    __TYPE_ARRAY = 33
    __TYPE_VARIABLE_POINTER = 34
    __TYPE_VARIABLE = 35
    __TYPE_TOKEN = 36
    __TYPE_NEW_ARRAY = 37
    __TYPE_FOREACH = 38
    __TYPE_PARALLEL = 39
    __TYPE_FORMULA = 40
    __CODE_NONE = 50
    __CODE_RETURN = 51
    __CODE_BREAK = 52
    __CODE_CONTINUE = 53
    __re_f = re.compile('[-+]?\d+(\.\d*)$')
    __re_d = re.compile('[-+]?\d+$')
    __re_b = re.compile('[-+]?0[bB][01]+$')
    __re_x = re.compile('[-+]?0[xX][\dA-Fa-f]+$')
    __re_if = re.compile('if\s')
    __re_others = re.compile('others\s')
    __re_elseif = re.compile('elseif\s')
    __re_while = re.compile('while\s')
    __re_for = re.compile('for\s')
    __re_foreach = re.compile('foreach\s')
    __re_switch = re.compile('switch\s')
    __re_case = re.compile('case\s')
    __re_when = re.compile('when\s')
    __re_parallel = re.compile('parallel\s')
    __SPECIAL_CHARS = [r']', r'(', r')', r'[', r'+', r'-', r'*', r'/', r'=',
                       r':', r';', r'!', r'{', r'}', r'%', r'&', r'#', r'"',
                       r'<', r'>', r',', r'?']

    def __init__(self, dic, name, lines, option):
        self.dic = dic
        self.name = name
        self.status = self.__CODE_NONE
        self.lines = self.parse(lines)
        if option == 'nonoverlap':
            self.nonoverlap = [[], [], []]
        else:
            self.nonoverlap = None
        if option == 'sequential':
            self.sequential = [[], []]
        else:
            self.sequential = None
        ## FIXME: void, array

    def parse(self, lines):
        result = []
        i = 0
        while i < len(lines):
            line = lines[i]
            if line == '--':
                result.append([self.__TYPE_DECISION, []])
            elif line == 'return':
                result.append([self.__TYPE_RETURN, []])
            elif line == 'break':
                result.append([self.__TYPE_BREAK, []])
            elif line == 'continue':
                result.append([self.__TYPE_CONTINUE, []])
            elif line == '{':
                inner_func = []
                i, inner_func = self.get_block(lines, i)
                result.append([self.__TYPE_BLOCK, self.parse(inner_func)])
            elif self.__re_if.match(line):
                inner_blocks = []
                while 1:
                    current_line = lines[i]
                    if self.__re_if.match(current_line):
                        condition = self.parse_(current_line[2:].strip())
                    elif self.__re_elseif.match(current_line):
                        condition = self.parse_(current_line[6:].strip())
                    else:
                        condition = [self.__TYPE_CONDITION, None]
                    inner_block = []
                    i, inner_block = self.get_block(lines, i + 1)
                    if condition is None:
                        inner_blocks = []
                        break
                    entry = []
                    entry.append(condition)
                    entry.append(self.parse(inner_block))
                    inner_blocks.append(entry)
                    if i + 1 >= len(lines):
                        break
                    next_line = lines[i + 1]
                    if not self.__re_elseif.match(next_line) and \
                       next_line != 'else':
                        break
                    i = i + 1
                if inner_blocks:
                    result.append([self.__TYPE_IF, inner_blocks])
            elif self.__re_while.match(line):
                condition = self.parse_(line[5:].strip())
                inner_block = []
                i, inner_block = self.get_block(lines, i + 1)
                result.append([self.__TYPE_WHILE,
                               [condition, self.parse(inner_block)]])
            elif self.__re_for.match(line):
                init = self.parse([line[3:].strip()]) ## FIXME(?)
                condition = self.parse_(lines[i + 1])
                reset = self.parse([lines[i + 2]]) ## FIXME(?)
                inner_block = []                
                i, inner_block = self.get_block(lines, i + 3)
                if condition is not None:
                    result.append([self.__TYPE_FOR,
                                   [[init, condition, reset],
                                    self.parse(inner_block)]])
            elif self.__re_foreach.match(line):
                name = line[7:].strip()
                var = lines[i + 1]
                i, inner_block = self.get_block(lines, i + 2)
                result.append([self.__TYPE_FOREACH,
                               [[name, var], self.parse(inner_block)]])
            elif self.__re_switch.match(line):
                index = self.parse_(line[6:].strip())
                inner_block = []
                i, inner_block = self.get_block(lines, i + 1)
                result.append([self.__TYPE_SWITCH,
                               [index, self.parse(inner_block)]])
            elif self.__re_case.match(line):
                left = self.parse_(line[4:].strip())
                i, block = self.get_block(lines, i + 1)
                inner_blocks = []
                j = 0
                while 1:
                    current_line = block[j]
                    if self.__re_when.match(current_line):
                        right = current_line[4:].strip()
                    else: # 'others'
                        right = None
                    inner_block = []
                    j, inner_block = self.get_block(block, j + 1)
                    if right is not None:
                        argument = AyaArgument(right)
                        while argument.has_more_tokens():
                            entry = []
                            right = argument.next_token()
                            tokens = AyaStatement(right).tokens
                            if tokens[0] in ['-', '+']:
                                value_min = self.parse_statement([tokens.pop(0),
                                                            tokens.pop(0)]) ## FIXME: parse_
                            else:
                                value_min = self.parse_statement([tokens.pop(0)]) ## FIXME: parse_
                            value_max = value_min
                            if tokens:
                                if tokens[0] != '-':
                                    logging.debug(
                                        'syntax error in function '
                                        '"{0}": when {1}'.format(self.name, right))
                                    continue
                                else:
                                    tokens.pop(0)
                                if len(tokens) > 2 or \
                                   (len(tokens) == 2 and \
                                    tokens[0] not in ['-', '+']):
                                    logging.debug(
                                        'syntax error in function '
                                        '"{0}": when {1}'.format(self.name, right))
                                    continue
                                else:
                                    value_max = self.parse_statement(tokens) ## FIXME: parse_
                            entry.append([value_min, value_max])
                            entry.append(self.parse(inner_block))
                            inner_blocks.append(entry)
                    else:
                        entry = []
                        entry.append(right)
                        entry.append(self.parse(inner_block))
                        inner_blocks.append(entry)
                    if j + 1 == len(block):
                        break
                    next_line = block[j + 1]
                    if not self.__re_when.match(next_line) and \
                       next_line != 'others':
                        break
                    j += 1
                result.append([self.__TYPE_CASE, [left, inner_blocks]])
            elif self.__re_parallel.match(line): ## FIXME
                pass
            else:
                result.append([self.__TYPE_FORMULA, self.parse_(line)]) ## FIXME
            i += 1
        result.append([self.__TYPE_DECISION, []])
        return result

    def find_close_(self, token_open, tokens, position):
        nest = 0
        current = position
        if token_open == '[':
            token_close = ']'
        elif token_open == '(':
            token_close = ')'
        while tokens[current:].count(token_close) > 0:
            pos_new = tokens.index(token_close, current)
            if pos_new == 0:
                break
            nest = tokens[position:pos_new].count(token_open) - tokens[position:pos_new].count(token_close) # - 1
            if nest > 0:
                current = pos_new + 1
            else:
                current = pos_new
                break
        return current

    def iter_position(self, tokens, ope_list, reverse=0):
        position = 0
        len_tokens = len(tokens)
        while 1:
            new_pos = len_tokens
            if reverse:
                tokens.reverse()
            for ope in ope_list:
                if ope in tokens[position:]:
                    temp_pos = tokens.index(ope, position)
                    new_pos = min(new_pos, temp_pos)
            if reverse:
                tokens.reverse()
            position = new_pos
            if position >= len_tokens:
                break
            else:
                if reverse:
                    yield len_tokens - position
                else:
                    yield position
            position += 1

    def find_position(self, tokens, ope_list, position, reverse=0):
        len_tokens = len(tokens)
        new_pos = len_tokens
        if reverse:
            tokens.reverse()
        for ope in ope_list:
            if ope in tokens[position:]:
                temp_pos = tokens.index(ope, position)
                new_pos = min(new_pos, temp_pos)
        if reverse:
            tokens.reverse()
        position = new_pos
        if position >= len_tokens:
            return -1
        else:
            if reverse:
                return len_tokens - 1 - position
            else:
                return position

    def get_left_(self, tokens, position):
        position -= 1
        left = tokens.pop(position)
        if isinstance(left, (str, unicode)): # not processed
            left = self.new_parse([left])
        return left

    def get_right_(self, tokens, position):
        right = tokens.pop(position)
        if isinstance(right, (str, unicode)): # not processed
            if right in ['-', '+']:
                right = self.new_parse([right, tokens.pop(position)])
            else:
                right = self.new_parse([right])
        return right

    def new_parse(self, tokens): ## FIXME
        if len(tokens) == 0: ## FIXME
            return []
        position = self.find_position(tokens, ['(', '['], 0)
        while position >= 0:
            pos_end = self.find_close_(tokens[position], tokens, position + 1)
            temp = []
            token_open = tokens.pop(position) # open
            for i in range(pos_end - position - 1):
                temp.append(tokens.pop(position))
            tokens.pop(position) # close
            if token_open == '[': # array
                name = tokens.pop(position - 1)
                tokens.insert(position - 1, [self.__TYPE_ARRAY,
                                             [name, [self.__TYPE_FORMULA, self.new_parse(temp)]]])
            else:
                ope_list = ['!', '++', '--', '*', '/', '%', '+', '-', '&',
                            '==', '!=', '>=', '<=', '>', '<', '_in_', '!_in_',
                            '&&', '||', '=', ':=',
                            '+=', '-=', '*=', '/=', '%=',
                            '+:=', '-:=', '*:=', '/:=', '%:=', ',=', ','] ## FIXME
                if position == 0 or \
                   tokens[position - 1] in ope_list: # FORMULA
                    tokens.insert(position, [self.__TYPE_FORMULA,
                                             self.new_parse(temp)])
                else: # should be function
                    name = tokens.pop(position - 1)
                    if self.dic.aya.get_system_functions().exists(name):
                        
                        tokens.insert(position - 1, [self.__TYPE_SYSTEM_FUNCTION,
                                                     [name,
                                                      [self.new_parse(temp)]]]) ## CHECK: arguments
                        
                    else:
                        tokens.insert(position - 1, [self.__TYPE_FUNCTION,
                                                     [name,
                                                      [self.new_parse(temp)]]]) ## CHECK: arguments
            position = self.find_position(tokens, ['(', '['], 0)
        for position in self.iter_position(tokens, ['!']):
            tokens.pop(position)
            right = self.get_right_(tokens, position)
            tokens.insert(position, [self.__TYPE_CONDITION,
                                     [None, [self.__TYPE_OPERATOR, '!'], right]])
        for position in self.iter_position(tokens, ['++', '--']):
            if tokens.pop(position) == '++':
                type_ = self.__TYPE_INC
            else:
                type_ = self.__TYPE_DEC
            var = tokens.pop(position - 1)
            if isinstance(var, (str, unicode)): # not processed
                right = self.new_parse([var]) ## FIXME
            tokens.insert(position - 1, [type_, var])
        position = self.find_position(tokens, ['*', '/', '%'], 0, reverse=1)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            left = self.get_left_(tokens, position)
            tokens.insert(position - 1, [self.__TYPE_STATEMENT, left, ope, right])
            position = self.find_position(tokens, ['*', '/', '%'], 0, reverse=1)
        position = 0
        position = self.find_position(tokens, ['+', '-'], position)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            ope_list = ['!_in_', '_in_',
                        '+:=', '-:=', '*:=', '/:=', '%:=',
                        ':=', '+=', '-=', '*=', '/=', '%=', '<=',
                        '>=', '==', '!=', '&&', '||', ',=', '++', '--',
                        ',', '=', '!', '+', '-', '/', '*', '%', '&']
            if position == 0 or tokens[position - 1] in ope_list:
                left = [self.__TYPE_INT, 0]
                tokens.insert(position, [self.__TYPE_STATEMENT, left, ope, right])
            else:
                left = self.get_left_(tokens, position)
                tokens.insert(position - 1, [self.__TYPE_STATEMENT, left, ope, right])
            position = self.find_position(tokens, ['+', '-'], position)
        for position in self.iter_position(tokens, ['&']):
            type_ = tokens[position + 1][0]
            var_ = tokens[position + 1][1]
            if type_ == self.__TYPE_ARRAY:
                tokens.insert(position, [self.__TYPE_ARRAY_POINTER, var_])
            elif type_ == self.__TYPE_VARIABLE:
                tokens.insert(position, [self.__TYPE_VARIABLE_POINTER, var_])
            elif type_ == self.__TYPE_TOKEN:
                tokens.insert(position, [self.__TYPE_VARIABLE_POINTER,
                                         [var_, None]])
            else:
                logging.debug(
                    'syntax error in function "{0}": '
                    'illegal argument "{1}"'.format(self.name, tokens))
        position = self.find_position(tokens, ['==', '!=', '>=', '<=', '>', '<', '_in_', '!_in_'], 0, reverse=1)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            left = self.get_left_(tokens, position)
            tokens.insert(position - 1, [self.__TYPE_CONDITION, [left, ope, right]])
            position = self.find_position(tokens, ['==', '!=', '>=', '<=', '>', '<', '_in_', '!_in_'], 0, reverse=1)
        position = self.find_position(tokens, ['&&'], 0, reverse=1)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            left = self.get_left_(tokens, position)
            tokens.insert(position - 1, [self.__TYPE_CONDITION, [left, ope, right]])
            position = self.find_position(tokens, ['&&'], 0, reverse=1)
        position = self.find_position(tokens, ['||'], 0, reverse=1)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            left = self.get_left_(tokens, position)
            tokens.insert(position - 1, [self.__TYPE_CONDITION, [left, ope, right]])
            position = self.find_position(tokens, ['||'], 0, reverse=1)
        position = self.find_position(tokens, ['=', ':='], 0, reverse=1)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            left = self.get_left_(tokens, position)
            tokens.insert(position - 1, [self.__TYPE_SUBSTITUTION, [left, ope, right]])
            position = self.find_position(tokens, ['=', ':='], 0, reverse=1)
        position = self.find_position(tokens, ['+=', '-=', '*=', '/=', '%=', '+:=', '-:=', '*:=', '/:=', '%:=', ',='], 0, reverse=1)
        while position >= 0:
            ope = [self.__TYPE_OPERATOR, tokens.pop(position)]
            right = self.get_right_(tokens, position)
            left = self.get_left_(tokens, position)
            tokens.insert(position - 1, [self.__TYPE_SUBSTITUTION, [left, ope, right]])
            position = self.find_position(tokens, ['+=', '-=', '*=', '/=', '%=', '+:=', '-:=', '*:=', '/:=', '%:=', ',='], 0, reverse=1)
        temp = []
        position = self.find_position(tokens, [','], 0, reverse=0)
        while position >= 0:
            tokens.pop(position)
            temp.append(self.get_right_(tokens, position))
            if position > 0:
                temp.insert(-1, self.get_left_(tokens, position))
            position = self.find_position(tokens, [','], 0, reverse=0)
        if temp:
            tokens.insert(position, [self.__TYPE_NEW_ARRAY, temp])
        for i in range(len(tokens)):
            if isinstance(tokens[i], (str, unicode)):
                token = tokens.pop(i)
                tokens.insert(i, self.parse_token(token))
        return tokens[0] ## FIXME

    def parse_(self, line): ## FIXME
        statement = AyaStatement(line)
        return self.new_parse(statement.tokens)

    def parse_statement(self, statement_tokens):
        n_tokens = len(statement_tokens)
        statement = []
        if n_tokens == 1:
            statement = [self.__TYPE_STATEMENT,
                         self.parse_token(statement_tokens[0])]
        elif statement_tokens[0] == '+':
            statement = self.parse_statement(statement_tokens[1:])
        elif statement_tokens[0] == '-':
            tokens = ['0']
            tokens.extend(statement_tokens)
            statement = self.parse_statement(tokens)
        else:
            ope_index = None
            for ope in ['+', '-']:
                if ope in statement_tokens:
                    new_index = statement_tokens.index(ope)
                    if ope_index is None or new_index < ope_index:
                        ope_index = new_index
            if ope_index is None:
                statement_tokens.reverse()
                try:
                    for ope in ['*', '/', '%']:
                        if ope in statement_tokens:
                            new_index = statement_tokens.index(ope)
                            if ope_index is None or new_index < ope_index:
                                ope_index = new_index
                    if ope_index is not None:
                        ope_index = -1 - ope_index
                finally:
                    statement_tokens.reverse()
            if ope_index in [None, -1, 0, n_tokens - 1]:
                return None
            else:
                ope = [self.__TYPE_OPERATOR, statement_tokens[ope_index]]
                if len(statement_tokens[:ope_index]) == 1:
                    if statement_tokens[0].startswith('('):
                        tokens = AyaStatement(statement_tokens[0][1:-1]).tokens
                        left = self.parse_statement(tokens)
                    else:
                        left = self.parse_token(
                            statement_tokens[:ope_index][0])
                else:
                    left = self.parse_statement(statement_tokens[:ope_index])
                if len(statement_tokens[ope_index + 1:]) == 1:
                    if statement_tokens[-1].startswith('('):
                        tokens = AyaStatement(
                            statement_tokens[ope_index + 1][1:-1]).tokens
                        right = self.parse_statement(tokens)
                    else:
                        right = self.parse_token(
                            statement_tokens[ope_index + 1:][0])
                else:
                    right = self.parse_statement(
                        statement_tokens[ope_index + 1:])
                statement = [self.__TYPE_STATEMENT, left, ope, right]
        return statement

    def parse_condition(self, condition_tokens):
        n_tokens = len(condition_tokens)
        condition = None
        ope_index = None
        condition_tokens.reverse()
        try:
            for ope in ['&&', '||']:
                if ope in condition_tokens:
                    new_index = condition_tokens.index(ope)
                    if ope_index is None or new_index < ope_index:
                        ope_index = new_index
            if ope_index is not None:
                ope_index = -1 - ope_index
        finally:
            condition_tokens.reverse()
        if ope_index is None:
            for ope in ['==', '!=', '>', '<', '>=', '<=', '_in_', '!_in_']:
                if ope in condition_tokens:
                    new_index = condition_tokens.index(ope)
                    if ope_index is None or new_index < ope_index:
                        ope_index = new_index
            if ope_index in [None, -1, 0, n_tokens - 1]:
                logging.debug(
                    'syntax error in function "{0}": '
                    'illegal condition "{1}"'.format(self.name, ' '.join(condition_tokens)))
                return None
            ope = [self.__TYPE_OPERATOR, condition_tokens[ope_index]]
            if len(condition_tokens[:ope_index]) == 1:
                left = self.parse_statement([condition_tokens[:ope_index][0]])
            else:
                left = self.parse_statement(condition_tokens[:ope_index])
            if len(condition_tokens[ope_index + 1:]) == 1:
                right = self.parse_statement([condition_tokens[ope_index + 1:][0]])
            else:
                right = self.parse_statement(condition_tokens[ope_index + 1:])
            condition = [self.__TYPE_CONDITION, [left, ope, right]]
        else:
            ope = [self.__TYPE_OPERATOR, condition_tokens[ope_index]]
            left = self.parse_condition(condition_tokens[:ope_index])
            right = self.parse_condition(condition_tokens[ope_index + 1:])
            if left is not None and right is not None:
                condition = [self.__TYPE_CONDITION, [left, ope, right]]
        return condition

    def parse_argument(self, args): ## FIXME
        argument = AyaArgument(args)
        arguments = []
        while argument.has_more_tokens():
            token = argument.next_token()
            if token.startswith('&'):
                result = self.parse_token(token[1:])
                if result[0] == self.__TYPE_ARRAY:
                    arguments.append([self.__TYPE_ARRAY_POINTER, result[1]])
                elif result[0] == self.__TYPE_VARIABLE:
                    arguments.append([self.__TYPE_VARIABLE_POINTER, result[1]])
                elif result[0] == self.__TYPE_TOKEN:
                    arguments.append([self.__TYPE_VARIABLE_POINTER,
                                      [result[1], None]])
                else:
                    logging.debug(
                        'syntax error in function "{0}": '
                        'illegal argument "{1}"'.format(self.name, token))
            elif token.startswith('('):
                if not token.endswith(')'):
                    logging.debug(
                        'syntax error in function "{0}": '
                        'unbalanced "(" in the string({1})'.format(self.name, token))
                    return None
                else:
                    statement = AyaStatement(token[1:-1])
                    arguments.append(self.parse_statement(statement.tokens))
            else:
                arguments.append(self.parse_statement([token]))
        return arguments

    def parse_token(self, token):
        result = []
        if self.__re_f.match(token):
            result = [self.__TYPE_FLOAT, token]
        elif self.__re_d.match(token):
            result = [self.__TYPE_INT, token]
        elif token.startswith('"'):
            text = token[1:]
            if text.endswith('"'):
                text = text[:-1]
            if text.count('"') > 0:
                logging.debug(
                    'syntax error in function "{0}": '
                    '\'"\' in string "{1}"'.format(self.name, text))
            if '%' not in text:
                result = [self.__TYPE_STRING_LITERAL, text]
            else:
                result = [self.__TYPE_STRING, text]
        elif token.startswith("'"):
            text = token[1:]
            if text.endswith("'"):
                text = text[:-1]
            if text.count("'") > 0:
                logging.debug(
                    'syntax error in function "{0}": '
                    "\"'\" in string \"{1}\"".format(self.name, text))
            result = [self.__TYPE_STRING_LITERAL, text]
        else:
            pos_parenthesis_open = token.find('(')
            pos_block_open = token.find('[')
            if pos_parenthesis_open == 0 and find_not_quoted(token, ',') != -1: ## FIXME: Array
                if not token.endswith(')'):
                    logging.debug(
                        'syntax error: unbalnced "(" in "{0}"'.format(token))
                else:
                    result = [self.__TYPE_NEW_ARRAY,
                              self.parse_argument(token[1:-1])] ## FIXME
            elif pos_parenthesis_open != -1 and \
               (pos_block_open == -1 or \
                pos_parenthesis_open < pos_block_open): # function
                if not token.endswith(')'):
                    logging.debug(
                        'syntax error: unbalnced "(" in "{0}"'.format(token))
                else:
                    func_name = token[:pos_parenthesis_open]
                    arguments = self.parse_argument(
                        token[pos_parenthesis_open + 1:-1])
                    for char in self.__SPECIAL_CHARS:
                        if char in func_name:
                            logging.debug(
                                'illegal character "{0}" in '
                                'the name of function "{1}"'.format(char, token))
                            break
                    else:
                        if self.dic.aya.get_system_functions().exists(
                            func_name):
                            if func_name == 'LOGGING':
                                result = [self.__TYPE_SYSTEM_FUNCTION,
                                          [func_name, arguments,
                                           token[pos_parenthesis_open + 1:-1]]]
                            else:
                                result = [self.__TYPE_SYSTEM_FUNCTION,
                                          [func_name, arguments]]
                        else:
                            result = [self.__TYPE_FUNCTION,
                                      [func_name, arguments]]
            elif pos_block_open != -1: # array
                if not token.endswith(']'):
                    logging.debug(
                        'syntax error: unbalnced "[" in "{0}"'.format(token))
                else:
                    array_name = token[:pos_block_open]
                    index = self.parse_token(token[pos_block_open + 1:-1])
                    for char in self.__SPECIAL_CHARS:
                        if char in array_name:
                            logging.debug(
                                'illegal character "{0}" in '
                                'the name of array "{1}"'.format(char, token))
                            break
                    else:
                        result = [self.__TYPE_ARRAY, [array_name, index]]
            else: # variable or function
                for char in self.__SPECIAL_CHARS:
                    if char in token:
                        logging.debug(
                            'syntax error in function "{0}": '
                            'illegal character "{1}" in the name of '
                            'function/variable "{2}"'.format(self.name, char, token))
                        break
                else:
                    result = [self.__TYPE_TOKEN, token]
        return result

    def call(self, argv=None):
        namespace = AyaNamespace(self.dic.aya)
        _argv = []
        if not argv:
            namespace.put('_argc', 0)
        else:
            namespace.put('_argc', len(argv))
            for i in range(len(argv)):
                if isinstance(argv[i], dict):
                    _argv.append(argv[i]['value'])
                else:
                    _argv.append(argv[i])
        namespace.put('_argv', _argv)
        self.status = self.__CODE_NONE
        result = self.evaluate(namespace, self.lines, -1, 0) ## FIXME
        if argv:
            for i in range(len(argv)):
                if isinstance(argv[i], dict):
                    value = _argv[i]
                    name = argv[i]['name']
                    namespace = argv[i]['namespace']
                    index = argv[i]['index']
                    namespace.put(name, value, index)
        return result

    def evaluate(self, namespace, lines, index_to_return, is_inner_block, is_block=1, connect=0): ## FIXME
        result = []
        alternatives = []
        for line in lines:
            if not line:
                continue
            if line[0] in [self.__TYPE_DECISION, self.__TYPE_RETURN,
                           self.__TYPE_BREAK, self.__TYPE_CONTINUE] or \
               self.status in [self.__CODE_RETURN, self.__CODE_BREAK,
                               self.__CODE_CONTINUE]:
                if alternatives:
                    if is_inner_block:
                        if index_to_return < 0:
                            result.append(random.choice(alternatives))
                        elif index_to_return <= len(alternatives) - 1:
                            result.append(alternatives[index_to_return])
                        else: # out of range
                            result.append('')
                    else:
                        result.append(alternatives)
                    alternatives = []
                if line[0] == self.__TYPE_RETURN or \
                   self.status == self.__CODE_RETURN:
                    self.status = self.__CODE_RETURN
                    break
                elif line[0] == self.__TYPE_BREAK or \
                     self.status == self.__CODE_BREAK:
                    self.status = self.__CODE_BREAK
                    break
                elif line[0] == self.__TYPE_CONTINUE or \
                     self.status == self.__CODE_CONTINUE:
                    self.status = self.__CODE_CONTINUE
                    break
            elif line[0] == self.__TYPE_BLOCK:
                inner_func = line[1]
                local_namespace = AyaNamespace(self.dic.aya, namespace)
                result_of_inner_func = self.evaluate(local_namespace,
                                                     inner_func, -1, 1) ## FIXME
                if result_of_inner_func is not None:
                    alternatives.append(result_of_inner_func)
            elif line[0] == self.__TYPE_SUBSTITUTION:
                left, ope, right = line[1]
                ope = ope[1]
                if ope in [':=', '+:=', '-:=', '*:=', '/:=', '%:=']:
                    type_float = 1
                else:
                    type_float = 0
                right_result = self.evaluate(namespace, [right], -1 , 1, 0, 1) ## FIXME
                if ope not in ['=', ':=']:
                    left_result = self.evaluate_token(namespace, left) 
                    right_result = self.operation(left_result, ope[0],
                                                  right_result, type_float)
                    ope = ope[1:]
                result_of_substitution =  self.substitute(namespace, left, ope, right_result)
                if connect: ## FIXME
                    alternatives.append(result_of_substitution)
            elif line[0] == self.__TYPE_INC or \
                 line[0] == self.__TYPE_DEC: # ++/--
                if line[0] == self.__TYPE_INC:
                    ope = '++'
                elif line[0] == self.__TYPE_DEC:
                    ope = '--'
                else:
                    return None # should not reach here
                var_name = line[1]
                if var_name.startswith('_'):
                    target_namespace = namespace
                else:
                    target_namespace = self.dic.aya.get_global_namespace()
                value = self.evaluate(namespace, [[self.__TYPE_TOKEN, var_name]], -1, 1, 0) ## FIXME
                index = None
                if isinstance(value, (int, float)):
                    if ope == '++':
                        target_namespace.put(var_name, int(value) + 1, index)
                    elif ope == '--':
                        target_namespace.put(var_name, int(value) - 1, index)
                    else:
                        return None # should not reach here
                else:
                    logging.debug(
                        'illegal increment/decrement:'
                        'type of variable {0} is not number'.format(var_name))
            elif line[0] == self.__TYPE_IF:
                inner_blocks = line[1]
                n_blocks = len(inner_blocks)
                for j in range(n_blocks):
                    entry = inner_blocks[j]
                    condition = entry[0]
                    inner_block = entry[1]
                    if self.evaluate(namespace, [condition], -1, 1, 0): ## FIXME
                        local_namespace = AyaNamespace(self.dic.aya, namespace)
                        result_of_inner_block = self.evaluate(local_namespace,
                                                              inner_block,
                                                              -1, 1) ## FIXME
                        if result_of_inner_block is not None:
                            alternatives.append(result_of_inner_block)
                        break
            elif line[0] == self.__TYPE_WHILE:
                condition = line[1][0]
                inner_block = line[1][1]
                assert condition[0] == self.__TYPE_CONDITION
                while self.evaluate_condition(namespace, condition):
                    local_namespace = AyaNamespace(self.dic.aya, namespace)
                    result_of_inner_block = self.evaluate(local_namespace,
                                                          inner_block, -1, 1) ## FIXME
                    if result_of_inner_block is not None:
                        alternatives.append(result_of_inner_block)
                    if self.status == self.__CODE_RETURN:
                        break
                    if self.status == self.__CODE_BREAK:
                        self.status = self.__CODE_NONE
                        break
                    if self.status == self.__CODE_CONTINUE:
                        self.status = self.__CODE_NONE
            elif line[0] == self.__TYPE_FOR:
                init = line[1][0][0]
                condition = line[1][0][1]
                reset = line[1][0][2]
                inner_block = line[1][1]
                self.evaluate(namespace, init, -1, 1, 0) ## FIXME
                assert condition[0] == self.__TYPE_CONDITION
                while self.evaluate_condition(namespace, condition):
                    local_namespace = AyaNamespace(self.dic.aya, namespace)
                    result_of_inner_block = self.evaluate(local_namespace,
                                                          inner_block, -1, 1) ## FIXME
                    if result_of_inner_block is not None:
                        alternatives.append(result_of_inner_block)
                    if self.status == self.__CODE_RETURN:
                        break
                    if self.status == self.__CODE_BREAK:
                        self.status = self.__CODE_NONE
                        break
                    if self.status == self.__CODE_CONTINUE:
                        self.status = self.__CODE_NONE
                    self.evaluate(namespace, reset, -1, 1, 0) ## FIXME
            elif line[0] == self.__TYPE_SWITCH:
                index = self.evaluate_token(namespace, line[1][0])
                inner_block = line[1][1]
                try:
                    index = int(index)
                except ValueError:
                    index = 0
                local_namespace = AyaNamespace(self.dic.aya, namespace)
                result_of_inner_block = self.evaluate(local_namespace,
                                                      inner_block, index, 1) ## FIXME
                if result_of_inner_block is not None:
                    alternatives.append(result_of_inner_block)
            elif line[0] == self.__TYPE_CASE:
                left = self.evaluate_token(namespace, line[1][0])
                inner_blocks = line[1][1]
                n_blocks = len(inner_blocks)
                default_result = None
                for j in range(n_blocks):
                    entry = inner_blocks[j]
                    inner_block = entry[1]
                    local_namespace = AyaNamespace(self.dic.aya, namespace)
                    if entry[0] is not None:
                        value_min, value_max = entry[0]
                        value_min = self.evaluate_statement(namespace, value_min, 1)
                        value_max = self.evaluate_statement(namespace, value_max, 1)
                        if value_min <= left <= value_max:
                            result_of_inner_block = self.evaluate(
                                local_namespace, inner_block, -1, 1) ## FIXME
                            if result_of_inner_block is not None:
                                alternatives.append(result_of_inner_block)
                                break
                    else:
                        default_result = self.evaluate(local_namespace,
                                                       inner_block, -1, 1) ## FIXME
                else:
                    if default_result:
                        alternatives.append(default_result)
            elif line[0] == self.__TYPE_STATEMENT:
                result_of_func = self.evaluate_statement(namespace, line, 0)
                if result_of_func is not None:
                    alternatives.append(result_of_func)
            elif line[0] == self.__TYPE_CONDITION:
                condition = line
                if condition is None or \
                   self.evaluate_condition(namespace, condition):
                    alternatives.append(1)
                else:
                    alternatives.append(0)
            elif line[0] == self.__TYPE_FORMULA:
                result_of_formula = self.evaluate(namespace, [line[1]], -1, 1, 0, connect=connect) ## FIXME
                if result_of_formula is not None:
                    alternatives.append(result_of_formula)
            elif line[0] == self.__TYPE_NEW_ARRAY:
                temp = []
                for item in line[1]:
                    member_of_array = self.evaluate(namespace, [item], -1, 1, 0, 1) ## FIXME
                    temp.append(member_of_array)
                alternatives.append(temp)
            elif line[0] == self.__TYPE_ARRAY:
                system_functions = self.dic.aya.get_system_functions()
                if isinstance(line[1][0], list) or \
                   system_functions.exists(line[1][0]):
                    if isinstance(line[1][0], list): ## FIXME
                        array = self.evaluate(namespace, [line[1][0]], -1, 1, 0) ## FIXME
                    else:
                        array = self.evaluate(namespace, [[self.__TYPE_SYSTEM_FUNCTION, [line[1][0], []]]], -1, 1, 0)
                    if isinstance(array, (str, unicode)):
                        temp = self.evaluate(namespace, [line[1][1]], -1, 1, 0) ## FIXME
                        if isinstance(temp, list):
                            assert len(temp) == 2
                            index, delimiter = temp
                        else:
                            index = temp
                            delimiter = ','
                        assert isinstance(index, int)
                        assert isinstance(delimiter, (str, unicode))
                        result_of_array = array.split(delimiter)[index]
                        alternatives.append(result_of_array)
                    elif isinstance(array, list):
                        index = self.evaluate(namespace, [line[1][1]], -1, 1, 0) ## FIXME
                        assert isinstance(index, int)
                        result_of_array = array[index]
                        alternatives.append(result_of_array)
                    else:
                        logging.debug(
                            'Oops: {0} {1}'.format(repr(array), line[1][1]))
                else:
                    var_name = line[1][0]
                    if var_name.startswith('_'):
                        target_namespace = namespace
                    else:
                        target_namespace = self.dic.aya.get_global_namespace()
                    temp = self.evaluate(namespace, [line[1][1]], -1, 1, 0) ## FIXME
                    if isinstance(temp, list) and len(temp) > 1:
                        assert len(temp) == 2
                        index, delimiter = temp
                        assert isinstance(delimiter, (str, unicode))
                    else:
                        index = temp
                        delimiter = None
                    ##assert isinstance(index, int)
                    if isinstance(index, int) and \
                            target_namespace.exists(var_name):
                        if delimiter is not None:
                            array = target_namespace.get(var_name)
                            temp = array.split(delimiter)
                            if len(temp) > index:
                                result_of_array = array.split(delimiter)[index]
                            else:
                                result_of_array = ''
                        else:
                            result_of_array = target_namespace.get(var_name, index)
                        alternatives.append(result_of_array)
                    else:
                        result_of_array = ''
                        alternatives.append(result_of_array)
            elif line[0] in [self.__TYPE_INT, self.__TYPE_TOKEN,
                             self.__TYPE_SYSTEM_FUNCTION,
                             self.__TYPE_STRING_LITERAL,
                             self.__TYPE_FUNCTION,
                             self.__TYPE_STRING,
                             self.__TYPE_ARRAY_POINTER,
                             self.__TYPE_VARIABLE_POINTER]:
                result_of_eval = self.evaluate_token(namespace, line)
                if result_of_eval is not None:
                    alternatives.append(result_of_eval)
            elif line[0] == self.__TYPE_FOREACH:
                var_name, temp = line[1][0]
                if var_name.startswith('_'):
                    target_namespace = namespace
                else:
                    target_namespace = self.dic.aya.get_global_namespace()
                array = target_namespace.get_array(var_name)
                for item in array:
                    if temp.startswith('_'):
                        target_namespace = namespace
                    else:
                        target_namespace = self.dic.aya.get_global_namespace()
                    target_namespace.put(temp, item)
                    result_of_block = self.evaluate(namespace, line[1][1], -1, 1) ## FIXME
                    if self.status == self.__CODE_RETURN:
                        break
                    if self.status == self.__CODE_BREAK:
                        self.status = self.__CODE_NONE
                        break
                    if self.status == self.__CODE_CONTINUE:
                        self.status = self.__CODE_NONE
            else: ## FIXME
                result_of_eval = self.evaluate_token(namespace, line)
                if result_of_eval is not None:
                    alternatives.append(result_of_eval)
        if not is_inner_block:
            if self.sequential is not None:
                list_ = []
                for alt in result:
                    list_.append(len(alt))
                if self.sequential[0] != list_:
                    self.sequential[0] = list_
                    self.sequential[1] = [0] * len(result)
                else:
                    for index in range(len(result)):
                        current = self.sequential[1][index]
                        if current < len(result[index]) - 1:
                            self.sequential[1][index] = current + 1
                            break
                        else:
                            self.sequential[1][index] = 0
            if self.nonoverlap is not None:
                list_ = []
                for alt in result:
                    list_.append(len(alt))
                if self.nonoverlap[0] != list_:
                    self.nonoverlap[0] = list_
                    self.nonoverlap[2] = []
                if not self.nonoverlap[2]:
                    self.nonoverlap[2].append([0] * len(result))
                    while 1:
                        new = []
                        new.extend(self.nonoverlap[2][-1])
                        for index in range(len(result)):
                            if new[index] < len(result[index]) - 1:
                                new[index] += 1
                                self.nonoverlap[2].append(new)
                                break
                            else:
                                new[index] = 0
                        else:
                            break
                next = random.choice(range(len(self.nonoverlap[2])))
                self.nonoverlap[1] = self.nonoverlap[2][next]
                del self.nonoverlap[2][next]
            for index in range(len(result)):
                if self.sequential is not None:
                    result[index] = result[index][self.sequential[1][index]]
                elif self.nonoverlap is not None:
                    result[index] = result[index][self.nonoverlap[1][index]]
                else:
                    result[index] = random.choice(result[index])
        if not result:
            if not is_block and alternatives: ## FIXME
                return alternatives[-1] ## FIXME
            return None
        elif len(result) == 1:
            return result[0]
        else:
            return ''.join([unicode(s) for s in result])

    def substitute(self, namespace, left, ope, right):
        if left[0] is not self.__TYPE_ARRAY:
            var_name = left[1]
        else:
            var_name = left[1][0]
        if var_name.startswith('_'):
            target_namespace = namespace
        else:
            target_namespace = self.dic.aya.get_global_namespace()
        if left[0] is not self.__TYPE_ARRAY:
            target_namespace.put(var_name, right)
        else:
            index = self.evaluate(namespace, [left[1][1]], -1, 1, 0)
            try:
                index = int(index)
            except ValueError:
                logging.debug('Could not convert {0} to an integer'.format(index))
            else:
                if ope == '=':
                    elem = right
                elif ope == ':=':
                    if isinstance(right, int):
                        elem = float(right)
                    else:
                        elem = right
                else:
                    return None # should not reach here
                target_namespace.put(var_name, elem, index)
        return right

    def evaluate_token(self, namespace, token):
        result = '' # default
        if token[0] == self.__TYPE_TOKEN:
            if self.__re_b.match(token[1]):
                pos = self.__re_d.search(token[1]).start()
                result = int(token[1][pos:], 2)
            elif self.__re_x.match(token[1]):
                result = int(token[1], 16)
            else:
                func = self.dic.get_function(token[1])
                system_functions = self.dic.aya.get_system_functions()
                if func:
                    result = func.call()
                elif system_functions.exists(token[1]):
                    result = system_functions.call(namespace, token[1], [])
                else:
                    if token[1].startswith('_'):
                        target_namespace = namespace
                    else:
                        target_namespace = self.dic.aya.get_global_namespace()
                    if target_namespace.exists(token[1]):
                        result = target_namespace.get(token[1])
        elif token[0] == self.__TYPE_STRING_LITERAL:
            result = token[1]
        elif token[0] == self.__TYPE_STRING:
            result = self.evaluate_string(namespace, token[1])
        elif token[0] == self.__TYPE_INT:
            result = int(token[1])
        elif token[0] == self.__TYPE_FLOAT:
            result = float(token[1])
        elif token[0] == self.__TYPE_SYSTEM_FUNCTION:
            system_functions = self.dic.aya.get_system_functions()
            func_name = token[1][0]
            ##assert system_functions.exists(func_name)
            ##raise Exception(''.join(('function ', func_name, ' not found.')))
            arguments = self.evaluate(namespace, token[1][1], -1, 1, 0, 1) ## FIXME
            if not isinstance(arguments, list): ## FIXME
                arguments = [arguments]
            if func_name == 'LOGGING':
                arguments.insert(0, token[1][2])
                arguments.insert(0, self.name)
                arguments.insert(0, self.dic.aya.logfile)
                result = system_functions.call(namespace, func_name, arguments)
            else:
                result = system_functions.call(namespace, func_name, arguments)
        elif token[0] == self.__TYPE_FUNCTION:
            func_name = token[1][0]
            func = self.dic.get_function(func_name)
            ##assert func is not None:
            ##raise Exception(''.join(('function ', func_name, ' not found.')))
            arguments = self.evaluate_argument(namespace, func_name,
                                               token[1][1], 0)
            result = func.call(arguments)
        elif token[0] == self.__TYPE_ARRAY:
            result = self.evaluate(namespace, [token], -1, 1, 0) ## FIXME
        elif token[0] == self.__TYPE_NEW_ARRAY:
            result = self.evaluate(namespace, [token], -1, 1, 0) ## FIXME
        elif token[0] == self.__TYPE_VARIABLE:
            var_name = token[1][0]
            if var_name.startswith('_'):
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            if target_namespace.exists(var_name):
                result = target_namespace.get(var_name)
        elif token[0] == self.__TYPE_ARRAY_POINTER:
            var_name = token[1][0]
            if var_name.startswith('_'):
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            index = self.evaluate(namespace, [token[1][1]], -1, 1, 0)
            try:
                index = int(index)
            except:
                logging.debug(
                    'index of array has to be integer: {0}[{1}]'.format(var_name, token[1][1]))
            else:
                value = target_namespace.get(var_name, index)
                result = {'name': var_name,
                          'index': index,
                          'namespace': target_namespace,
                          'value': value}
        elif token[0] == self.__TYPE_VARIABLE_POINTER:
            var_name = token[1][0]
            if var_name.startswith('_'):
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            value = target_namespace.get(var_name)
            result = {'name': var_name,
                      'index': None,
                      'namespace': target_namespace,
                      'value': value}
        else:
            logging.debug('error in evaluate_token: {0}'.format(token))
        return result

    def evaluate_condition(self, namespace, condition):
        result = 0
        if condition[1] is None:
            return 1
        left = condition[1][0]
        ope = condition[1][1]
        right = condition[1][2]
        assert ope[0] == self.__TYPE_OPERATOR
        if left is None: # '!'
            left_result = 1
        else:
            left_result = self.evaluate(namespace, [left], -1, 1, 0) ## FIXME
        right_result = self.evaluate(namespace, [right], -1, 1, 0) ## FIXME
        if ope[1] == '==':
            result = (left_result == right_result)
        elif ope[1] == '!=':
            result = (left_result != right_result)
        elif ope[1] == '_in_':
            if isinstance(right_result, (unicode, str)) and isinstance(left_result, (unicode, str)):
                if left_result in right_result:
                    result = 1
                else:
                    result = 0
            else:
                result = 0
        elif ope[1] == '!_in_':
            if isinstance(right_result, (unicode, str)) and isinstance(left_result, (unicode, str)):
                if left_result not in right_result:
                    result = 1
                else:
                    result = 0
            else:
                result = 0
        elif ope[1] == '<':
            result = left_result < right_result
        elif ope[1] == '>':
            result = left_result > right_result
        elif ope[1] == '<=':
            result = left_result <= right_result
        elif ope[1] == '>=':
            result = left_result >= right_result
        elif ope[1] == '||':
            result = left_result or right_result
        elif ope[1] == '&&':
            result = left_result and right_result
        elif ope[1] == '!':
            result = not right_result
        else:
            pass
        return result

    def evaluate_statement(self, namespace, statement, type_float):
        num = len(statement[1:])
        if num == 0:
            return ''
        type_ = statement[0]
        token = statement[1]
        if type_ == self.__TYPE_STATEMENT:
            left = self.evaluate_statement(namespace, token, type_float)
        else:
            left = self.evaluate_token(namespace, statement)
        if num == 3:
            ope = statement[2][1]
            type_ = statement[3][0]
            if type_ == self.__TYPE_INT:
                token = statement[3][1]
                if type_float:
                    right = float(token)
                else:
                    right = int(token)
            elif type_ == self.__TYPE_FLOAT:
                token = statement[3][1]
                if type_float:
                    right = float(token)
                else:
                    right = int(float(token))
            elif type_ == self.__TYPE_STATEMENT:
                right = self.evaluate_statement(namespace, statement[3],
                                                type_float)
            else:
                right = self.evaluate(namespace, [statement[3]], -1, 1, 0) ## FIXME
            result = self.operation(left, ope, right, type_float)
        else:
            result = left
        return result

    def operation(self, left, ope, right, type_float):
        if not isinstance(left, list):
            try:
                if type_float:
                    left = float(left)
                    right = float(right)
                elif ope != '+' or \
                     (not isinstance(left, (str, unicode)) and not isinstance(right, (str, unicode))):
                    left = int(left)
                    right = int(right)
                else:
                    left = unicode(left)
                    right = unicode(right)
            except:
                left = unicode(left)
                right = unicode(right)
        try:
            if ope == '+':
                return left + right
            elif ope == '-':
                return left - right
            elif ope == '*':
                return left * right
            elif ope == '/':
                if right == 0:
                    return 0
                else:
                    return left / right
            elif ope == '%':
                return left % right
            elif ope == ',':
                assert isinstance(left, list)
                result = []
                result.extend(left)
                if isinstance(right, list):
                    result.extend(right)
                else:
                    result.append(right)
                return result
        except:
            logging.debug(
                'illegal operation: {0}'.format(' '.join((unicode(left), unicode(ope), unicode(right)))))
            return ''

    def get_block(self, parent, startpoint):
        result = []
        n_lines = len(parent)
        inner_nest_level = 0
        for i in range(startpoint, n_lines):
            inner_content = parent[i]
            if inner_content == '{':
                if inner_nest_level > 0:
                    result.append(inner_content)
                inner_nest_level += 1
            elif inner_content == '}':
                inner_nest_level -= 1
                if inner_nest_level > 0:
                    result.append(inner_content)
            else:
                result.append(inner_content)
            if inner_nest_level == 0:
                return i, result
        return startpoint, result

    def evaluate_string(self, namespace, line):
        history = [] # %[n]
        buf = ''
        startpoint = 0
        system_functions = self.dic.aya.get_system_functions()
        while startpoint < len(line):
            pos = line.find('%', startpoint)
            if pos < 0:
                buf = ''.join((buf, line[startpoint:]))
                startpoint = len(line)
                continue
            else:
                buf = ''.join((buf, line[startpoint:pos]))
                startpoint = pos
            if pos == len(line) - 1: # XXX
                break
            if line[pos + 1] == '(':
                start_ = pos + 2
                nest = 0
                current_ = start_
                while line[current_:].count(')') > 0:
                    close_ = line.index(')', current_)
                    if close_ == 0:
                        break
                    nest = line[start_:close_].count('(') - line[start_:close_].count(')')
                    if nest > 0:
                        current_ = close_ + 1
                    else:
                        current_ = close_
                        break
                lines_ = self.parse([line[start_:current_]])
                result_ = self.evaluate(namespace, lines_, -1, 1, 0, 1)
                buf = ''.join((buf, unicode(result_)))
                startpoint = current_ + 1
                continue
            endpoint = len(line)
            for char in self.__SPECIAL_CHARS:
                pos = line.find(char, startpoint + 2, endpoint)
                if 0 < pos < endpoint:
                    endpoint = pos
            if  line[startpoint + 1] == '[': # history
                if line[endpoint] != ']':
                    logging.debug(
                        'unbalanced "%[" or illegal index in '
                        'the string({0})'.format(line))
                    buf = ''
                    break
                index_token = self.parse_token(line[startpoint + 2:endpoint])
                index = self.evaluate_token(namespace, index_token)
                try:
                    index = int(index)
                except:
                    logging.debug(
                        'illegal history index in the string({0})'.format(line))
                else:
                    if 0 <= index < len(history):
                        buf = ''.join((buf, self.format(history[index])))
                startpoint = endpoint + 1
                continue
            replaced = 0
            while endpoint > startpoint + 1:
                token = line[startpoint + 1:endpoint]
                func = self.dic.get_function(token)
                is_system_func = system_functions.exists(token)
                if func is not None or is_system_func:
                    if endpoint < len(line) and \
                       line[endpoint] == '(':
                        end_of_parenthesis = line.find(')', endpoint + 1)
                        if end_of_parenthesis < 0:
                            logging.debug(
                                'unbalanced "(" in the string({0})'.format(line))
                            startpoint = len(line)
                            buf = ''
                            break
                        func_name = token
                        arguments = self.parse_argument(
                            line[endpoint + 1:end_of_parenthesis])
                        arguments = self.evaluate_argument(
                            namespace, func_name, arguments, is_system_func)
                        if is_system_func:
                            if func_name == 'LOGGING':
                                arguments.insert(
                                    0, line[endpoint + 1:end_of_parenthesis])
                                arguments.insert(0, self.name)
                                arguments.insert(0, self.dic.aya.logfile)
                                result_of_func = system_functions.call(
                                    namespace, func_name, arguments)
                            else:
                                result_of_func = system_functions.call(
                                    namespace, func_name, arguments)
                        else:
                            result_of_func = func.call(arguments)
                        if result_of_func is None:
                            result_of_func = ''
                        history.append(result_of_func)
                        buf = ''.join((buf, self.format(result_of_func)))
                        startpoint = end_of_parenthesis + 1
                        replaced = 1
                        break
                    elif func is not None:
                        result_of_func = func.call()
                        history.append(result_of_func)
                        buf = ''.join((buf, self.format(result_of_func)))
                        startpoint = endpoint
                        replaced = 1
                        break
                    else:
                        result_of_func = system_functions.call(
                            namespace, token, [])
                        if result_of_func is None:
                            result_of_func = ''
                        history.append(result_of_func)
                        buf = ''.join((buf, self.format(result_of_func)))
                        startpoint = endpoint
                        replaced = 1
                        break
                else:
                    if token.startswith('_'):
                        target_namespace = namespace
                    else:
                        target_namespace = self.dic.aya.get_global_namespace()
                    if target_namespace.exists(token):
                        have_index = 0
                        index = None
                        if endpoint < len(line) and line[endpoint] == '[':
                            end_of_block = line.find(']', endpoint + 1)
                            if end_of_block < 0:
                                logging.debug(
                                    'unbalanced "[" or '
                                    'illegal index in the string({0})'.format(line))
                                startpoint = len(line)
                                buf = ''
                                break
                            have_index = 1
                            index_token = self.parse_token(
                                line[endpoint + 1:end_of_block])
                            index = self.evaluate_token(namespace, index_token)
                            try:
                                index = int(index)
                            except ValueError:
                                have_index = 0
                                index = None
                        value = target_namespace.get(token, index)
                        if value is not None:
                            content_of_var = value
                            history.append(content_of_var)
                            buf = ''.join((buf,
                                           self.format(content_of_var)))
                            if have_index:
                                startpoint = end_of_block + 1
                            else:
                                startpoint = endpoint
                            replaced = 1
                            break
                endpoint -= 1
            if not replaced:
                buf = ''.join((buf, line[startpoint]))
                startpoint += 1
        return buf

    def format(self, input_num):
        if isinstance(input_num, float):
            result = str(round(input_num, 6))
        else:
            result = unicode(input_num)
        return result

    def evaluate_argument(self, namespace, name, argument, is_system_func):
        arguments = []
        for i in range(len(argument)):
            if is_system_func and \
               self.dic.aya.get_system_functions().not_to_evaluate(name, i):
                arguments.append(argument[i][1][1])
            else:
                value = self.evaluate_statement(namespace, argument[i], 1)
                if type(value) == list:
                    arguments.extend(value)
                else:
                    arguments.append(value)
        return arguments

    def is_substitution(self, line):
        statement = AyaStatement(line)
        if statement.countTokens() >= 3:
            statement.next_token() # left
            ope = statement.next_token()
            ope_list = ['=', ':=',
                        '+=', '-=', '*=', '/=', '%=',
                        '+:=', '-:=', '*:=', '/:=', '%:=',
                        ',=']
            if ope in ope_list:
                return 1
        return 0

    def is_inc_or_dec(self, line):
        if len(line) <= 2:
            return 0
        if line.endswith('++') or line.endswith('--'):
            return 1
        else:
            return 0


class AyaSystemFunctions(object):

    def __init__(self, aya):
        self.aya = aya
        self.current_charset = None
        self.fcharset = {}
        self.charsetlib = None
        self.saori_statuscode = ''
        self.saori_header = []
        self.saori_value = {}
        self.saori_protocol = ''
        self.errno = 0
        self.functions = {
            'ACOS': [self.ACOS, [None], [1], None],
            'ANY': [],
            'ARRAYSIZE': [self.ARRAYSIZE, [0], [1], None],
            'ASEARCH': [],
            'ASEARCHEX': [],
            'ASIN': [self.ASIN, [None], [1], None],
            'ATAN': [self.ATAN, [None], [1], None],
            'BINSTRTOI': [self.BINSTRTOI, [None], [1], None],
            'CEIL': [self.CEIL, [None], [1], None],
            'CHARSETLIB': [self.CHARSETLIB, [None], [1], None],
            'CHR': [self.CHR, [None], [1], None],
            'CHRCODE': [self.CHRCODE, [None], [1], None],
            'COS': [self.COS, [None], [1], None],
            'CUTSPACE': [self.CUTSPACE, [None], [1], None],
            'CVINT': [self.CVINT, [0], [1], None],
            'CVREAL': [self.CVREAL, [0], [1], None],
            'CVSTR': [self.CVSTR, [0], [1], None],
            'ERASE': [self.ERASE, [None], [3], None],
            'ERASEVAR': [self.ERASEVAR, [None], [1], None],
            'EVAL': [self.EVAL, [None], [1], None],
            'FATTRIB': [],
            'FCHARSET': [self.FCHARSET, [None], [1], None],
            'FCLOSE': [self.FCLOSE, [None], [1], None],
            'FCOPY': [self.FCOPY, [None], [2], 259],
            'FDEL': [self.FDEL, [None], [1], 269],
            'FENUM': [self.FENUM, [None], [1, 2], 290],
            'FLOOR': [self.FLOOR, [None], [1], None],
            'FMOVE': [self.FMOVE, [None], [2], 264],
            'FOPEN': [self.FOPEN, [None], [2], 256],
            'FREAD': [self.FREAD, [None], [1], None],
            'FRENAME': [self.FRENAME, [None], [2], 273],
            'FSIZE': [self.FSIZE, [None], [1], 278],
            'FWRITE': [self.FWRITE, [None], [2], None],
            'FWRITE2': [self.FWRITE2, [None], [2], None],
            'GETDELIM': [self.GETDELIM, [0], [1], None],
            'GETLASTERROR': [self.GETLASTERROR, [None], [None], None],
            'GETMEMINFO': [],
            'GETSETTING': [self.GETSETTING, [None], [1], None],
            'GETSTRBYTES': [self.GETSTRBYTES, [None], [1, 2], None],
            'GETTICKCOUNT': [self.GETTICKCOUNT, [None], [0], None],
            'GETTIME': [self.GETTIME, [None], [0], None],
            'GETTYPE': [self.GETTYPE, [None], [1], None],
            'HEXSTRTOI': [self.HEXSTRTOI, [None], [1], None],
            'IARRAY': [self.IARRAY, [None], [None], None],
            'INSERT': [self.INSERT, [None], [3], None],
            'ISFUNC': [self.ISFUNC, [None], [1], None],
            'ISINTSTR': [self.ISINTSTR, [None], [1], None],
            'ISREALSTR': [self.ISREALSTR, [None], [1], None],
            'ISVAR': [self.ISVAR, [None], [1], None],
            'LETTONAME': [self.LETTONAME, [None], [2], None],
            'LOADLIB': [self.LOADLIB, [None], [1], 16],
            'LOG': [self.LOG, [None], [1], None],
            'LOG10': [self.LOG10, [None], [1], None],
            'LOGGING': [],
            'LSO': [],
            'MKDIR': [self.MKDIR, [None], [1], 282],
            'POW': [self.POW, [None], [2], None],
            'RAND': [self.RAND, [None], [0, 1], None],
            'RE_GETLEN': [],
            'RE_GETPOS': [],
            'RE_GETSTR': [],
            'RE_GREP': [],
            'RE_MATCH': [],
            'RE_REPLACE': [],
            'RE_SEARCH': [],
            'RE_SPLIT': [self.RE_SPLIT, [None], [2], None],
            'REPLACE': [self.REPLACE, [None], [3], None],
            'REQUESTLIB': [self.REQUESTLIB, [None], [2], None],
            'RMDIR': [self.RMDIR, [None], [1], 286],
            'ROUND': [self.ROUND, [None], [1], None],
            'SAVEVAR': [self.SAVEVAR, [None], [None], None],
            'SETDELIM': [self.SETDELIM, [0], [2], None],
            'SETLASTERROR': [self.SETLASTERROR, [None], [1], None],
            'SIN': [self.SIN, [None], [1], None],
            'SPLIT': [self.SPLIT, [None], [2, 3], None],
            'SPLITPATH': [self.SPLITPATH, [None], [1], None ],
            'SQRT': [self.SQRT, [None], [1], None],
            'STRFORM': [],
            'STRLEN': [self.STRLEN, [1], [1], None],
            'STRSTR': [self.STRSTR, [3], [3], None],
            'SUBSTR': [self.SUBSTR, [None], [3], None],
            'TAN': [self.TAN, [None], [1], None],
            'TOBINSTR': [self.TOBINSTR, [None], [1], None],
            'TOHEXSTR': [self.TOHEXSTR, [None], [1], None],
            'TOINT': [self.TOINT, [None], [1], None],
            'TOLOWER': [self.TOLOWER, [None], [1], None],
            'TOREAL': [self.TOREAL, [None], [1], None],
            'TOSTR': [self.TOSTR, [None], [1], None],
            'TOUPPER': [self.TOUPPER, [None], [1], None],
            'UNLOADLIB': [self.UNLOADLIB, [None], [1], None],
            
            
            #'ASC': [self.ASC, [None], [1], None],
            #'IASC': [self.IASC, [None], [1], None],
            #'LIB.STATUSCODE': [self.LIB_STATUSCODE, [None], [0], None],
            #'LIB.HEADER': [self.LIB_HEADER, [None], [1], None],
            #'LIB.KEY': [self.LIB_HEADER, [None], [1], None], # alias
            #'LIB.VALUE': [self.LIB_VALUE, [None], [1], None],
            #'LIB.PROTOCOL': [self.LIB_PROTOCOL, [None], [0], None],
            #'LOGGING': [self.LOGGING, [None], [4], None]
        }

    def exists(self, name):
        return name in self.functions

    def call(self, namespace, name, argv):
        self.errno = 0
        if name not in self.functions:
            return ''
        elif not self.functions[name]: # not implemented yet
            logging.warning(
                'aya5.py: SYSTEM FUNCTION "{0}" is not implemented yet.'.format(name))
            return ''
        elif self.check_num_args(name, argv):
            return self.functions[name][0](namespace, argv)
        else:
            return ''

    def not_to_evaluate(self, name, index):
        if index in self.functions[name][1]:
            return 1
        else:
            return 0

    def check_num_args(self, name, argv):
        list_num = self.functions[name][2]
        if list_num == [None]:
            return 1
        else:
            if len(argv) in list_num:
                return 1
            list_num.sort()
            if len(argv) < list_num[0]:
                errno = self.functions[name][3]
                if errno is not None:
                    self.errno = errno
                logging.debug(
                    ''.join((str(name),
                             ': called with too few argument(s)')))
                return 0
            return 1

    def ACOS(self, namespace, argv):
        try:
            result = math.acos(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)

    def ANY(self, namespace, argv):
        pass

    def ARRAYSIZE(self, namespace, argv):
        return len(argv) ## FIXME

    def ASEARCH(self, namespace, argv):
        pass

    def ASEARCHEX(self, namespace, argv):
        pass

    def ASIN(self, namespace, argv):
        try:
            result = math.asin(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)

    def ATAN(self, namespace, argv):
        try:
            result = math.atan(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)
                                
    def BINSTRTOI(self, namespace, argv):
        try:
            return int(str(argv[0]), 2)
        except:
            return 0

    def CEIL(self, namespace, argv):
        try:
            return int(math.ceil(float(argv[0])))
        except:
            return -1

    def CHARSETLIB(self, namespace, argv):
        try:
            value = int(argv[0])
        except:
            return
        if value == 0:
            self.charsetlib = 'Shift_JIS'
        elif value == 1:
            self.charsetlib = 'UTF-8'
        elif value == 127:
            self.charsetlib = self.aya.charset

    def CHR(self, namespace, argv):
        try:
            return unichr(argv[0])
        except:
            return ''

    def CHRCODE(self, namespace, argv):
        line = str(argv[0])
        if line:
            return line[0].encode('utf-16', 'ignore') # UCS-2
        else:
            return ''

    def COS(self, namespace, argv):
        try:
            result = math.cos(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)
                                        
    def CUTSPACE(self, namespace, argv):
        return str(argv[0]).strip()

    def CVINT(self, namespace, argv):
        var = str(argv[0])
        target_namespace = self.select_namespace(namespace, var)
        token = target_namespace.get(var)
        try:
            result = int(token)
        except:
            result = 0
        target_namespace.put(var, result)
        return None

    def CVREAL(self, namespace, argv):
        var = str(argv[0])
        target_namespace = self.select_namespace(namespace, var)
        token = target_namespace.get(var)
        try:
            result = float(token)
        except:
            result = 0.0
        target_namespace.put(var, result)
        return None

    def CVSTR(self, namespace, argv):
        name = str(argv[0])
        target_namespace = self.select_namespace(namespace, name)
        value = str(target_namespace.get(name))
        target_namespace.put(name, value)
        return None

    def ERASE(self, namespace, argv):
        line = str(argv[0])
        try:
            start = int(argv[1])
            num = int(argv[2])
        except:
            return ''
        return ''.join((line[:start], line[start + num:]))

    def ERASEVAR(self, namespace, argv):
        var = str(argv[0])
        target_namespace = self.select_namespace(namespace, var)
        target_namespace.remove(var)

    def EVAL(self, namespace, argv):
        script = argv[0]
        func = AyaFunction(self.aya.dic, '', [script], None)
        result = func.call()
        return result

    def FATTRIB(self, namespace, argv):
        pass

    def FCHARSET(self, namespace, argv):
        try:
            value = int(argv[0])
        except:
            return
        if value == 0:
            self.current_charset = 'Shift_JIS'
        elif value == 1:
            self.current_charset = 'UTF-8'
        elif value == 127:
            self.current_charset = self.aya.charset

    def FCLOSE(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        if norm_path in self.aya.filelist:
            self.aya.filelist[norm_path].close()
            del self.aya.filelist[norm_path]
            del self.f_charset[norm_path]
        return None

    def FCOPY(self, namespace, argv):
        src = get_normalized_path(str(argv[0]), encode=0)
        head, tail = os.path.split(src)
        dst = ''.join((get_normalized_path(str(argv[1]), encode=0), '/', tail))
        src_path = os.path.join(self.aya.aya_dir, src)
        dst_path = os.path.join(self.aya.aya_dir, dst)
        result = 0
        if not os.path.isfile(src_path):
            self.errno = 260
        elif not os.path.isdir(dst_path):
            self.errno = 261
        else:
            try:
                shutil.copyfile(src_path, dst_path)
            except:
                self.errno = 262
            else:
                result = 1
        return result

    def FDEL(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, filename)
        result = 0
        if not os.path.isfile(path):
            self.errno = 270
        else:
            try:
                os.remove(path)
            except:
                self.errno = 271
            else:
                result = 1
        return result

    def FENUM(self, namespace, argv):
        if len(argv) >= 2:
            separator = str(argv[1])
        else:
            separator = ','
        dirname = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, dirname)
        filelist = []
        try:
            filelist = os.listdir(path)
        except:
            self.errno = 291
        result = ''
        for index in range(len(filelist)):
            path = os.path.join(self.aya.aya_dir, dirname, filelist[index])
            if os.path.isdir(path):
                result = ''.join((result, '\\'))
            result = ''.join((result, filelist[index]))
            if index != len(filelist) - 1:
                result = ''.join((result, separator))
        return result

    def FLOOR(self, namespace, argv):
        try:
            return int(math.floor(float(argv[0])))
        except:
            return -1

    def FMOVE(self, namespace, argv):
        src = get_normalized_path(str(argv[0]), encode=0)
        head, tail = os.path.split(src)
        dst = ''.join((get_normalized_path(str(argv[1]), encode=0), '/', tail))
        src_path = os.path.join(self.aya.aya_dir, src)
        dst_path = os.path.join(self.aya.aya_dir, dst)
        result = 0
        head, tail = os.path.split(dst_path)
        if not os.path.isfile(src_path):
            self.errno = 265
        elif not os.path.isdir(head):
            self.errno = 266
        else:
            try:
                os.rename(src_path, dst_path)
            except:
                self.errno = 267
            else:
                result = 1
        return result

    def FOPEN(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        accessmode = str(argv[1])
        result = 0
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        if norm_path in self.aya.filelist:
            result = 2
        else:
            try:
                self.aya.filelist[norm_path] = open(path, accessmode[0])
            except:
                self.errno = 257
            else:
                if self.current_charset is None:
                    self.current_charset = self.aya.charset
                self.f_charset[norm_path] = self.current_charset
                result = 1
        return result

    def FREAD(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        result = -1
        if norm_path in self.aya.filelist:
            f = self.aya.filelist[norm_path]
            result = unicode(
                f.readline(), self.fcharset[norm_path], 'ignore')
            if not result:
                result = -1
            elif result.endswith('\r\n'):
                result = result[:-2]
            elif result.endswith('\n'):
                result = result[:-1]
        return result

    def FRENAME(self, namespace, argv):
        src = get_normalized_path(str(argv[0]), encode=0)
        dst = get_normalized_path(str(argv[1]), encode=0)
        src_path = os.path.join(self.aya.aya_dir, src)
        dst_path = os.path.join(self.aya.aya_dir, dst)
        result = 0
        head, tail = os.path.split(dst_path)
        if not os.path.exists(src_path):
            self.errno = 274
        elif not os.path.isdir(head):
            self.errno = 275
        else:
            try:
                os.rename(src_path, dst_path)
            except:
                self.errno = 276
            else:
                result = 1
        return result

    def FSIZE(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, filename)
        size = -1
        if not os.path.exists(path):
            self.errno = 279
        else:
            try:
                size = os.path.getsize(path)
            except:
                self.errno = 280
        return size

    def FWRITE(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        data = ''.join((str(argv[1]), '\n'))
        if norm_path in self.aya.filelist:
            f = self.aya.filelist[norm_path]
            f.write(data.encode(self.fcharset[norm_path], 'ignore'))
        return None

    def FWRITE2(self, namespace, argv):
        filename = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        data = str(argv[1])
        if norm_path in self.aya.filelist:
            f = self.aya.filelist[norm_path]
            f.write(data.encode(self.fcharset[norm_path], 'ignore'))
        return None

    def GETDELIM(self, namespace, argv):
        name = str(argv[0])
        target_namespace = self.select_namespace(namespace, name)
        return target_namespace.get_separator(name)

    def GETLASTERROR(self, namespace, argv):
        return self.errno

    def GETMEMINFO(self, namespace, argv):
        pass

    def GETSETTING(self, namespace, argv):
        try:
            value = int(argv[0])
        except:
            return ''
        if value == 0:
            result = '5'
        elif value == 1:
            if self.aya.charset == 'Shift_JIS':
                result = 0
            elif self.aya.charset == 'UTF-8':
                result = 1
            else:
                result = 2
        elif value == 2:
            result = self.aya.aya_dir
        else:
            result = ''
        return result

    def GETSTRBYTES(self, namespace, argv):
        line = str(argv[0])
        if len(argv) > 1:
            try:
                value = int(argv[1])
            except:
                value = 0
        else:
            value = 0
        if value == 0:
            result = len(line.encode('Shift_JIS', 'ignore'))
        elif value == 1:
            result = len(line.encode('utf-8', 'ignore'))
        elif value == 2:
            result = len(line.encode(self.aya.charset, 'ignore'))
        else:
            result = -1
        return result

    def GETTICKCOUNT(self, namespace, argv):
        t = time.localtime(time.time())
        past = time.time() - self.aya.get_boot_time()
        return int(past * 1000.0)

    def GETTIME(self, namespace, argv):
        year_, month_, day_, hour_, minute_, second_, wday_, yday_, isdst_ = time.localtime()
        wday_ = (wday_ + 1) % 7
        return [year_, month_, day_, wday_, hour_, minute_, second_]

    def GETTYPE(self, namespace, argv):
        if isinstance(argv[0], int):
            result = 1
        elif isinstance(argv[0], float):
            result = 2
        elif isinstance(argv[0], (unicode, str)):
            result = 3
        elif 0: ## FIXME: array
            result = 4
        else:
            result = 0
        return result

    def HEXSTRTOI(self, namespace, argv):
        try:
            return int(str(argv[0]), 16)
        except:
            return 0

    def IARRAY(self, namespace, argv):
        return [] # AyaVariable('', new_array=1) ## FIXME

    def INSERT(self, namespace, argv):
        line = unicode(argv[0])
        try:
            start = int(argv[1])
        except:
            return ''
        to_insert = unicode(argv[2])
        if start < 0:
            start = 0
        return ''.join((line[:start], to_insert, line[start:]))

    def ISFUNC(self, namespace, argv):
        if not isinstance(argv[0], (unicode, str)):
            return 0
        elif self.aya.dic.get_function(argv[0]) is not None:
            return 1
        elif self.aya.get_system_functions().exists(argv[0]):
            return 2
        else:
            return 0

    def ISINTSTR(self, namespace, argv):
        try:
            int(str(argv[0]))
            return 1
        except:
            return 0

    def ISREALSTR(self, namespace, argv):
        try:
            float(str(argv[0]))
            return 1
        except:
            return 0

    def ISVAR(self, namespace, argv):
        var = str(argv[0])
        if var.startswith('_'):
            if namespace.exists(var):
                return 2
            else:
                return 0
        else:
            if self.aya.get_global_namespace().exists(var):
                return 1
            else:
                return 0

    def LETTONAME(self, namespace, argv):
        var = str(argv[0])
        value = argv[1]
        if not var:
            return None
        target_namespace = self.select_namespace(namespace, var)
        target_namespace.put(var, value)
        return None

    def LOADLIB(self, namespace, argv):
        dll = str(argv[0])
        result = 0
        if dll:
            if self.charsetlib is None:
                self.charsetlib = self.aya.charset
            self.aya.saori_library.set_charset(self.charsetlib)
            result = self.aya.saori_library.load(dll, self.aya.aya_dir)
            if result == 0:
                self.errno = 17
        return result

    def LOG(self, namespace, argv):
        try:
            float(argv[0])
        except:
            return -1
        if float(argv[0]) == 0:
            return 0
        result = math.log(float(argv[0]))
        return self.select_math_type(result)

    def LOG10(self, namespace, argv):
        try:
            float(argv[0])
        except:
            return -1
        if float(argv[0]) == 0:
            return 0
        result = math.log10(float(argv[0]))
        return self.select_math_type(result)

    def LSO(self, namespace, argv):
        pass

    def MKDIR(self, namespace, argv):
        dirname = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, dirname)
        result = 0
        head, tail = os.path.split(path)
        if not os.path.isdir(head):
          self.errno = 283
        else:
            try:
                os.mkdir(path, 0755)
            except:
                self.errno = 284
            else:
                result = 1
        return result

    def POW(self, namespace, argv):
        try:
            result = math.pow(float(argv[0]), float(argv[1]))
        except:
            return -1
        return self.select_math_type(result)

    def RAND(self, namespace, argv):
        if not argv:
            return int(random.randrange(0, 100, 1))
        else:
            try:
                int(argv[0])
            except:
                return -1
            return int(random.randrange(0, int(argv[0]), 1))

    def RE_GETLEN(self, namespace, argv):
        pass

    def RE_GETPOS(self, namespace, argv):
        pass

    def RE_GETSTR(self, namespace, argv):
        pass

    def RE_GREP(self, namespace, argv):
        pass

    def RE_MATCH(self, namespace, argv):
        pass

    def RE_REPLACE(self, namespace, argv):
        pass

    def RE_SEARCH(self, namespace, argv):
        pass

    def RE_SPLIT(self, namespace, argv):
        line = unicode(argv[0])
        result = line.split(unicode(argv[1]))
        return result ## FIXME

    def REPLACE(self, namespace, argv):
        line = unicode(argv[0])
        old = unicode(argv[1])
        new = unicode(argv[2])
        return line.replace(old, new)

    def REQUESTLIB(self, namespace, argv):
        response = self.aya.saori_library.request(argv[0], argv[1])
        header = response.splitlines()
        line = header.pop(0)
        self.saori_statuscode = ''
        self.saori_header = []
        self.saori_value = {}
        self.saori_protocol = ''
        if line:
            line = line.strip()
            if ' ' in line:
                self.saori_protocol, self.saori_statuscode = [x.strip() for x in line.split(' ', 1)]
            for line in header:
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                if key:
                    self.saori_header.append(key)
                    self.saori_value[key] = value
        return response

    def RMDIR(self, namespace, argv):
        dirname = get_normalized_path(str(argv[0]), encode=0)
        path = os.path.join(self.aya.aya_dir, dirname)
        result = 0
        if not os.path.isdir(path):
            self.errno = 287
        else:
            try:
                os.rmdir(path)
            except:
                self.errno = 288
            else:
                result = 1
        return result

    def ROUND(self, namespace, argv):
        try:
            value = math.floor(float(argv[0]) + 0.5)
        except:
            return -1
        return int(value)

    def SAVEVAR(self, namespace, argv):
        self.aya.get_global_namespace().save_database()

    def SETDELIM(self, namespace, argv):
        name = unicode(argv[0])
        separator = unicode(argv[1])
        target_namespace = self.select_namespace(namespace, name)
        target_namespace.set_separator(name, separator)
        return None

    def SETLASTERROR(self, namespace, argv):
        try:
            value = int(argv[0])
        except:
            return
        self.errno = value

    def SIN(self, namespace, argv):
        try:
            result = math.sin(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)
                                        
    def SPLIT(self, namespace, argv):
        line = unicode(argv[0])
        result = line.split(unicode(argv[1]), int(argv[2]))
        return result

    def SPLITPATH(self, namespace, argv):
        line = unicode(argv[0])
        drive, path = os.path.splitdrive(line)
        dirname, filename = os.path.split(path)
        basename, ext = os.path.splitext(filename)
        return [drive, dirname, basename, ext]

    def SQRT(self, namespace, argv):
        try:
            arg = float(argv[0])
        except:
            return -1
        if arg < 0.:
            return -1
        else:
            result = math.sqrt(arg)
            return self.select_math_type(result)

    def STRFORM(self, namespace, argv):
        pass

    def STRLEN(self, namespace, argv):
        line = unicode(argv[0])
        return len(line)

    def STRSTR(self, namespace, argv):
        line = unicode(argv[0])
        to_find = unicode(argv[1])
        try:
            start = int(argv[2])
        except:
            return -1
        result = line.find(to_find, start)
        return result

    def SUBSTR(self, namespace, argv):
        line = unicode(argv[0])
        try:
            start = int(argv[1])
            num = int(argv[2])
        except:
            return ''
        return line[start:start + num]

    def TAN(self, namespace, argv):
        try:
            result = math.tan(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)

    def TOBINSTR(self, namespace, argv):
        try:
            i = int(argv[0])
        except:
            return ''
        if i < 0:
            i = abs(i)
            numsin = '-'
        else:
            numsin = ''
        line = ''
        while i:
            mod = i % 2
            i /= 2
            line = ''.join((str(mod), line))
        line = ''.join((numsin, line))
        return line

    def TOHEXSTR(self, namespace, argv):
        try:
            return '{0:x}'.format(int(argv[0]))
        except:
            return ''

    def TOINT(self, namespace, argv):
        token = str(argv[0])
        try:
            value = int(token)
        except:
            return 0
        else:
            return value

    def TOLOWER(self, namespace, argv):
        return unicode(argv[0]).lower()

    def TOREAL(self, namespace, argv):
        token = str(argv[0])
        try:
            value = float(token)
        except:
            return 0.0
        else:
            return value

    def TOSTR(self, namespace, argv):
        return unicode(argv[0])

    def TOUPPER(self, namespace, argv):
        return unicode(argv[0]).upper()

    def UNLOADLIB(self, namespace, argv):
        if str(argv[0]):
            self.aya.saori_library.unload(str(argv[0]))
        return None

    def LOGGING(self, namespace, argv):
        if argv[0] is None:
            return None
        logfile = argv[0]
        line = ''.join(('> function ', str(argv[1]), ' ： ', str(argv[2])))
        if argv[3] is not None:
            line = ''.join((line, ' = '))
            if isinstance(argv[3], int) or isinstance(argv[3], float):
                line = ''.join((line, str(argv[3])))
            else:
                line = ''.join((line, '"', str(argv[3]), '"'))
        line = ''.join((line, '\n'))
        logfile.write(line)
        logfile.write('\n')
        return None

    def ASC(self, namespace, argv):
        try:
            int(argv[0])
        except:
            return ''
        index = int(argv[0])
        if 0 <= index < 0x80: ## FIXME
            return chr(index)
        else:
            return ' '

    def IASC(self, namespace, argv):
        if isinstance(argv[0], str):
            return -1
        try:
            char = argv[0][0]
        except:
            return -1
        return ord(char)

    def LIB_STATUSCODE(self, namespace, argv):
        return self.saori_statuscode

    def LIB_HEADER(self, namespace, argv):
        try:
            int(argv[0])
        except:
            return ''
        result = ''
        header_list = self.saori_header
        if header_list and int(argv[0]) < len(header_list):
            result = header_list[int(argv[0])]
        return result

    def LIB_VALUE(self, namespace, argv):
        result = ''
        if isinstance(argv[0], int):
            header_list = self.saori_header
            if header_list and int(argv[0]) < len(header_list):
                key = header_list[int(argv[0])]
        else:
            key = str(argv[0])
        if key in self.saori_value:
            result = self.saori_value[key]
        return result

    def LIB_PROTOCOL(self, namespace, argv):
        return self.aya.saori_protocol

    def select_math_type(self, value):
        if math.floor(value) == value:
            return int(value)
        else:
            return value

    def select_namespace(self, namespace, name):
        if name.startswith('_'):
            return namespace
        else:
            return self.aya.get_global_namespace()


class AyaNamespace(object):

    def __init__(self, aya, parent=None):
        self.aya = aya
        self.parent = parent
        self.table = {}

    def put(self, name, content, index=None):
        if self.parent is not None and self.parent.exists(name):
            self.parent.put(name, content, index)
        elif index is None:
            if not self.exists(name):
                self.table[name] = AyaVariable(name)
            self.table[name].put(content)
        elif self.exists(name) and index >=0:
            self.table[name].put(content, index)
        else:
            pass # ERROR

    def get(self, name, index=None):
        if name in self.table:
            return self.table[name].get(index)
        elif self.parent is not None and self.parent.exists(name):
            return self.parent.get(name, index)
        else:
            return None

    def get_array(self, name):
        if name in self.table:
            return self.table[name].get_array()
        elif self.parent is not None and self.parent.exists(name):
            return self.parent.get_array(name)
        else:
            return []

    def get_separator(self, name):
        if self.parent is not None and self.parent.exists(name):
            return self.parent.get_separator(name)
        elif name in self.table:
            self.table[name].get_separator()
        else:
            return '' # ERROR

    def set_separator(self, name, separator):
        if self.parent is not None and self.parent.exists(name):
            self.parent.set_separator(name, separator)
        elif name in self.table:
            self.table[name].set_separator(separator)
        else:
            pass # ERROR

    def get_size(self, name):
        if name in self.table:
            return self.table[name].get_size()
        elif self.parent is not None and self.parent.exists(name):
            return self.parent.get_size(name)
        else:
            return 0

    def remove(self, name): # only works with local table
        if name in self.table:
            del self.table[name]

    def exists(self, name):
        result = name in self.table or \
                 (self.parent is not None and self.parent.exists(name))
        return result


class AyaGlobalNamespace(AyaNamespace):

    def load_database(self, aya):
        try:
            with open(aya.dbpath) as f:
                line = f.readline()
                if not line.startswith('# Format: v1.2'):
                    f.close()
                    return 1
                for line in f:
                    comma = line.find(',')
                    if comma >= 0:
                        key = line[:comma]
                    else:
                        continue
                    value = line[comma + 1:].strip()
                    comma = find_not_quoted(value, ',')
                    if comma >= 0:
                        separator = value[comma + 1:].strip()
                        separator = separator[1:-1]
                        value = value[:comma].strip()
                        value = value[1:-1]
                        self.put(key, unicode(value, self.aya.charset))
                        self.table[key].set_separator(unicode(separator, self.aya.charset))
                    elif value != 'None':
                        if '.' in value:
                            self.put(key, float(value))
                        else:
                            self.put(key, int(value))
                    else:
                        pass
        except:
            return 1
        return 0

    def save_database(self):
        try:
            with open(self.aya.dbpath, 'w') as f:
                f.write('# Format: v1.2\n')
                for key in self.table.keys():
                    line = self.table[key].dump(self.aya.charset)
                    if line is not None:
                        f.write(''.join((line, '\n')))
        except IOError:
            logging.debug('aya.py: cannot write database (ignored)')
            return


class AyaStatement(object):

    __SPECIAL_CHARS = '=+-*/<>|&!:,_'

    def __init__(self, line):
        self.n_tokens = 0
        self.tokens = []
        self.position_of_next_token = 0
        self.tokenize(line)

    def tokenize(self, line): ## FIXME: '[', ']'
        token_startpoint = 0
        block_nest_level = 0
        length = len(line)
        i = 0
        while i < length:
            c = line[i]
            if c == '(':
                block_nest_level += 1
                self.append_unless_empty(line[token_startpoint:i].strip())
                self.tokens.append('(')
                i += 1
                token_startpoint = i
            elif c == ')':
                block_nest_level -= 1
                self.append_unless_empty(line[token_startpoint:i].strip())
                self.tokens.append(')')
                i += 1
                token_startpoint = i
            elif c == '[':
                self.append_unless_empty(line[token_startpoint:i].strip())
                self.tokens.append('[')
                i += 1
                token_startpoint = i
            elif c == ']':
                self.append_unless_empty(line[token_startpoint:i].strip())
                self.tokens.append(']')
                i += 1
                token_startpoint = i
            elif c == '"':
                position = line.find('"', i + 1)
                if position < 0: ## FIXME
                    raise SystemExit
                i = position
                self.tokens.append(line[token_startpoint:position + 1])
                token_startpoint = position + 1
                i = position + 1
            elif c == "'":
                position = line.find("'", i + 1)
                if position < 0: ## FIXME
                    raise SystemExit
                i = position
                self.tokens.append(line[token_startpoint:position + 1])
                token_startpoint = position + 1
                i = position + 1
            elif c == ' ' or c == '\t' or c == unicode('　', 'utf-8'):
                self.append_unless_empty(line[token_startpoint:i].strip())
                i += 1
                token_startpoint = i
            elif c in self.__SPECIAL_CHARS:
                ope_list = ['!_in_', '_in_',
                            '+:=', '-:=', '*:=', '/:=', '%:=',
                            ':=', '+=', '-=', '*=', '/=', '%=', '<=',
                            '>=', '==', '!=', '&&', '||', ',=', '++', '--',
                            ',', '=', '!', '+', '-', '/', '*', '%', '&']
                for ope in ope_list:
                    if line[i:].startswith(ope): ## FIXME
                        self.append_unless_empty(line[token_startpoint:i].strip())
                        num = len(ope)
                        self.tokens.append(line[i:i + num])
                        i += num
                        token_startpoint = i
                        break
                else:
                    i += 1
            else:
                i += 1
        self.append_unless_empty(line[token_startpoint:].strip())
        self.n_tokens = len(self.tokens)

    def append_unless_empty(self, token):
        if token:
            self.tokens.append(token)

    def has_more_tokens(self):
        return (self.position_of_next_token < self.n_tokens)

    def countTokens(self):
        return self.n_tokens

    def next_token(self):
        if not self.has_more_tokens():
            return None
        result = self.tokens[self.position_of_next_token]
        self.position_of_next_token += 1
        return result


class AyaVariable(object):

    __TYPE_STRING = 0
    __TYPE_INT = 1
    __TYPE_REAL = 2
    __TYPE_ARRAY = 3
    __TYPE_NEW_ARRAY = 4

    def __init__(self, name, new_array=0):
        self.name = name
        self.line = ''
        self.separator = ','
        if new_array:
            self.type = self.__TYPE_NEW_ARRAY
        else:
            self.type = None
        self.array = []

    def get_array(self):
        return self.array

    def get_separator(self):
        return self.separator

    def set_separator(self, separator):
        if self.type != self.__TYPE_STRING:
            return
        self.separator = separator
        self.reset()

    def reset(self):
        if self.type != self.__TYPE_STRING:
            return
        self.position = 0
        self.is_empty = 0
        self.array = []
        while not self.is_empty:
            separator_position = self.line.find(self.separator, self.position)
            if separator_position == -1:
                token = self.line[self.position:]
                self.is_empty = 1
            else:
                token = self.line[self.position:separator_position]
                self.position = separator_position + len(self.separator)
            self.array.append(token)

    def get_size(self):
        return len(self.array)

    def get(self, index=None):
        if 0 <= index < len(self.array):
            value = self.array[index]
            if self.type == self.__TYPE_STRING:
                result = unicode(value)
            elif self.type == self.__TYPE_INT:
                result = int(value)
            elif self.type == self.__TYPE_REAL:
                result = float(value)
            elif self.type == self.__TYPE_ARRAY:
                result = value
            elif self.type == self.__TYPE_NEW_ARRAY:
                result = value ## FIXME
            else:
                result = None # should not reach here
        elif index is None:
            if self.type == self.__TYPE_STRING:
                result = unicode(self.line)
            elif self.type == self.__TYPE_INT:
                result = int(self.line)
            elif self.type == self.__TYPE_REAL:
                result = float(self.line)
            elif self.type == self.__TYPE_NEW_ARRAY:
                result = self.array ## FIXME(?)
            else:
                result = ''
        else:
            result = ''
        return result

    def put(self, value, index=None):
        if index is None:
            self.line = unicode(value)
            if isinstance(value, (unicode, str)):
                self.type = self.__TYPE_STRING
            elif isinstance(value, int):
                self.type = self.__TYPE_INT
            elif isinstance(value, float):
                self.type = self.__TYPE_REAL
            elif isinstance(value, list):
                self.type = self.__TYPE_NEW_ARRAY # CHECK
                self.array = value
            self.reset()
        elif index < 0:
            pass
        else:
            if self.type == self.__TYPE_STRING:
                self.line = ''
                for i in range(len(self.array)):
                    if i == index:
                        self.line = ''.join((self.line, unicode(value)))
                    else:
                        self.line = ''.join((self.line, self.array[i]))
                    if i != len(self.array)-1:
                        self.line = ''.join((self.line, self.separator))
                if index >= len(self.array):
                    for i in range(len(self.array), index + 1):
                        if i == index:
                            self.line = ''.join((self.line, self.separator,
                                                 unicode(value)))
                        else:
                            self.line = ''.join((self.line, self.separator,
                                                 ''))
                self.reset()
            elif self.type == self.__TYPE_ARRAY:
                if 0 <= index < len(self.array):
                    self.array[index] = value
            elif self.type == self.__TYPE_NEW_ARRAY:
                if 0 <= index < len(self.array):
                    self.array[index] = value
                elif index > 0: ## FIXME
                    logging.info('!!! WARNING !!!')
                    logging.info('NOT YET IMPLEMNETED')
                else:
                    pass # ERROR
            else:
                pass # ERROR

    def dump(self, charset):
        line = None
        if self.type == self.__TYPE_STRING:
            line = '{0}, "{1}", "{2}"'.format(self.name, self.line, self.separator)
            line = line.encode(charset, 'ignore')
        elif self.type == self.__TYPE_NEW_ARRAY:
            pass ## FIXME
        elif self.type != self.__TYPE_ARRAY:
            line = '{0}, {1}'.format(self.name, self.line)
            line = line.encode(charset, 'ignore')
        else:
            pass
        return line


class AyaArgument(object):

    def __init__(self, line):
        self.line = line.strip()
        self.length = len(self.line)
        self.current_position = 0

    def has_more_tokens(self):
        return (self.current_position != -1 and \
                self.current_position < self.length)

    def next_token(self):
        if not self.has_more_tokens():
            return None
        startpoint = self.current_position
        self.current_position = self.position_of_next_token()
        if self.current_position == -1:
            token = self.line[startpoint:]
        else:
            token = self.line[startpoint:self.current_position-1]
        return token.strip()

    def position_of_next_token(self):
        locked = 1
        position = self.current_position
        parenthesis_nest_level = 0
        while position < self.length:
            c = self.line[position]
            if c == '"':
                if not locked:
                    return position
                while position < self.length-1:
                    position += 1
                    if self.line[position] == '"':
                        break
            elif c == '(':
                parenthesis_nest_level += 1
            elif c == ')':
                parenthesis_nest_level -= 1
            elif c == ',':
                if parenthesis_nest_level == 0:
                    locked = 0
            else:
                if not locked:
                    return position
            position += 1
        return -1


class AyaSaoriLibrary(object):

    def __init__(self, saori, top_dir):
        self.saori_list = {}
        self.saori = saori
        self.current_charset = None
        self.charset = {}

    def set_charset(self, charset):
        assert charset in ['Shift_JIS', 'UTF-8']
        self.current_charset = charset

    def load(self, name, top_dir):
        result = 0
        if self.saori and name not in self.saori_list:
            module = self.saori.request(name)
            if module:
                self.saori_list[name] = module
        if name in self.saori_list:
            result = self.saori_list[name].load(top_dir)
        self.charset[name] = self.current_charset
        return result

    def unload(self, name=None):
        if name:
            if name in self.saori_list:
                self.saori_list[name].unload()
                del self.saori_list[name]
                del self.charset[name]
        else:
            for key in self.saori_list.keys():
                self.saori_list[key].unload()
        return None

    def request(self, name, req):
        result = '' # FIXME
        if name and name in self.saori_list:
            result = self.saori_list[name].request(
                req.encode(self.current_charset))
        return result


def test(top_dir, function, argv, dll='aya.dll'):
    aya = Shiori(dll)
    print 'DIR:', top_dir
    aya.load(top_dir)
    print dir(aya)
    print aya.dic.functions
    result = aya.request('GET SHIORI/3.0\r\n' \
                         'ID: {0}\r\n' \
                         'Sender: AYA\r\n' \
                         'SecurityLevel: local\r\n\r\n'.format(function))
    #result = aya.dic.get_function(function).call(argv)
    print unicode(result, aya.charset)

if __name__ == '__main__':
    USAGE= 'Usage(1): aya.py <dir> <dll_name> <function> [<arguments>]\n' \
           'Usage(2): aya.py encrypt <in_file> <out_file>'
    if len(sys.argv) < 3:
        print USAGE
    elif sys.argv[1] == 'encrypt':
        if len(sys.argv) != 4:
            print USAGE
        else:
            path = os.path.join(os.curdir, sys.argv[2])
            with open(path) as inputf:
                path = os.path.join(os.curdir, sys.argv[3])
                with open(path, 'w') as outputf:
                    while 1:
                        c = inputf.read(1)
                        if c == '':
                            break
                        outputf.write(encrypt_char(c))
    else:
        test(sys.argv[1], sys.argv[3], sys.argv[4:], sys.argv[2])
