/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// NewPKIStore.h: interface for the NewPKIStore class.
//
//////////////////////////////////////////////////////////////////////

#ifndef NEWPKISTORE_H
#define NEWPKISTORE_H

#include <openssl/x509.h>
#include <openssl/evp.h>
#include <mString.h>
#include <ASN1/Asn1Cert.h>
#include <ASN1/Asn1Conf.h>
#include <PKI_CERT.h>
#include "SQL/SQL_Conn.h"

#include "EntityLog.h"
#include "MailInfo.h"
#include "AsynchJobs.h"
class AsynchJobs;



/*!
	This class holds the basic functionnalities for a store
*/
class NewPKIStore
{
public:
	static int CaStatus2StoreStatus(int ca_status, int unknow_status);
	/*! \brief This is the constructor.
	 *  \param EntityName [IN] The name of the entity.
	 *  \param e [IN] The ENGINE, can be NULL.
	 */
	NewPKIStore(const mString & EntityName, ENGINE * e);

	/*! \brief This is the destructor.
	 */
	virtual ~NewPKIStore();

	/*! \brief This function creates the tables needed to work.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	virtual bool CreateTables(const SQL_Connection * DbConn) = 0;

	/*! \brief Checks if the mailer is up and running.
	 *  \return true on success, false on failure.
	 */
	bool MailerIsUp();

	/*! \brief This function sets the AsynchJobs instance, used to send emails.
	 *  \param Jobs [IN] The AsynchJobs instance.
	 */
	void set_Jobs(AsynchJobs * Jobs);
	
	/*! \brief This function converts binary datas to a string.
	 *  \param datas [IN] The binary datas.
	 *  \param len [IN] The binary datas length.
	 *  \param strId [OUT] The result string.
	 *  \return true on success, false on failure.
	 */
	static bool BINARYtoString(const unsigned char * datas, int len, mString &strId);

	/*! \brief This function converts a string to binary datas.
	 *  \param strId [IN] The string.
	 *  \param len [OUT] The binary datas length.
	 *  \return The binary datas or NULL on failure.
	 */
	static unsigned char * StringtoBINARY(const mString & strId, int & len);

	/*! \brief This function converts a ASN1_BIT_STRING to a string.
	 *  \param bit [IN] The ASN1_BIT_STRING.
	 *  \param strId [OUT] The result string.
	 *  \return true on success, false on failure.
	 */
	static bool ASN1_BIT_STRINGtoString(const ASN1_BIT_STRING * bit, mString &strId);

	/*! \brief This function converts a string to a ASN1_BIT_STRING.
	 *  \param strId [IN] The string.
	 *  \return The ASN1_BIT_STRING or NULL on failure.
	 */
	static ASN1_BIT_STRING * StringtoASN1_BIT_STRING(const mString & strId);
	
	/*! \brief This function converts a string to a Asn1OctetString.
	 *  \param strId [IN] The string.
	 *  \param transactionID [OUT] The result Asn1OctetString.
	 *  \return true on success, false on failure.
	 */
	static bool StringtoTransactionID(const mString & strId, Asn1OctetString & transactionID);
	
	/*! \brief This function converts a Asn1OctetString to a string.
	 *  \param transactionID [IN] The Asn1OctetString.
	 *  \param strId [OUT] The result string.
	 *  \return true on success, false on failure.
	 */
	static bool transactionIDtoString(const Asn1OctetString & transactionID, mString & strId);

	/*! \brief This function converts an EVP_PKEY to a hash string.
	 *  \param pubKey [IN] The EVP_PKEY.
	 *  \param strKey [OUT] The result string.
	 *  \return true on success, false on failure.
	 */
	static bool EVP_PKEYtoHash(const EVP_PKEY * pubKey, mString &strKey);

	/*! \brief This function converts a X509_PUBKEY to a hash string.
	 *  \param pubKey [IN] The X509_PUBKEY.
	 *  \param strKey [OUT] The result string.
	 *  \return true on success, false on failure.
	 */
	static bool X509_PUBKEYtoHash(const X509_PUBKEY * pubKey, mString &strKey);

	/*! \brief This function converts a X509_NAME to a hash string.
	 *  \param dn [IN] The X509_NAME.
	 *  \param strHash [OUT] The result string.
	 *  \return true on success, false on failure.
	 */
	static bool X509_NAMEtoHash(const X509_NAME * dn, mString &strHash);

	/*! \brief This function serializes a X509_NAME.
	 *  \param dn [IN] The X509_NAME.
	 *  \param sDn [OUT] The serialized DN.
	 *  \return true on success, false on failure.
	 */
	static bool X509_NAMEtoString(const X509_NAME * dn, mString & sDn);
	
	/*! \brief This function unserializes a X509_NAME.
	 *  \param sDn [IN] The serialized DN.
	 *  \return The X509_NAME on success, NULL on failure.
	 */
	static X509_NAME * StringtoX509_NAME(const mString & sDn);

	/*! \brief This function sets the SQL connection.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	bool SetDbConn(const SQL_Connection * DbConn);

	/*! \brief This function sets the logging class.
	 *  \param Logging [IN] The logging class.
	 */
	void set_Logging(const EntityLog * Logging);

	/*! \brief This function sets the certificate of the entity using this class.
	 *  \param EntityCert [IN] The certificate of the entity.
	 */
	void SetEntityCert(const PKI_CERT & EntityCert);
protected:
	#define MAX_TID_LEN SHA_DIGEST_LENGTH

	mString m_EntityName;
	const EntityLog * m_Logging;
	ENGINE * m_Engine;
	bool SendMail(const mString & Author, const MailInfo & Mail, bool AdminMail);
	PKI_CERT m_EntityCert;
	SQL_Connection * m_DbConn;
	bool GenerateTransactionID(Asn1OctetString & transactionID, const X509_PUBKEY *sender, const X509_PUBKEY *recipient, unsigned long counter) const;

private:
	AsynchJobs * m_Jobs;
	unsigned long m_Type;
};

#endif
