/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ENTITY_PKI_ASN1_H
#define ENTITY_PKI_ASN1_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include "Entity.h"
#include <ASN1/Asn1Helper.h>
#include <ASN1/Asn1Conf.h>


/*!
	This structure represent an entity conf.
*/
typedef struct st_STORED_ENTITY_CONF
{
	ENTITY_CONF * conf;					//!< The conf.
	X509 * certificate;					//!< The certificate of the entity.
}STORED_ENTITY_CONF;
DECLARE_ASN1_ITEM(STORED_ENTITY_CONF)


/*!
	This structure represent an entity conf.
*/
typedef struct st_STORED_ENTITY_CONF_BETA4
{
	ENTITY_CONF_BETA4 * conf;					//!< The conf.
	X509 * certificate;					//!< The certificate of the entity.
}STORED_ENTITY_CONF_BETA4;
DECLARE_ASN1_ITEM(STORED_ENTITY_CONF_BETA4)


/*!
	This structure represent the local conf.
*/
typedef struct st_PKI_CONF
{
	STACK_OF(STORED_ENTITY_CONF) * confs;		//!< The confs in "clear text" format
	EXPORTED_PKI_CONF * crypt_confs;			//!< The same as confs but crypted for the entities
	STACK_OF(REP_ENTRY_INFO) * Repositories;	//!< The repositories list
	PUBS_INFO * Publications;					//!< The Publications list
	INTERNAL_PKI_CA * cas;						//!< The PKI internal CA.
	ENTITY_CONF * conf;							//!< The PKI entity conf
	STACK_OF(ENTITY_LINKS) * links;				//!< The entities links
}PKI_CONF;
DECLARE_ASN1_ITEM(PKI_CONF)

/*!
	This structure represent the local conf.
*/
typedef struct st_PKI_CONF_BETA4
{
	STACK_OF(STORED_ENTITY_CONF_BETA4) * confs;		//!< The confs in "clear text" format
	EXPORTED_PKI_CONF * crypt_confs;			//!< The same as confs but crypted for the entities
	STACK_OF(REP_ENTRY_INFO) * Repositories;	//!< The repositories list
	PUBS_INFO * Publications;					//!< The Publications list
	INTERNAL_PKI_CA * cas;						//!< The PKI internal CA.
	ENTITY_CONF_BETA4 * conf;					//!< The PKI entity conf
	STACK_OF(ENTITY_LINKS) * links;				//!< The entities links
}PKI_CONF_BETA4;
DECLARE_ASN1_ITEM(PKI_CONF_BETA4)

/*!
  This class is the representation of STORED_ENTITY_CONF
*/
class StoredEntityConf : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	StoredEntityConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	StoredEntityConf(const StoredEntityConf & other);

	/*! \brief This is the destructor.
	 */
	~StoredEntityConf();


	/*! \brief This function converts a STORED_ENTITY_CONF to a StoredEntityConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const STORED_ENTITY_CONF * Datas);

	/*! \brief This function converts a StoredEntityConf to a STORED_ENTITY_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(STORED_ENTITY_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of STORED_ENTITY_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::certificate.
	 *  \param c_certificate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certificate(const PKI_CERT & c_certificate);

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	const PKI_CERT & get_certificate() const;

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	PKI_CERT & get_certificate();


	/*! \brief This function sets ENTITY_CONF::conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const EntityConf & c_conf);

	/*! \brief This function returns ENTITY_CONF::conf.
	 *  \return The value of ENTITY_CONF::conf.
	 */
	const EntityConf & get_conf() const;

	/*! \brief This function returns ENTITY_CONF::conf.
	 *  \return The value of ENTITY_CONF::conf.
	 */
	EntityConf & get_conf();


	/*! \brief This function copies a StoredEntityConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const StoredEntityConf & other);
	/*! \brief This member is an empty instance of StoredEntityConf.
	 */
	static StoredEntityConf EmptyInstance;

private:
	PKI_CERT m_certificate;
	EntityConf m_conf;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of STORED_ENTITY_CONF_BETA4
*/
class StoredEntityConfBeta4 : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	StoredEntityConfBeta4();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	StoredEntityConfBeta4(const StoredEntityConfBeta4 & other);

	/*! \brief This is the destructor.
	 */
	~StoredEntityConfBeta4();


	/*! \brief This function converts a STORED_ENTITY_CONF_BETA4 to a StoredEntityConfBeta4.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const STORED_ENTITY_CONF_BETA4 * Datas);

	/*! \brief This function converts a StoredEntityConfBeta4 to a STORED_ENTITY_CONF_BETA4.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(STORED_ENTITY_CONF_BETA4 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of STORED_ENTITY_CONF_BETA4.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::certificate.
	 *  \param c_certificate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certificate(const PKI_CERT & c_certificate);

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	const PKI_CERT & get_certificate() const;

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	PKI_CERT & get_certificate();


	/*! \brief This function sets ENTITY_CONF_BETA4::conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const EntityConfBeta4 & c_conf);

	/*! \brief This function returns ENTITY_CONF_BETA4::conf.
	 *  \return The value of ENTITY_CONF_BETA4::conf.
	 */
	const EntityConfBeta4 & get_conf() const;

	/*! \brief This function returns ENTITY_CONF_BETA4::conf.
	 *  \return The value of ENTITY_CONF_BETA4::conf.
	 */
	EntityConfBeta4 & get_conf();


	/*! \brief This function copies a StoredEntityConfBeta4 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const StoredEntityConfBeta4 & other);
	/*! \brief This member is an empty instance of StoredEntityConfBeta4.
	 */
	static StoredEntityConfBeta4 EmptyInstance;

private:
	PKI_CERT m_certificate;
	EntityConfBeta4 m_conf;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PKI_CONF
*/
class PkiConf : public NewPKIObject, public LocalEntityConf
{
public:
	/*! \brief This is the constructor.
	 */
	PkiConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PkiConf(const PkiConf & other);

	/*! \brief This is the destructor.
	 */
	~PkiConf();


	/*! \brief This function converts a PKI_CONF to a PkiConf.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKI_CONF * Datas);

	/*! \brief This function converts a PkiConf to a PKI_CONF.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKI_CONF ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PKI_CONF.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets PUBS_INFO::Publications.
	 *  \param c_publications [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_publications(const PubsInfo & c_publications);

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	const PubsInfo & get_publications() const;

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	PubsInfo & get_publications();


	/*! \brief This function sets REP_ENTRY_INFO::Repositories.
	 *  \param c_repositories [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositories(const mVector< RepEntryInfo > & c_repositories);

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	const mVector< RepEntryInfo > & get_repositories() const;

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	mVector< RepEntryInfo > & get_repositories();


	/*! \brief This function sets INTERNAL_PKI_CA::cas.
	 *  \param c_cas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cas(const InternalPkiCa & c_cas);

	/*! \brief This function returns INTERNAL_PKI_CA::cas.
	 *  \return The value of INTERNAL_PKI_CA::cas.
	 */
	const InternalPkiCa & get_cas() const;

	/*! \brief This function returns INTERNAL_PKI_CA::cas.
	 *  \return The value of INTERNAL_PKI_CA::cas.
	 */
	InternalPkiCa & get_cas();


	/*! \brief This function sets ENTITY_CONF::conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const EntityConf & c_conf);

	/*! \brief This function returns ENTITY_CONF::conf.
	 *  \return The value of ENTITY_CONF::conf.
	 */
	const EntityConf & get_conf() const;

	/*! \brief This function returns ENTITY_CONF::conf.
	 *  \return The value of ENTITY_CONF::conf.
	 */
	EntityConf & get_conf();


	/*! \brief This function sets STORED_ENTITY_CONF::confs.
	 *  \param c_confs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_confs(const mVector< StoredEntityConf > & c_confs);

	/*! \brief This function returns STORED_ENTITY_CONF::confs.
	 *  \return The value of STORED_ENTITY_CONF::confs.
	 */
	const mVector< StoredEntityConf > & get_confs() const;

	/*! \brief This function returns STORED_ENTITY_CONF::confs.
	 *  \return The value of STORED_ENTITY_CONF::confs.
	 */
	mVector< StoredEntityConf > & get_confs();


	/*! \brief This function sets EXPORTED_PKI_CONF::crypt_confs.
	 *  \param c_cryptConfs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cryptConfs(const ExportedPkiConf & c_cryptConfs);

	/*! \brief This function returns EXPORTED_PKI_CONF::crypt_confs.
	 *  \return The value of EXPORTED_PKI_CONF::crypt_confs.
	 */
	const ExportedPkiConf & get_cryptConfs() const;

	/*! \brief This function returns EXPORTED_PKI_CONF::crypt_confs.
	 *  \return The value of EXPORTED_PKI_CONF::crypt_confs.
	 */
	ExportedPkiConf & get_cryptConfs();


	/*! \brief This function sets ENTITY_LINKS::links.
	 *  \param c_links [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_links(const mVector< EntityLinks > & c_links);

	/*! \brief This function returns ENTITY_LINKS::links.
	 *  \return The value of ENTITY_LINKS::links.
	 */
	const mVector< EntityLinks > & get_links() const;

	/*! \brief This function returns ENTITY_LINKS::links.
	 *  \return The value of ENTITY_LINKS::links.
	 */
	mVector< EntityLinks > & get_links();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a PkiConf to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PkiConf & other);
	/*! \brief This member is an empty instance of PkiConf.
	 */
	static PkiConf EmptyInstance;

private:
	PubsInfo m_publications;
	mVector< RepEntryInfo > m_repositories;
	InternalPkiCa m_cas;
	EntityConf m_conf;
	mVector< StoredEntityConf > m_confs;
	ExportedPkiConf m_cryptConfs;
	mVector< EntityLinks > m_links;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PKI_CONF_BETA4
*/
class PkiConfBeta4 : public NewPKIObject, public LocalEntityConfBeta4
{
public:
	/*! \brief This is the constructor.
	 */
	PkiConfBeta4();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PkiConfBeta4(const PkiConfBeta4 & other);

	/*! \brief This is the destructor.
	 */
	~PkiConfBeta4();


	/*! \brief This function converts a PKI_CONF_BETA4 to a PkiConfBeta4.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKI_CONF_BETA4 * Datas);

	/*! \brief This function converts a PkiConfBeta4 to a PKI_CONF_BETA4.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKI_CONF_BETA4 ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PKI_CONF_BETA4.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets PUBS_INFO::Publications.
	 *  \param c_publications [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_publications(const PubsInfo & c_publications);

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	const PubsInfo & get_publications() const;

	/*! \brief This function returns PUBS_INFO::Publications.
	 *  \return The value of PUBS_INFO::Publications.
	 */
	PubsInfo & get_publications();


	/*! \brief This function sets REP_ENTRY_INFO::Repositories.
	 *  \param c_repositories [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositories(const mVector< RepEntryInfo > & c_repositories);

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	const mVector< RepEntryInfo > & get_repositories() const;

	/*! \brief This function returns REP_ENTRY_INFO::Repositories.
	 *  \return The value of REP_ENTRY_INFO::Repositories.
	 */
	mVector< RepEntryInfo > & get_repositories();


	/*! \brief This function sets INTERNAL_PKI_CA::cas.
	 *  \param c_cas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cas(const InternalPkiCa & c_cas);

	/*! \brief This function returns INTERNAL_PKI_CA::cas.
	 *  \return The value of INTERNAL_PKI_CA::cas.
	 */
	const InternalPkiCa & get_cas() const;

	/*! \brief This function returns INTERNAL_PKI_CA::cas.
	 *  \return The value of INTERNAL_PKI_CA::cas.
	 */
	InternalPkiCa & get_cas();


	/*! \brief This function sets ENTITY_CONF_BETA4::conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const EntityConfBeta4 & c_conf);

	/*! \brief This function returns ENTITY_CONF_BETA4::conf.
	 *  \return The value of ENTITY_CONF_BETA4::conf.
	 */
	const EntityConfBeta4 & get_conf() const;

	/*! \brief This function returns ENTITY_CONF_BETA4::conf.
	 *  \return The value of ENTITY_CONF_BETA4::conf.
	 */
	EntityConfBeta4 & get_conf();


	/*! \brief This function sets STORED_ENTITY_CONF_BETA4::confs.
	 *  \param c_confs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_confs(const mVector< StoredEntityConfBeta4 > & c_confs);

	/*! \brief This function returns STORED_ENTITY_CONF_BETA4::confs.
	 *  \return The value of STORED_ENTITY_CONF_BETA4::confs.
	 */
	const mVector< StoredEntityConfBeta4 > & get_confs() const;

	/*! \brief This function returns STORED_ENTITY_CONF_BETA4::confs.
	 *  \return The value of STORED_ENTITY_CONF_BETA4::confs.
	 */
	mVector< StoredEntityConfBeta4 > & get_confs();


	/*! \brief This function sets EXPORTED_PKI_CONF::crypt_confs.
	 *  \param c_cryptConfs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cryptConfs(const ExportedPkiConf & c_cryptConfs);

	/*! \brief This function returns EXPORTED_PKI_CONF::crypt_confs.
	 *  \return The value of EXPORTED_PKI_CONF::crypt_confs.
	 */
	const ExportedPkiConf & get_cryptConfs() const;

	/*! \brief This function returns EXPORTED_PKI_CONF::crypt_confs.
	 *  \return The value of EXPORTED_PKI_CONF::crypt_confs.
	 */
	ExportedPkiConf & get_cryptConfs();


	/*! \brief This function sets ENTITY_LINKS::links.
	 *  \param c_links [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_links(const mVector< EntityLinks > & c_links);

	/*! \brief This function returns ENTITY_LINKS::links.
	 *  \return The value of ENTITY_LINKS::links.
	 */
	const mVector< EntityLinks > & get_links() const;

	/*! \brief This function returns ENTITY_LINKS::links.
	 *  \return The value of ENTITY_LINKS::links.
	 */
	mVector< EntityLinks > & get_links();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a PkiConfBeta4 to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PkiConfBeta4 & other);
	/*! \brief This member is an empty instance of PkiConfBeta4.
	 */
	static PkiConfBeta4 EmptyInstance;

private:
	PubsInfo m_publications;
	mVector< RepEntryInfo > m_repositories;
	InternalPkiCa m_cas;
	EntityConfBeta4 m_conf;
	mVector< StoredEntityConfBeta4 > m_confs;
	ExportedPkiConf m_cryptConfs;
	mVector< EntityLinks > m_links;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ENTITY_PKI_ASN1_H
