/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef CA_HANDLER_ASN1_H
#define CA_HANDLER_ASN1_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <PKI_CERT.h>
#include <PKI_CRL.h>
#include <ASN1/Asn1Helper.h>

/*!
	This structure represents the body of a CA.
*/
typedef struct st_INTERNAL_CA_BODY
{
	ASN1_INTEGER * ca_type;			//!< The type of CA.
	ASN1_INTEGER * ca_serial;		//!< The current serial.
	X509_CRL * lastCrl;				//!< The last CRL.
}INTERNAL_CA_BODY;
DECLARE_ASN1_ITEM(INTERNAL_CA_BODY)

/*!
	This structure represents the signature of certificate entry.
*/
typedef struct st_INTERNAL_CA_CERT_SIG
{
	ASN1_BIT_STRING * signature;	//!< The signature
	X509_ALGOR * sig_alg;			//!< The signature algo
}INTERNAL_CA_CERT_SIG;
DECLARE_ASN1_ITEM(INTERNAL_CA_CERT_SIG)

#define INTERNAL_CA_CERT_sign(d, x,pkey) \
	ASN1_item_sign(InternalCaCert::get_ASN1_ITEM(), x.get_sigAlg(), NULL, x.get_signature(), (char *)d, pkey, EVP_sha1())

#define INTERNAL_CA_CERT_verify(d, x,pkey) \
	ASN1_item_verify(InternalCaCert::get_ASN1_ITEM(), x.get_sigAlg(), x.get_signature(),(char *)d, pkey)




/*!
  This class is the representation of INTERNAL_CA_CERT_SIG
*/
class InternalCaCertSig : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalCaCertSig();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalCaCertSig(const InternalCaCertSig & other);

	/*! \brief This is the destructor.
	 */
	~InternalCaCertSig();


	/*! \brief This function converts a INTERNAL_CA_CERT_SIG to a InternalCaCertSig.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_CA_CERT_SIG * Datas);

	/*! \brief This function converts a InternalCaCertSig to a INTERNAL_CA_CERT_SIG.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_CA_CERT_SIG ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_CA_CERT_SIG.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_ALGOR::sig_alg.
	 *  \param c_sigAlg [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sigAlg(const X509_ALGOR * c_sigAlg);

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	const X509_ALGOR * get_sigAlg() const;

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	X509_ALGOR * get_sigAlg();


	/*! \brief This function sets ASN1_BIT_STRING::signature.
	 *  \param c_signature [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signature(const ASN1_BIT_STRING * c_signature);

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	const ASN1_BIT_STRING * get_signature() const;

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	ASN1_BIT_STRING * get_signature();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a InternalCaCertSig to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalCaCertSig & other);
	/*! \brief This member is an empty instance of InternalCaCertSig.
	 */
	static InternalCaCertSig EmptyInstance;

private:
	X509_ALGOR * m_sigAlg;
	ASN1_BIT_STRING * m_signature;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of INTERNAL_CA_BODY
*/
class InternalCaBody : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalCaBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalCaBody(const InternalCaBody & other);

	/*! \brief This is the destructor.
	 */
	~InternalCaBody();


	/*! \brief This function converts a INTERNAL_CA_BODY to a InternalCaBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_CA_BODY * Datas);

	/*! \brief This function converts a InternalCaBody to a INTERNAL_CA_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_CA_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_CA_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::ca_serial.
	 *  \param c_caSerial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caSerial(unsigned long c_caSerial);

	/*! \brief This function returns ASN1_INTEGER::ca_serial.
	 *  \return The value of ASN1_INTEGER::ca_serial.
	 */
	unsigned long get_caSerial() const;


	/*! \brief This function sets ASN1_INTEGER::ca_type.
	 *  \param c_caType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caType(unsigned long c_caType);

	/*! \brief This function returns ASN1_INTEGER::ca_type.
	 *  \return The value of ASN1_INTEGER::ca_type.
	 */
	unsigned long get_caType() const;


	/*! \brief This function sets X509_CRL::lastCrl.
	 *  \param c_lastcrl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lastcrl(const PKI_CRL & c_lastcrl);

	/*! \brief This function returns X509_CRL::lastCrl.
	 *  \return The value of X509_CRL::lastCrl.
	 */
	const PKI_CRL & get_lastcrl() const;

	/*! \brief This function returns X509_CRL::lastCrl.
	 *  \return The value of X509_CRL::lastCrl.
	 */
	PKI_CRL & get_lastcrl();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a InternalCaBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalCaBody & other);
	/*! \brief This member is an empty instance of InternalCaBody.
	 */
	static InternalCaBody EmptyInstance;

private:
	unsigned long m_caSerial;
	unsigned long m_caType;
	PKI_CRL m_lastcrl;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef CA_HANDLER_ASN1_H
