/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// AsynchJobs.cpp: implementation of the AsynchJobs class.
//
//////////////////////////////////////////////////////////////////////

#include "AsynchJobs.h"
#include "MIME.h"
#include "svintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


AsynchJobs::AsynchJobs()
{
	m_DbConn = NULL;

	hThreadMailQueue.Create(ThreadMailQueue, this);
	hThreadConfigurationPush.Create(ThreadConfigurationPush, this);

}

AsynchJobs::~AsynchJobs()
{
	StopAll();
	if(m_DbConn)
		delete m_DbConn;
}

bool AsynchJobs::StartMailer()
{
	if(!hThreadMailQueue.Start())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	return true;
}

void AsynchJobs::StopMailer()
{
	hThreadMailQueue.Stop();
}

bool AsynchJobs::StartConfPush(bool PushToAllRep)
{
	m_PushToAllRep = PushToAllRep;
	if(!hThreadConfigurationPush.Start())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

void AsynchJobs::StopConfPush()
{
	hThreadConfigurationPush.Stop();
}

void AsynchJobs::StopAll()
{
	StopMailer();
	StopConfPush();
}

void AsynchJobs::SetEntityCert(const PKI_CERT & cert)
{
	m_EntityCert = cert;
}

void AsynchJobs::SetEntityName(const mString & EntityName)
{
	m_EntityName = EntityName;
}

bool AsynchJobs::SetEntityParentsCert(const mVector<PKI_CERT> & EntityParentCerts)
{
	m_AdminMailQueueLock.EnterCS();
	m_EntityParentCerts = EntityParentCerts;
	m_AdminMailQueueLock.LeaveCS();
	return true;
}

void AsynchJobs::SetLogger(EntityLog *log)
{
	m_Logging = log;
}

bool AsynchJobs::SetAdminMails(const mVector<PkiAdminEntry> & Admins)
{
	size_t i;

	m_AdminMailQueueLock.EnterCS();
	m_AdminMails.clear();
	for(i=0; i<Admins.size(); i++)
	{
		m_AdminMails.push_back(Admins[i].get_email());
	}
	m_AdminMailQueueLock.LeaveCS();
	return true;
}

bool AsynchJobs::SetMailerInfo(mString MailFrom, const EmailConf & conf)
{
	m_AdminMailQueueLock.EnterCS();
	m_MailerConf = conf;
	m_AdminMailQueueLock.LeaveCS();

	if(!m_Mailer.SetInfo(MailFrom, m_MailerConf.get_server(), m_MailerConf.get_port(), m_Logging))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}



bool AsynchJobs::SendMail(const mString & Author, const MailInfo & Mail, bool AdminMail) const
{
	if(!Mail)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	MailQueueEntry q;

	q.get_mail().set_subject(Mail.get_Subject());
	q.get_mail().set_body(Mail.get_Body());
	q.get_mail().set_signmail(Mail.get_SignMail()?1:0);
	q.set_author(Author);
	q.set_adminmail(AdminMail?1:0);

	// If it's not an admin mail we get the recipient from
	// the mail
	if(!AdminMail)
	{
		if(!Mail.get_MailTo().size())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		q.get_recipients().push_back(Mail.get_MailTo());
	}
	// If it's an admin mail
	// the recipients will be given later
	
	// Set attachement
	if(Mail.c_get_Attach().get_Buffer() && Mail.c_get_Attach().get_BufferLen())
	{	
		if(!q.get_mail().get_attach().Copy(Mail.c_get_Attach().get_Buffer(), Mail.c_get_Attach().get_BufferLen()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	q.get_mail().set_attachname(Mail.get_AttachName());
	q.get_mail().set_attachtype(Mail.get_AttachType());

	q.get_mail().set_isOK();
	if(!InsertMail(q))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}


bool AsynchJobs::InsertMail(MailQueueEntry & q) const
{
	mString pem;
	if(!q.to_PEM(pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(MAIL_INSERT, pem.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	q.set_id(sql.GetLastID());

	m_AdminMailQueueLock.EnterCS();
	((AsynchJobs*)this)->m_AdminMailQueue.push_back(q);
	m_AdminMailQueueLock.LeaveCS();
	return true;
}

void AsynchJobs::ThreadMailQueue(const NewpkiThread * Thread, void *param)
{
	AsynchJobs * me_this = (AsynchJobs *)param;
	MIME mimeEncoder;
	bool ret;
	mString Body;
	mString mime;
	mVector<mString> Recipients;
	time_t lastLoad = 0;;
	time_t currTime;
	MailQueueEntry currMail;

	do 
	{
		me_this->m_AdminMailQueueLock.EnterCS();
		do
		{
			time(&currTime);
			if( (currTime - lastLoad) >= MAILS_DB_RELOAD )
			{
				me_this->LoadMails();
				lastLoad = currTime;
			}
			me_this->m_AdminMailQueueLock.LeaveCS();
			NewpkiThread::Sleep(100);
			me_this->m_AdminMailQueueLock.EnterCS();
		}
		while(!Thread->ShouldStop() && (!me_this->m_AdminMailQueue.size() || !me_this->m_Mailer || !me_this->m_EntityParentCerts.size()));
		
		if(Thread->ShouldStop())
		{
			me_this->m_AdminMailQueueLock.LeaveCS();
			return;
		}
		
		currMail = me_this->m_AdminMailQueue[0];
		me_this->m_AdminMailQueue.erase(me_this->m_AdminMailQueue.begin());
		me_this->m_AdminMailQueueLock.LeaveCS();

		Body = _sv("Entity: ");
		Body += me_this->m_EntityName;
		Body += _sv("\nAuthor: ");
		Body += currMail.get_author();
		Body += "\n\n";
		Body += currMail.get_mail().get_body();

		if(currMail.get_recipients().size())
			Recipients = currMail.get_recipients();
		else
			Recipients = me_this->m_AdminMails;

		if( (ret = mimeEncoder.GenerateMIME(mime, Body.c_str(), currMail.get_mail().get_attach().get_Buffer(), currMail.get_mail().get_attach().get_BufferLen(), currMail.get_mail().get_attachname().c_str(), currMail.get_mail().get_attachtype().c_str())) )
		{
			if(currMail.get_mail().get_signmail())
				ret = mimeEncoder.GenerateSMIME(mime, mime.c_str(), me_this->m_EntityCert.GetX509(), me_this->m_EntityCert.GetPrivateKey().GetRsaKey(), me_this->m_EntityParentCerts);
		}
		if(ret)
		{
			ret = me_this->m_Mailer.Send((currMail.get_adminmail() == 1), Recipients, currMail.get_mail().get_subject(), mime);
		}

		// The email was successfully sent we can 
		// now remove it from the DB
		if(ret)
		{
			me_this->RemoveMail(currMail.get_id());
			currMail.Clear();
		}
	}
	while(true);
}


void AsynchJobs::RemoveMail(unsigned long id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	if(req.sprintf(MAIL_DELETE, id) <= 0)
		return;
	sql.Execute(req);
}


void AsynchJobs::LoadMails()
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString pem;
	long NumRows;
	long i;
	size_t j;
	MailQueueEntry newEntry;
	if(!sql.Execute(MAIL_SELECT))
	{
		return;
	}
	if(!sql.NumRows(&NumRows))
	{
		return;
	}
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i, "mail", pem))
		{
			return;
		}
		if(!newEntry.from_PEM(pem))
		{
			return;
		}
		if(!sql.Value(i, "mail_id", pem))
		{
			return;
		}

		newEntry.set_id(pem.c_ulng());

		// Do we already have this mail in the stack ?
		for(j=0; j<m_AdminMailQueue.size(); j++)
		{
			if(m_AdminMailQueue[j].get_id() == newEntry.get_id())
				break;
		}
		// The mail wasn't found
		if(j == m_AdminMailQueue.size())
		{
			m_AdminMailQueueLock.EnterCS();
			m_AdminMailQueue.push_back(newEntry);
			m_AdminMailQueueLock.LeaveCS();
		}
	}
}





bool AsynchJobs::SetRepositories(const mVector<RepEntryInfo> &Repositories)
{
	m_RepositoriesLock.LockWrite();
	m_Repositories = Repositories;
	m_RepositoriesLock.UnlockWrite();
	return true;
}


void AsynchJobs::ThreadConfigurationPush(const NewpkiThread * Thread, void *param)
{
	AsynchJobs * me_this = (AsynchJobs *)param;
	time_t waitTime;

	//We wait a bit
	do
	{
		NewpkiThread::Sleep(500);
	}
	while(!Thread->ShouldStop() &&
		!me_this->m_Repositories.size());

	waitTime = 15;
	while(!Thread->ShouldStop())
	{

		me_this->ConfPushLock.EnterCS();
			//Do we have a conf ?
			if(!me_this->m_PkiConf)
			{
				me_this->ConfPushLock.LeaveCS();
				goto wait_next;
			}		


			// We deploy new configuration on repositories
			if(!me_this->Private_PushConfiguration(Thread))
			{
				waitTime = 60;
			}
			else
			{
				me_this->m_PkiConf.Clear();
				waitTime = 15;
			}
		me_this->ConfPushLock.LeaveCS();

wait_next:
		if(!Thread->SleepInterrupt(waitTime))
			break;
	}
}

bool AsynchJobs::Private_PushConfiguration(const NewpkiThread * Thread)
{
	size_t i;
	mString RepName;
	mString RepAddress;
	unsigned int RepPort;
	PKI_CERT RepCert;
	mString err;
	bool Success = false;
	PkiClient ClientPki(NULL);

	// We deploy conf on first available repository
	// the repository will be in charge on deploying it
	// on all other repositories
	m_RepositoriesLock.LockRead();
	for(i=0; i<m_Repositories.size(); i++)
	{
		if(Thread->ShouldStop())
		{
			m_RepositoriesLock.UnlockRead();
			return true;
		}

		// If the current rep is me or is firewalled or aloready
		// known the conf, we ignore it
		if(m_EntityCert == m_Repositories[i].get_repositoryssl() ||
			ASN1_BIT_STRING_get_bit(m_Repositories[i].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED) ||
			RepositoryKnowsObject(m_PkiConf.get_repPath(), m_Repositories[i].get_repositoryssl()))
			continue;


		// We copy the datas, so we can unlock
		RepName = m_Repositories[i].get_name();
		RepAddress = m_Repositories[i].get_address();
		RepPort = m_Repositories[i].get_port();
		RepCert = m_Repositories[i].get_repositoryssl();
		m_RepositoriesLock.UnlockRead();

		ERR_clear_error();

		m_Logging->LogMessage(LOG_STATUS_TYPE_REQUEST, LOG_MESSAGE_TYPE_CONFIG_PUSH, 0, NULL, LOG_NO_OBJECTID, RepName.c_str());
		
		//We connect to the repository
		if(!ConnectToRepository(m_EntityCert, RepName, RepAddress, RepPort, RepCert, &ClientPki, 30, err))
		{
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to push conf on entity %s (%s:%ld)\nReason:%s"), RepName.c_str(), RepAddress.c_str(), RepPort, err.c_str());
			m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_CONFIG_PUSH, 0, NULL, LOG_NO_OBJECTID, RepName.c_str(), err.c_str());
		}
		else
		{
			if(!ClientPki.PushConfiguration(m_PkiConf))
			{			
				NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to push conf on entity %s (%s:%ld)\nReason:%s"), RepName.c_str(), RepAddress.c_str(), RepPort, ClientPki.GetError());
				m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_CONFIG_PUSH, 0, NULL, LOG_NO_OBJECTID, RepName.c_str(), ClientPki.GetError());
			}
			else
			{
				NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Successfully pushed conf on entity %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
				m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LOG_MESSAGE_TYPE_CONFIG_PUSH, 0, NULL, LOG_NO_OBJECTID, RepName.c_str());
				if(!m_PushToAllRep)
				{
					ClientPki.CloseConnection();
					return true;
				}
				AddRepositoryToObjectsPath(m_PkiConf.get_repPath(), m_Repositories[i].get_repositoryssl());
				Success = true;
			}
		}
		m_RepositoriesLock.LockRead();
	}
	m_RepositoriesLock.UnlockRead();
	ClientPki.CloseConnection();
	return Success;
}


bool AsynchJobs::SetPkiConf(const ExportedPkiConf & PkiConf)
{
	ConfPushLock.EnterCS();
	if(!(m_PkiConf = PkiConf))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfPushLock.LeaveCS();
		return false;
	}
	ConfPushLock.LeaveCS();
	return true;
}

bool AsynchJobs::ConnectToRepository(const PKI_CERT & EntityCert, const RepEntryInfo & currRep, PkiClient * ClientPki, int TimeOut, mString & err)
{
	return ConnectToRepository(EntityCert, currRep.get_name(), currRep.get_address(), currRep.get_port(), currRep.get_repositoryssl(), ClientPki, TimeOut, err);
}

bool AsynchJobs::ConnectToRepository(const PKI_CERT & EntityCert, const mString & RepName, const mString & RepAddress, unsigned int RepPort, const PKI_CERT & RepCert, PkiClient * ClientPki, int TimeOut, mString & err)
{
	PKI_CERT peerCert;
	int UserType;
	long Pos;
	const char * cn;
	AdminReqLogin Login;

	err="";

	ClientPki->CloseConnection();

	ClientPki->SetReadTimeOut(TimeOut);

	//We connect to the repository
	if(!ClientPki->Connect(RepAddress.c_str(), RepPort, EntityCert, NULL))
	{
		err = ClientPki->GetError();
		return false;
	}
	Login.set_entity(RepName);
	if(!ClientPki->UserLogin(Login, UserType))
	{
		err = ClientPki->GetError();
		return false;
	}
	
	//Does the cert correspond to the one expected ?
	ClientPki->GetEntityCert(peerCert);
	if(!peerCert)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNEXPECTED_CERT);
		ERR_to_mstring(err);
		return false;
	}

	if(peerCert != RepCert)
	{
		// Is the CN the same ?
		Pos = peerCert.GetCertDN().SeekEntryName("commonName", HASHTABLE_NOT_FOUND);
		if(Pos != HASHTABLE_NOT_FOUND)
		{
			cn = peerCert.GetCertDN().Get(Pos);
			if(cn)
			{
				if(RepName == cn)
				{
					return true;
				}
			}
		}

		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNEXPECTED_CERT);
		ERR_to_mstring(err);
		return false;
	}
	return true;
}


void AsynchJobs::SetX509_ACL_Validator(const X509_ACL_Validator *AclValidator)
{
	m_AclValidator = AclValidator;
}

bool AsynchJobs::CreateTables(SQL_Connection *DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {MAIL_JOBS_STACK, NULL};
	mString req;
	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}

bool AsynchJobs::SetDbConn(SQL_Connection *DbConn)
{
	if(m_DbConn)
	{
		delete m_DbConn;
		m_DbConn = NULL;
	}
	try
	{
		m_DbConn = DbConn->Clone();
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_DbConn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	return true;
}


bool AsynchJobs::AddRepositoryToObjectsPath(STACK_OF(X509_PUBKEY) * rep_path, const PKI_CERT & Cert)
{
	X509_PUBKEY * currPubkey;

	if(RepositoryKnowsObject(rep_path, Cert))
		return true;

	currPubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)Cert.GetX509_PUBKEY());
	if(!currPubkey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(SKM_sk_push(X509_PUBKEY, rep_path, currPubkey) < 0)
	{
		X509_PUBKEY_free(currPubkey);
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool AsynchJobs::RepositoryKnowsObject(const STACK_OF(X509_PUBKEY) * rep_path, const PKI_CERT & Cert)
{
	int j;
	X509_PUBKEY * currPubKey;

	for(j=0; j<SKM_sk_num(X509_PUBKEY, rep_path); j++)
	{
		currPubKey = SKM_sk_value(X509_PUBKEY, rep_path, j);
		if(!currPubKey) continue;

		if(Cert == currPubKey)
		{
			return true;
		}
	}

	return false;
}

bool AsynchJobs::MailerIsUp()
{
	return (m_Mailer != 0);
}

const SmtpClient & AsynchJobs::GetMailer() const
{
	return m_Mailer;
}
