--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Narval.Configurator.Abstract_Actors_Coordination;
with Narval.Local_Configuration;
use Narval.Local_Configuration;

package body Narval.Actors.Actives.Consumers.Basic_Generic is

   use Log4ada.Loggers;

   procedure Do_Nothing (Object : System.Address);
   procedure Do_Nothing_On_Start (Object : System.Address;
                                     Run_Number : Integer);

   procedure Do_Nothing (Object : System.Address) is
      use type System.Address;
      pragma Unreferenced (Object);
   begin
      null;
   end Do_Nothing;

   procedure Do_Nothing_On_Start (Object : System.Address;
                                     Run_Number : Integer) is
      use type System.Address;
      pragma Unreferenced (Object, Run_Number);
   begin
      null;
   end Do_Nothing_On_Start;

   -----------------
   -- Initialiser --
   -----------------

   procedure On_Initialise
     (Object : access Generic_Consumer_Type) is
      use Shared_Library;
      use Configurator.Abstract_Actors_Coordination;
      use Ada.Strings.Unbounded;
      Cursor : Configurator.Actors_Description.String_Vector.Cursor;
      use type Configurator.Actors_Description.String_Vector.Cursor;
   begin
      Consumers.On_Initialise
        (Consumer_Access (Object));
      Object.Inputs (1).Memory.Toggle_Dumping_Mode
        (Event_Based_Dumping => False);
      Object.Waiting_Delay := 0.1;
      Cursor := Configurator.Actors_Description.String_Vector.First
        (Object.Description.Arguments);
      loop
         exit when
           Cursor = Configurator.Actors_Description.String_Vector.No_Element;
         declare
            Argument : constant String := To_String
              (Configurator.Actors_Description.String_Vector.Element (Cursor));
         begin
            if Argument'Length > 13 then
               if Argument (1 .. 12) = "bibliotheque" then
                  Object.Library_Reference :=
                    Open_Library
                    (Argument (14 .. Argument'Last), RTLD_NOW);
               end if;
            end if;
         end;
         Cursor := Configurator.Actors_Description.String_Vector.Next (Cursor);
      end loop;
      Object.Init_Function := Library_Symbol
        (Object.Library_Reference,
         "fonction_initialisation");
      if Object.Init_Function = null then
         Fatal_Out (Object.Logger'Access,
                    "Fonction_Initialisation non definie");
         raise Initialisation_Failed;
      end if;
      Object.Buffer_Handling := Library_Symbol
        (Object.Library_Reference,
         "travail_en_acquisition");
      if Object.Buffer_Handling = null then
         Fatal_Out (Object.Logger'Access,
                    "Travail_En_Acquisition non definie");
         raise Initialisation_Failed;
      end if;
      Object.On_Unload := Library_Symbol
        (Object.Library_Reference, "sur_decharger");
      if Object.On_Unload = null then
         Object.On_Unload := Do_Nothing'Access;
      end if;
      Object.On_Reset_Com := Library_Symbol
        (Object.Library_Reference,
         "sur_remise_a_zero");
      if Object.On_Reset_Com = null then
         Object.On_Reset_Com := Do_Nothing'Access;
      end if;
      Object.On_Start := Library_Symbol
        (Object.Library_Reference, "sur_demarrer");
      if Object.On_Start = null then
         Object.On_Start :=
           Do_Nothing_On_Start'Access;
      end if;
      Object.On_Stop := Library_Symbol
        (Object.Library_Reference, "sur_arreter");
      if Object.On_Stop = null then
         Object.On_Stop := Do_Nothing'Access;
      end if;
      Object.On_Suspend := Library_Symbol
        (Object.Library_Reference, "sur_suspendre");
      if Object.On_Suspend = null then
         Object.On_Suspend := Do_Nothing'Access;
      end if;
      Object.On_Resume := Library_Symbol
        (Object.Library_Reference, "sur_reprendre");
      if Object.On_Resume = null then
         Object.On_Resume := Do_Nothing'Access;
      end if;
      if Object.Inputs (1).Swapped_Link then
         Object.Swap_To_C := 1;
      end if;
      Object.Common_Reference :=
        Object.Init_Function.all;
   exception
      when others =>
         delay 2.0;
         raise;
   end On_Initialise;

   ----------------------------
   -- Travail_En_Acquisition --
   ----------------------------

   procedure Buffer_Handling
     (Object : access Generic_Consumer_Type) is
   begin
      declare
         use Protected_Memory;
         use type System.Storage_Elements.Storage_Offset;
         Data : Raw_Access_Type;
         Manipulation_Buffer : Buffer_Enumeration_Type;
      begin
         select
            Object.Inputs (1).Memory.Lock_Buffer
              (Manipulation_Buffer);
            begin
               Data :=
                 Object.Inputs
                 (1).Memory.Get_Buffer_Structure (Manipulation_Buffer);
               Object.Buffer_Handling
                 (Object.Common_Reference,
                  Data.Address,
                  Data.Size,
                  Object.Swap_To_C);
               Object.Inputs
                 (1).Memory.Unlock_Buffer
                 (Manipulation_Buffer);
            exception
               when others =>
                  Object.Inputs
                    (1).Memory.Unlock_Buffer (Manipulation_Buffer);
            end;
         or
            delay Object.Waiting_Delay;
         end select;
      end;
   exception
      when E : others =>
         Error_Out (Object.Logger'Access,
                    "Travail_En_Acquisition consommateur generique", E);
   end Buffer_Handling;

   procedure On_Unload
     (Object : access Generic_Consumer_Type) is
      use Shared_Library;
      Function_Return : Integer;
   begin
      Consumers.On_Unload
        (Consumer_Access (Object));
      Object.On_Unload
        (Object.Common_Reference);
      Function_Return := Close_Library
        (Object.Library_Reference);
      Debug_Out (Object.Logger'Access,
                 "retour clore_bibliotheque :" & Function_Return'Img);
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access,
                    "Sur_Decharger", E);
         raise;
   end On_Unload;

   procedure On_Reset_Com
     (Object : access Generic_Consumer_Type) is
   begin
      Consumers.On_Reset_Com
        (Consumer_Access (Object));
      Object.On_Reset_Com
        (Object.Common_Reference);
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access,
                    "Sur_Remise_A_Zero", E);
         raise;
   end On_Reset_Com;

   procedure On_Start
     (Object : access Generic_Consumer_Type) is
      use Narval.Configurator.Abstract_Actors_Coordination;
      Run_Number : constant Integer := Integer'Value
        (Get_Image (Local_Configuration.Config_Local, "run_number"));
   begin
      Consumers.On_Start
        (Consumer_Access (Object));
      Object.On_Start
        (Object.Common_Reference, Run_Number);
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access,
                    "Sur_Demarrer", E);
         raise;
   end On_Start;

   procedure On_Stop
     (Object : access Generic_Consumer_Type) is
   begin
      Consumers.On_Stop
        (Consumer_Access (Object));
      Object.On_Stop
        (Object.Common_Reference);
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access,
                    "Sur_Arreter", E);
         raise;
   end On_Stop;

   procedure On_Suspend
     (Object : access Generic_Consumer_Type) is
   begin
      Consumers.On_Suspend
        (Consumer_Access (Object));
      Object.On_Suspend
        (Object.Common_Reference);
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access,
                    "Sur_Suspendre", E);
         raise;
   end On_Suspend;

   procedure On_Resume
     (Object : access Generic_Consumer_Type) is
   begin
      Consumers.On_Resume
        (Consumer_Access (Object));
      Object.On_Resume
        (Object.Common_Reference);
   exception
      when E : others =>
         Fatal_Out (Object.Logger'Access,
                    "Sur_Reprendre", E);
         raise;
   end On_Resume;

end Narval.Actors.Actives.Consumers.Basic_Generic;
